#
# Authors: Dan Walsh <dwalsh@redhat.com>
#
# Copyright (C) 2006 Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

import gettext
_ = gettext.translation('setroubleshoot-plugins', '/usr/share/locale', fallback=True).lgettext

from setroubleshoot.util import *
from setroubleshoot.Plugin import Plugin

class plugin(Plugin):
    summary = _('''
    SELinux prevented $SOURCE from $ACCESS files stored in directories marked with the default context.
    ''')
    
    problem_description = _('''
    SELinux prevented $SOURCE from $ACCESS files stored in directories marked with the default context.

    $SOURCE attempted to read one or more files or directories which are
    marked with the default context.  Confined domains usually do not need to
    access these directories, this access attempt could signal an intrusion
    attempt.  If this application and other confined applications need to
    access these files you can either change the file context of the files
    or set a boolean.
    ''')
    
    fix_description = _('''
    Changing the "$BOOLEAN" boolean to true will allow this access:
    "setsebool -P $BOOLEAN=1"
    ''')
    
    fix_cmd = 'setsebool -P $BOOLEAN=1'
    
    def __init__(self):
        Plugin.__init__(self, __name__)
        self.set_priority(75)

    def analyze(self, avc):
        if (avc.matches_target_types(['default_t']) and \
           (avc.has_tclass_in(['file', 'dir']))):
            if avc.all_accesses_are_in(avc.r_file_perms + avc.r_dir_perms):
                # MATCH
                avc.set_template_substitutions(BOOLEAN="read_default_t", ACCESS="reading")
                return self.report(avc, None,
                                   self.summary, self.problem_description,
                                   self.fix_description, self.fix_cmd)
            if avc.all_accesses_are_in(avc.create_file_perms + avc.rw_dir_perms):
                # MATCH
                avc.set_template_substitutions(BOOLEAN="read_default_t", ACCESS="reading and writing")
                return self.report(avc, None,
                                   self.summary, self.problem_description,
                                   self.fix_description, self.fix_cmd)
        return None
