#
# Authors: Karl MacMillan <kmacmillan@mentalrootkit.com>
#
# Copyright (C) 2006 Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

import gettext
_ = gettext.translation('setroubleshoot-plugins', '/usr/share/locale', fallback=True).lgettext

from setroubleshoot.util import *
from setroubleshoot.Plugin import Plugin

class plugin(Plugin):
    summary = _('''
    SELinux prevented the gss daemon from reading unprivileged user temporary files.
    ''')
    
    problem_description = _('''SELinux prevented the gss daemon from
    reading unprivileged user temporary files (e.g., files in /tmp). Allowing this
    access is low risk, but if you have not configured the gss daemon to
    read these files this access request could signal an intrusion
    attempt.''')
    
    fix_description = _('''
    Changing the "$BOOLEAN" boolean to true will allow this access:
    "setsebool -P $BOOLEAN=1."
    ''')
    
    fix_cmd = 'setsebool -P $BOOLEAN=1'

    def __init__(self):
        Plugin.__init__(self, __name__)

    def analyze(self, avc):
        if avc.matches_source_types(['gssd_t'])   and \
           avc.matches_target_types(['.*_tmp_t']) and \
           avc.has_tclass_in(['file', 'dir']):
            # MATCH
            avc.set_alt_path("/tmp")
            avc.set_template_substitutions(BOOLEAN="allow_gssd_read_tmp")
            return self.report(avc, _("Authorization"),
                               self.summary, self.problem_description,
                               self.fix_description, self.fix_cmd)
        else:
            return None
