/*
 * Copyright (C) 2005 Alex Murray <pragmatine@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SENSORS_APPLET_H
#define SENSORS_APPLET_H
#include <gtk/gtk.h>
#include <panel-applet.h>

typedef struct _SensorsApplet SensorsApplet;
typedef struct _ActiveSensor ActiveSensor;

#include "prefs-dialog.h"

#define SENSORS_APPLET_ICON PIXMAPS_DIR "sensors-applet-icon.png"

/* device icons */
#define CPU_ICON PIXMAPS_DIR "cpu-icon.png"
#define HDD_ICON PIXMAPS_DIR "hdd-icon.png"
#define BATTERY_ICON PIXMAPS_DIR "battery-icon.png"
#define MEM_ICON PIXMAPS_DIR "mem-icon.png"
#define GPU_ICON PIXMAPS_DIR "gpu-icon.png"
#define GENERIC_ICON PIXMAPS_DIR "generic-icon.png"
#define FAN_ICON PIXMAPS_DIR "fan-icon.png"
#define VOLTAGE_ICON NULL

#define DEFAULT_ICON_SIZE 24

typedef enum {
	UNUSED = 0, /* as a flag to test against later */
	ACPI,
	HDDTEMP,
	I2C_SYS,
	I2C_PROC,
	IBM_ACPI,
	I8K,
	LIBSENSORS,
	OMNIBOOK,
	PMU_SYS,
	SMU_SYS,
	N_SENSOR_INTERFACES
} SensorInterface;

static const gchar *sensor_interface[] = {
	"unused",
	"acpi",
	"hddtemp",
	"i2c-sys",
	"i2c-proc",
	"ibm-acpi",
	"i8k",
	"libsensors",
	"omnibook",
	"pmu-sys",
	"smu-sys",
};

/* enumeration used to identify columns in the GtkTreeStore data
 * structure and to access specific gconf keys too.
 */
enum {
	PATH_COLUMN = 0,
	ID_COLUMN,
	LABEL_COLUMN,
	INTERFACE_COLUMN,
	SENSOR_TYPE_COLUMN,
	ENABLE_COLUMN,
	VISIBLE_COLUMN,
       	ALARM_VALUE_COLUMN,
	ALARM_TYPE_COLUMN, 
	ALARM_ENABLE_COLUMN,
	ALARM_COMMAND_COLUMN,
	ALARM_TIMEOUT_COLUMN,
	MULTIPLIER_COLUMN,
	OFFSET_COLUMN,
	ICON_FILENAME_COLUMN,
	ICON_PIXBUF_COLUMN,
	N_COLUMNS
};


typedef enum {
	CURRENT_SENSOR = 0,
	FAN_SENSOR,
	TEMP_SENSOR,
	VOLTAGE_SENSOR
} SensorType;

/* for display mode */
typedef enum {
	DISPLAY_NONE = 0,
	DISPLAY_LABEL,
	DISPLAY_ICON,
} DisplayMode;

/* alarm types */
typedef enum {
	ALARM_WHEN_VALUE_GREATER_THAN_THRESHOLD = 0,
	ALARM_WHEN_VALUE_LESS_THAN_THRESHOLD
} AlarmType;


/* for identifying font sizes */
typedef enum {
	XX_LARGE = 0,
	X_LARGE,
	LARGE,
	MEDIUM,
	SMALL,
	X_SMALL,
	XX_SMALL
} FontSize;

/* constants for scaling icons as font size changes */
typedef enum {
	XX_SMALL_ICON_SIZE = 12,
	X_SMALL_ICON_SIZE = 16,
	SMALL_ICON_SIZE = 20,
	MEDIUM_ICON_SIZE = 24,
	LARGE_ICON_SIZE = 28,
	X_LARGE_ICON_SIZE = 32,
	XX_LARGE_ICON_SIZE = 36
} IconSize;

/* should always return the value in degrees celcius, revolutions per
 * minute, volts, amps, etc */
typedef gdouble (*GetSensorValueFunction)(const gchar *path, 
					  const gchar *id, 
					  SensorType type,
					  GError **error);

struct _SensorsApplet {
	/* the actual applet for this instance */
	PanelApplet* applet;

	GtkTreeStore *sensors;
	GtkTreeSelection *selection;

	GetSensorValueFunction get_sensor_value[N_SENSOR_INTERFACES];

	guint timeout_id;
	/* preferences and about windows (if Gtk < 2.6)*/
#ifndef HAVE_GTK_26
	GtkWidget *about_dialog;
#endif
	GtkDialog *prefs_dialog;

	/* primary table to contain the panel dispay - we pack the
	 * list of labels and sensor values into this container */
	GtkWidget *table;
	GList *active_sensors;
	GtkTooltips *tooltips;

	/* list of gchar* to the alarm commands, which have been
	 * allocated for each active alarm - these are then freed when
	 * the alarm is turned off */
	GSList *alarm_commands;
};


#define XX_LARGE_TEXT "xx-large"
#define X_LARGE_TEXT "x-large"
#define LARGE_TEXT "large"
#define MEDIUM_TEXT "medium"
#define SMALL_TEXT "small"
#define X_SMALL_TEXT "x-small"
#define XX_SMALL_TEXT "xx-small"


/* non-static function prototypes */
void sensors_applet_init(SensorsApplet *sensors_applet);
void sensors_applet_sensor_enabled(SensorsApplet *sensors_applet,
				   GtkTreePath *path);
void sensors_applet_sensor_disabled(SensorsApplet *sensors_applet,
				    GtkTreePath *path);
void sensors_applet_update_active_sensors(SensorsApplet *sensors_applet);
void sensors_applet_register_sensors_interface(SensorsApplet *sensors_applet,
					       SensorInterface interface,
					       GetSensorValueFunction get_sensor_value_function);

gboolean sensors_applet_add_sensor_full_details(SensorsApplet *sensors_applet,
						const gchar *path, 
						const gchar *id, 
						const gchar *label, 
						SensorInterface interface, 
						SensorType type, 
						gboolean enable,
						gdouble alarm_value,
						AlarmType alarm_type,
						gboolean alarm_enable,
						const gchar *alarm_command,
						gint alarm_timeout,
						gdouble multiplier,
						gdouble offset,
						const gchar *icon_filename);

gboolean sensors_applet_add_sensor(SensorsApplet *sensors_applet,
				   const gchar *path, 
				   const gchar *id, 
				   const gchar *label, 
				   SensorInterface interface, 
				   gboolean enable,
				   SensorType type,
				   const gchar *icon_filename); 

/**
 * to be called by things like prefs dialog to turn off a sensor alarm
 */
void sensors_applet_alarm_off(SensorsApplet *sensors_applet, GtkTreePath *path);
void sensors_applet_update_sensor(SensorsApplet *sensors_applet,
				  GtkTreePath *path);
void sensors_applet_display_layout_changed(SensorsApplet *sensors_applet);
void sensors_applet_icon_changed(SensorsApplet *sensors_applet,
				 GtkTreePath *path);
#endif /* SENSORS_APPLET_H */
