/*
For general Scribus (>=1.3.2) copyright and licensing information please refer
to the COPYING file provided with the program. Following this notice may exist
a copyright and/or license notice that predates the release of Scribus 1.3.2
for which a new license (GPL+exception) is in place.
*/
#include "commonstrings.h"
#include "customfdialog.h"
#include "importxfig.h"
#include "importxfigplugin.h"
#include "menumanager.h"
#include "page.h"
#include "prefscontext.h"
#include "prefsfile.h"
#include "prefsmanager.h"
#include "scraction.h"
#include "scribuscore.h"
#include "undomanager.h"
#include "util_formats.h"


int importxfig_getPluginAPIVersion()
{
	return PLUGIN_API_VERSION;
}

ScPlugin* importxfig_getPlugin()
{
	ImportXfigPlugin* plug = new ImportXfigPlugin();
	Q_CHECK_PTR(plug);
	return plug;
}

void importxfig_freePlugin(ScPlugin* plugin)
{
	ImportXfigPlugin* plug = dynamic_cast<ImportXfigPlugin*>(plugin);
	Q_ASSERT(plug);
	delete plug;
}

ImportXfigPlugin::ImportXfigPlugin() : LoadSavePlugin(),
	importAction(new ScrAction(ScrAction::DLL, QPixmap(), QPixmap(), "", QKeySequence(), this))
{
	// Set action info in languageChange, so we only have to do it in one
	// place. This includes registering file format support.
	languageChange();
}
/*
void ImportXfigPlugin::addToMainWindowMenu(ScribusMainWindow *mw)
{
	importAction->setEnabled(true);
	connect( importAction, SIGNAL(triggered()), SLOT(import()) );
	mw->scrMenuMgr->addMenuItem(importAction, "FileImport");
}
*/
void ImportXfigPlugin::languageChange()
{
	importAction->setText( tr("Import Xfig..."));
	// (Re)register file format support
	unregisterAll();
	registerFormats();
}

ImportXfigPlugin::~ImportXfigPlugin()
{
	unregisterAll();
};

const QString ImportXfigPlugin::fullTrName() const
{
	return QObject::tr("Xfig Importer");
}


const ScActionPlugin::AboutData* ImportXfigPlugin::getAboutData() const
{
	AboutData* about = new AboutData;
	about->authors = "Franz Schmid <franz@scribus.info>";
	about->shortDescription = tr("Imports Xfig Files");
	about->description = tr("Imports most Xfig files into the current document,\nconverting their vector data into Scribus objects.");
	about->license = "GPL";
	Q_CHECK_PTR(about);
	return about;
}

void ImportXfigPlugin::deleteAboutData(const AboutData* about) const
{
	Q_ASSERT(about);
	delete about;
}

void ImportXfigPlugin::registerFormats()
{
	FileFormat fmt(this);
	fmt.trName = FormatsManager::instance()->nameOfFormat(FormatsManager::XFIG); // Human readable name
	fmt.formatId = FORMATID_XFIGIMPORT;
	fmt.filter = FormatsManager::instance()->extensionsForFormat(FormatsManager::XFIG); // QFileDialog filter
	fmt.nameMatch = QRegExp("\\."+FormatsManager::instance()->extensionListForFormat(FormatsManager::XFIG, 1)+"$", Qt::CaseInsensitive);
	fmt.load = true;
	fmt.save = false;
	fmt.mimeTypes = FormatsManager::instance()->mimetypeOfFormat(FormatsManager::XFIG); // MIME types
	fmt.priority = 64; // Priority
	registerFormat(fmt);
}

bool ImportXfigPlugin::fileSupported(QIODevice* /* file */, const QString & fileName) const
{
	return true;
}

bool ImportXfigPlugin::loadFile(const QString & fileName, const FileFormat &, int flags, int /*index*/)
{
	// There's only one format to handle, so we just call import(...)
	return import(fileName, flags);
}

bool ImportXfigPlugin::import(QString fileName, int flags)
{
	if (!checkFlags(flags))
		return false;
	if (!(flags & lfInteractive))
		UndoManager::instance()->setUndoEnabled(false);
	if( fileName.isEmpty() )
	{
		flags |= lfInteractive;
		PrefsContext* prefs = PrefsManager::instance()->prefsFile->getPluginContext("importxfig");
		QString wdir = prefs->get("wdir", ".");
		CustomFDialog diaf(ScCore->primaryMainWindow(), wdir, QObject::tr("Open"), tr("All Supported Formats")+" (*.fig *.FIG);;All Files (*)");
		if (diaf.exec())
		{
			fileName = diaf.selectedFile();
			prefs->set("wdir", fileName.left(fileName.lastIndexOf("/")));
		}
		else
			return true;
	}
	m_Doc=ScCore->primaryMainWindow()->doc;
	UndoTransaction* activeTransaction = NULL;
	bool emptyDoc = (m_Doc == NULL);
	if (UndoManager::undoEnabled() && !emptyDoc)
	{
		activeTransaction = new UndoTransaction(UndoManager::instance()->beginTransaction(m_Doc->currentPage()->getUName(),
																						  Um::IImageFrame,
																						  Um::ImportXfig,
																						  fileName, Um::IXFIG));
	}
	else if (UndoManager::undoEnabled() && emptyDoc)
		UndoManager::instance()->setUndoEnabled(false);
	XfigPlug *dia = new XfigPlug(m_Doc, flags);
	Q_CHECK_PTR(dia);
	dia->import(fileName, flags);
	if (activeTransaction)
	{
		activeTransaction->commit();
		delete activeTransaction;
		activeTransaction = NULL;
	}
	else if (!(flags & lfInteractive))
		UndoManager::instance()->setUndoEnabled(true);
	delete dia;
	return true;
}
