/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <qtimer.h>
#include <qlabel.h>
#include <qslider.h>
#include <qpushbutton.h>
#include <qscrollview.h>
#include <qlayout.h>

#include <kmessagebox.h>
#include <klocale.h>

#include <libinstrudeo/isdrecording.h>
#include <libinstrudeo/isdvideoproperties.h>

#include "srkmainview.h"
#include "srkvideocontainer.h"
#include "srkvideocanvas.h"
#include "srkvideopanel.h"

/**
 * The control panel for the video, also containing the canvas.
 * Is an implementation of the corresponding .ui file.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

//-----CONSTRUCTORS-----
SRKVideoPanel::SRKVideoPanel(QWidget* parent, const char* name)
    : SRKVideoPanelBase(parent, name),
      timerCount(0),
      playing(false),
      position(0),
      timer(new QTimer(this)),
      recording(NULL)
{
    //init the timer
    connect(timer, SIGNAL(timeout()), this, SLOT(incPosition()));
    
    setEnableControls(false);

    //connect the popup and dirty slots
    connect(this, SIGNAL(firePopupMenu(QString&, QPoint&)), (SRKMainView*)parent, SLOT(popupMenu(QString&, QPoint&)));
    connect(this, SIGNAL(fireMarkDirty()), (SRKMainView*)parent, SLOT(markDirty()));
}

//-----DESTRUCTOR-----
SRKVideoPanel::~SRKVideoPanel()
{
    /*
     * Note: the recording is deleted in the SRKMainView class
     * Note2: the timer and the videoContainer are both children of this widget and
     *        get deleted automatically
     */
}

//-----PUBLIC METHODS-----
SRKError::SRKErrorCode SRKVideoPanel::loadRecording(ISDRecording* recording)
{
    SRKError::SRKErrorCode retVal;

    //save a reference
    this->recording = recording;
    
    //enable the buttons and slider
    setEnableControls(true);

    //init the sliders range
    slider->setRange(0, recording->getVideoProperties()->getLength());
    slider->setValue(0);

    retVal = videoContainer->loadRecording(recording);
    //init the commentboxes and canvas
    updateImage(0);

    resolutionLbl->setText(ISDUtils::getInstance()->intToString(recording->getVideoProperties()->getWidth())+"x"+
			   ISDUtils::getInstance()->intToString(recording->getVideoProperties()->getHeight())+" pixels");

    return retVal;
}
SRKError::SRKErrorCode SRKVideoPanel::closeRecording()
{
    setEnableControls(false);
    
    //recording gets deleted in SRKMainView class
    recording = NULL;

    SRKError::SRKErrorCode retVal = videoContainer->closeRecording();
    return retVal;
}
SRKError::SRKErrorCode SRKVideoPanel::addComment(ISDCommentbox::commentboxType type)
{
    return videoContainer->addComment(type, position);
}
SRKError::SRKErrorCode SRKVideoPanel::deleteActiveComment()
{
    return videoContainer->deleteActiveComment();
}
SRKError::SRKErrorCode SRKVideoPanel::editActiveCommentProperties()
{
    return videoContainer->editActiveCommentProperties();
}
SRKError::SRKErrorCode SRKVideoPanel::endActiveCommentHere()
{
    return videoContainer->endActiveCommentAtPosition(position);
}
SRKError::SRKErrorCode SRKVideoPanel::changeZoomFactor(int factor)
{
    return videoContainer->changeZoomFactor(factor);;
}
SRKError::SRKErrorCode SRKVideoPanel::changeCommentsLanguage(QString& lang)
{
    currentCommentsLanguage = lang;
    
    if (recording!=NULL) {
	updateImage(position);
    }

    return videoContainer->changeCommentsLanguage(lang);
}
SRKError::SRKErrorCode SRKVideoPanel::update()
{
    if (recording!=NULL) {
	videoContainer->updateCanvas();
	videoContainer->updateComments();
    }

    RETURN_SUCCESS;
}

//-----PUBLIC SLOTS-----
void SRKVideoPanel::stop()
{
    playing = false;
    timer->stop();
    slider->setValue(0);
    updateImage(0);
}
void SRKVideoPanel::pause()
{
    playing = false;
    timer->stop();
    //this is called to smooth the image on pause
    updateImage(position);
}

void SRKVideoPanel::play()
{
    playing = true;
    timer->start(TIMER_TIMEOUT);
}
void SRKVideoPanel::popupMenu(QString& name, QPoint& pos)
{
    emit firePopupMenu(name, pos);
}
void SRKVideoPanel::markDirty()
{
    emit fireMarkDirty();
}

//-----PROTECTED SLOTS-----
void SRKVideoPanel::incPosition()
{
    position += TIMER_TIMEOUT;
    int videoLength = recording->getVideoProperties()->getLength();
    
    //check for end of stream
    if (position>videoLength){
	//can't call stop() here, because we want to stay at the end position
	playing = false;
	timer->stop();
	slider->setValue(slider->maxValue());
	return;
    }
    
    /*
     * This is done automatically because the slider and canvas are connected.
     * Thanks to Dave Cohen for this.
     */
    //updateImage(position);
    slider->setValue(position);
}

void SRKVideoPanel::updateImage(int position)
{
    bool noVideoChange, noCommentChange;

    //sync the timer pos with the slider pos
    this->position = position;
    int videoLength = recording->getVideoProperties()->getLength();

    positionLbl->setText(ISDUtils::getInstance()->intToString(position)+"/"+
			 ISDUtils::getInstance()->intToString(videoLength)+" millis");
    
    Glib::ustring lang(currentCommentsLanguage.utf8().data());
    recording->update(position, &lang, noVideoChange, noCommentChange);

    //do nothing if nothing changed
    if (!noVideoChange) {
	videoContainer->updateCanvas();
    }

    if (!noCommentChange) {
	videoContainer->updateComments();
    }
}

//-----PROTECTED FUNCTIONS-----
void SRKVideoPanel::setEnableControls(bool b)
{
    playButton->setEnabled(b);
    pauseButton->setEnabled(b);
    stopButton->setEnabled(b);
    slider->setEnabled(b);
}

#include "srkvideopanel.moc"
