/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef SRKCOMMENTBOX_H
#define SRKCOMMENTBOX_H

/**
 * Virtual base class that represents an OpenGL commentbox.
 * This class doesn't really do any OpenGL drawing, this is
 * left entirely to the underlying instrudeo engine.
 * Only the four cornered resize widgets are drawn here,
 * so this is merely an overlay for the actual commentbox,
 * that provides resizing and positioning functionality.
 * Note that this widget follows the _scaled_ size of the
 * Instrudeo-widget, not it's real (unscaled) size.
 * Note: because of the mask, we cannot implement the
 * moving-functionality here, since mouseclicks don't get 
 * caught on transparent areas.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>
#include <qwidget.h>

using namespace std;

#define DEFAULT_HANDLE_WIDTH 10
#define DEFAULT_HANDLE_HEIGHT 10
#define DEFAULT_HANDLE_FILL_COLOR Qt::gray
#define DEFAULT_HANDLE_BORDER_COLOR Qt::black

#define DEFAULT_COMMENTBOX_WIDTH 300
#define DEFAULT_COMMENTBOX_HEIGHT 200

class ISDCommentbox;

class SRKCommentbox : public QWidget
{
    Q_OBJECT

 public:
    //-----CONSTRUCTOR-----
    SRKCommentbox(ISDCommentbox* isdBox, QWidget* parent = 0, const char* name = 0);
    
    virtual ~SRKCommentbox();

    //-----METHODS-----
    /**
     * Tell the commentbox some scale-change happened.
     */
    void scaleChanged();
    
    /**
     * Set this commentbox active.
     * Active commentboxes draw resize handles and will receive
     * other events (like deletion, change, etc.)
     */
    void setActive(bool active);
    
    /**
     * Tells if this commentbox is active.
     *
     * @return true if active, false otherwise.
     */
    bool isActive();

    /**
     * Instead of wrapping every function around the Instrudeo-commentbox,
     * we provide direct access to the methods of the library by supplying a 
     * reference to the Instrudeo commentbox.
     *
     * @return The Instrudeo-commentbox or NULL on error.
     */
    ISDCommentbox* getISDCommentbox();

 protected:
    //-----OVERLOADED METHODS-----
    virtual void updateMask();
    virtual void resizeEvent(QResizeEvent* e);
    virtual void moveEvent(QMoveEvent* e);
    
    //-----METHODS-----
    void resizeHandles();
        
    //-----VARIABLES-----
    ISDCommentbox* isdBox;
    bool active;
    

    //-----PROTECTED NESTED CLASS-----
    /**
     * This class represents the resize square in the four corners of the commentbox.
     */
    class SRKResizeHandle : public QWidget
    {
      public:
	enum ResizePosition {
	    UPPER_LEFT,
	    UPPER_RIGHT,
	    BOTTOM_LEFT,
	    BOTTOM_RIGHT
	};
	
	SRKResizeHandle(ResizePosition p, QWidget* parent = 0, const char* name = 0);
	virtual ~SRKResizeHandle();
	 
      protected:
	virtual void paintEvent(QPaintEvent* e);
	virtual void enterEvent(QEvent*);
	virtual void leaveEvent(QEvent*);
	virtual void mousePressEvent(QMouseEvent* e);
	virtual void mouseMoveEvent(QMouseEvent* e);
	virtual void mouseReleaseEvent(QMouseEvent* e);
	
	void init();
	
	QColor fillColor;
	QColor borderColor;
	ResizePosition mode;
	bool dragging;
	QPoint dragStartPos;
	QPoint dragPosBak;
	QSize dragSizeBak;
    };

    //-----VARIABLES-----
    SRKResizeHandle* upperLeftHandle;
    SRKResizeHandle* upperRightHandle;
    SRKResizeHandle* bottomLeftHandle;
    SRKResizeHandle* bottomRightHandle;
};

#endif
