/*
 * Copyright (C) 2014-2016 Olzhas Rakhimov
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/// @file risk_analysis.h
/// Contains the main system for performing analysis.

#ifndef SCRAM_SRC_RISK_ANALYSIS_H_
#define SCRAM_SRC_RISK_ANALYSIS_H_

#include <iostream>
#include <map>
#include <memory>
#include <string>

#include "analysis.h"
#include "event.h"
#include "fault_tree_analysis.h"
#include "importance_analysis.h"
#include "model.h"
#include "probability_analysis.h"
#include "settings.h"
#include "uncertainty_analysis.h"

namespace scram {
namespace core {

/// Main system that performs analyses.
class RiskAnalysis : public Analysis {
 public:
  /// Pointer aliases for convenience.
  /// @{
  using FaultTreeAnalysisPtr = std::unique_ptr<FaultTreeAnalysis>;
  using ProbabilityAnalysisPtr = std::unique_ptr<ProbabilityAnalysis>;
  using ImportanceAnalysisPtr = std::unique_ptr<ImportanceAnalysis>;
  using UncertaintyAnalysisPtr = std::unique_ptr<UncertaintyAnalysis>;
  /// @}

  /// Constructs RiskAnalysis with a valid model and analysis settings.
  ///
  /// @param[in] model  An analysis model with fault trees, events, etc.
  /// @param[in] settings  Analysis settings for the given model.
  RiskAnalysis(std::shared_ptr<const mef::Model> model,
               const Settings& settings);

  /// @returns The model under analysis.
  const mef::Model& model() const { return *model_; }

  /// Performs the main analysis operations.
  /// Analyzes the fault tree and performs computations.
  ///
  /// @note This function must be called
  ///       only after initializing the model
  ///       with or without its probabilities.
  void Analyze() noexcept;

  /// Reports all results generated by all analyses
  /// into XML formatted stream.
  /// The report is appended to the stream.
  ///
  /// @param[out] out  The output stream.
  ///
  /// @note This function must be called only after Analyze() function.
  void Report(std::ostream& out);

  /// Reports the results of analyses
  /// to a specified output destination.
  /// This function overwrites the file.
  ///
  /// @param[out] output  The output destination.
  ///
  /// @throws IOError  The output file is not accessible.
  ///
  /// @note This function must be called only after Analyze() function.
  void Report(std::string output);

  /// @returns Fault tree analyses performed on one-top-event fault trees.
  ///          The top gate identifier is used as the analysis identifier.
  const std::map<std::string, FaultTreeAnalysisPtr>&
  fault_tree_analyses() const {
    return fault_tree_analyses_;
  }

  /// @returns Probability analysis performed on
  ///          Boolean products generated by
  ///          fault tree analyses.
  const std::map<std::string, ProbabilityAnalysisPtr>&
  probability_analyses() const {
    return probability_analyses_;
  }

  /// @returns Importance analysis performed on
  ///          Boolean products generated by
  ///          fault tree analyses.
  const std::map<std::string, ImportanceAnalysisPtr>&
  importance_analyses() const {
    return importance_analyses_;
  }

  /// @returns Uncertainty analyses performed on
  ///          Boolean products generated by
  ///          fault tree analyses.
  const std::map<std::string, UncertaintyAnalysisPtr>&
  uncertainty_analyses() const {
    return uncertainty_analyses_;
  }

 private:
  /// Runs all possible analysis on a given target.
  /// Analysis types are deduced from the settings.
  ///
  /// @param[in] name  Identificator for analyses.
  /// @param[in] target  Analysis target.
  void RunAnalysis(const std::string& name, const mef::Gate& target) noexcept;

  /// Defines and runs Qualitative analysis on the target.
  /// Calls the Quantitative analysis if requested in settings.
  ///
  /// @tparam Algorithm  Qualitative analysis algorithm.
  ///
  /// @param[in] name  Identificator for analyses.
  /// @param[in] target  Analysis target.
  template <class Algorithm>
  void RunAnalysis(const std::string& name, const mef::Gate& target) noexcept;

  /// Defines and runs Quantitative analysis on the target.
  ///
  /// @tparam Algorithm  Qualitative analysis algorithm.
  /// @tparam Calculator  Quantitative analysis algorithm.
  ///
  /// @param[in] name  Identificator for analyses.
  /// @param[in] fta  The result of Qualitative analysis.
  ///
  /// @pre FaultTreeAnalyzer is ready to tolerate
  ///      giving its internals to Quantitative analyzers.
  template <class Algorithm, class Calculator>
  void RunAnalysis(const std::string& name,
                   FaultTreeAnalyzer<Algorithm>* fta) noexcept;

  /// Analysis model with constructs.
  std::shared_ptr<const mef::Model> model_;

  /// Analyses performed by this risk analysis run.
  /// @{
  std::map<std::string, FaultTreeAnalysisPtr> fault_tree_analyses_;
  std::map<std::string, ProbabilityAnalysisPtr> probability_analyses_;
  std::map<std::string, ImportanceAnalysisPtr> importance_analyses_;
  std::map<std::string, UncertaintyAnalysisPtr> uncertainty_analyses_;
  /// @}
};

}  // namespace core
}  // namespace scram

#endif  // SCRAM_SRC_RISK_ANALYSIS_H_
