/*******************************************************************************
 * Score Reading Trainer                                                       *
 * Copyright (C) 2004 by Jos Pablo Ezequiel Fernndez <pupeno@pupeno.com>     *
 *                                                                             *
 * This program is free software; you can redistribute it and/or               *
 * modify it under the terms of the GNU General Public License                 *
 * version 2 as published by the Free Software Foundation.                     *
 *                                                                             *
 * This program is distributed in the hope that it will be useful,             *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 * GNU General Public License for more details.                                *
 *                                                                             *
 * You should have received a copy of the GNU General Public License           *
 * along with this program; if not, write to the Free Software                 *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *******************************************************************************/

#include <qsignalmapper.h>
#include <qvaluelist.h>

#include <kapplication.h>
#include <klocale.h>
#include <kkeydialog.h>
#include <kaction.h>
#include <kstdaction.h>
#include <ktoolbar.h>
#include <kdebug.h>
#include <kconfig.h>
#include <kedittoolbar.h>

#include "Preferences.h"
#include "ScoreReadingTrainerView.h"
#include "Score.h"

#include "ScoreReadingTrainer.h"

ScoreReadingTrainer::ScoreReadingTrainer() : KMainWindow( 0, "ScoreReadingTrainer" ), scoreReadingTrainerView(new ScoreReadingTrainerView(this)){
	kdDebug() << "Running ScoreReadingTrainer::ScoreReadingTrainer()" << endl;

	// Don't accept drag and drop. What do you want to drag today ?
	setAcceptDrops(false);

	// Tell the KMainWindow that the score is indeed the main widget.
	setCentralWidget(scoreReadingTrainerView);

	// Setup our actions.
	setupActions();

	// Apply the saved mainwindow settings, if any, and ask the mainwindow to automatically save settings if changed: window size, toolbar position, icon size, etc.
	setAutoSaveSettings();

	// Show the toolbars
	toolBar()->show();

	// Start stoped.
	stop();

	// Load the preferences from the config file and put them in action.
	loadPreferences();
}

ScoreReadingTrainer::~ScoreReadingTrainer(){
	//kdDebug() << "Running ScoreReadingTrainer::~ScoreReadingTrainer()" << endl;

}

void ScoreReadingTrainer::saveProperties(KConfig *){
	//kdDebug() << "Running ScoreReadingTrainer::saveProperties(KConfig *)" << endl;

	// the 'config' object points to the session managed
	// config file.  anything you write here will be available
	// later when this app is restored
}

void ScoreReadingTrainer::readProperties(KConfig *){
	//kdDebug() << "Running ScoreReadingTrainer::readProperties(KConfig *)" << endl;

	// the 'config' object points to the session managed
	// config file.  this function is automatically called whenever
	// the app is being restored.  read in here whatever you wrote
	// in 'saveProperties'
}

void ScoreReadingTrainer::start(){
	// Enable de actions for the notes.
	for ( QValueList<KAction *>::iterator it = noteActions.begin(); it != noteActions.end(); ++it ){
		(*it)->setEnabled(true);
	}
	
	// Enable the stop action to be able to stop.
	stopAction->setEnabled(true);
	
	// Disable the start action, we are not intrested in re-starting.
	startAction->setEnabled(false);
	
	// Do the real start.
	scoreReadingTrainerView->start();

	// We are now started, started = true.
	started = true;
};

void ScoreReadingTrainer::stop(){
	// We are now stoped, started = false;
	started = false;
	
	// Disable de actions for the notes.
	for ( QValueList<KAction *>::iterator it = noteActions.begin(); it != noteActions.end(); ++it ){
		(*it)->setEnabled(false);
	}
		
	// Enable the start action to be able to start.
	startAction->setEnabled(true);
	
	// Disable the stop action, we are not intrested in re-stoping.
	stopAction->setEnabled(false);
	
	// Do the real stop.
	scoreReadingTrainerView->stop();
};

void ScoreReadingTrainer::optionsConfigureKeys(){
	// Show the dialog for configuring shortcuts.
	KKeyDialog::configureKeys(actionCollection(), "ScoreReadingTrainerui.rc");
}

void ScoreReadingTrainer::optionsConfigureToolbars(){
	// Save the settings of the main window before starting any reconfiguration.
#if KDE_IS_VERSION(3,1,0)
	saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
#else
	saveMainWindowSettings(KGlobal::config());
#endif
	// Create the dialog for configuration the toolbars.
	KEditToolbar editToolbarDialog(actionCollection());
	// When the changes to the toolbars are oked or applied in the toolbar dialog, we want to apply them to the running session.
	connect(&editToolbarDialog,SIGNAL(newToolbarConfig()),this,SLOT(newToolbarConfig()));
	// Show the dialog, modal.
	editToolbarDialog.exec();
}

void ScoreReadingTrainer::optionsPreferences(){
	//kdDebug() << "Running ScoreReadingTrainer::optionsPreferences()" << endl;

	// Pop up the preferences dialog
	Preferences *preferencesDialog = new Preferences();

	// When the users presses Ok or Apply, Preferences will save the new preferences to the configuration file, when that happens, we want to inmediatly load that configuration file and start acting upon it.
	connect(preferencesDialog, SIGNAL(okClicked()), this, SLOT(loadPreferences()));
	connect(preferencesDialog, SIGNAL(applyClicked()), this, SLOT(loadPreferences()));
	
	// And show it, modal.
	preferencesDialog->exec();

}

void ScoreReadingTrainer::newToolbarConfig(){
	//kdDebug() << "Running ScoreReadingTrainer::newToolbarConfig()" << endl;

	// This slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
	// Recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
	createGUI();

	// And save the new settings.
#if KDE_IS_VERSION(3,1,0)
	applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
#else
	applyMainWindowSettings(KGlobal::config());
#endif
}

void ScoreReadingTrainer::loadPreferences(){
	// kdDebug() << "Running ScoreReadingTrainer::loadPreferences()" << endl;
	// Get our config object for our config file.
	KConfig *config = kapp->config();

	// Set the group program for the training program
	config->setGroup("Program");

	// Load all the configs and apply them to our score widget.
	if(config->readEntry("trainingType", "manual") == "manual"){
		scoreReadingTrainerView->setAutomatic(false);

		QString clef = config->readEntry("clef", "G2");
		if(clef == "G2"){
			scoreReadingTrainerView->setClef(Score::ClefG2);
		} else if(clef == "F4"){
			scoreReadingTrainerView->setClef(Score::ClefF4);
		} else if(clef == "C3"){
			scoreReadingTrainerView->setClef(Score::ClefC3);
		} else if(clef == "C4"){
			scoreReadingTrainerView->setClef(Score::ClefC4);
		}

		QString keySignature = config->readEntry("keySignature", "C");
		if(keySignature == "C"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureC);
			setSharpIcons();
		} else if(keySignature == "G"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureG);
			setSharpIcons();
		} else if(keySignature == "D"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureD);
			setSharpIcons();
		} else if(keySignature == "A"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureA);
			setSharpIcons();
		} else if(keySignature == "E"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureE);
			setSharpIcons();
		} else if(keySignature == "B"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureB);
			setSharpIcons();
		} else if(keySignature == "F#"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureFSharp);
			setSharpIcons();
		} else if(keySignature == "C#"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureCSharp);
			setSharpIcons();
		} else if(keySignature == "F"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureF);
			setFlatIcons();
		} else if(keySignature == "Bb"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureBFlat);
			setFlatIcons();
		} else if(keySignature == "Eb"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureEFlat);
			setFlatIcons();
		} else if(keySignature == "Ab"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureAFlat);
			setFlatIcons();
		} else if(keySignature == "Db"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureDFlat);
			setFlatIcons();
		} else if(keySignature == "Gb"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureGFlat);
			setFlatIcons();
		} else if(keySignature == "Cb"){
			scoreReadingTrainerView->setKeySignature(Score::KeySignatureCFlat);
			setFlatIcons();
		}

		scoreReadingTrainerView->setBottomLimit(config->readNumEntry("bottomLimit", 7));
		scoreReadingTrainerView->setTopLimit(config->readNumEntry("topLimit", 10));

		scoreReadingTrainerView->setAccidentals(config->readBoolEntry("accidentals", false));
	} else {
		scoreReadingTrainerView->setAutomatic(true);
	}
	if(started){
		scoreReadingTrainerView->stop();
		scoreReadingTrainerView->start();
	}
}

void ScoreReadingTrainer::setupActions(){
	//kdDebug() << "Running ScoreReadingTrainer::setupActions()" << endl;

	// Setup the playr start and stop actions.
	startAction = new KAction(i18n("&Start"), "player_play", Qt::Key_S | Qt::CTRL, this, SLOT(start()), actionCollection(), "start");
	stopAction = new KAction(i18n("S&top"), "player_stop", Qt::Key_T | Qt::CTRL, this, SLOT(stop()), actionCollection(), "stop");
	KStdAction::quit(kapp, SLOT(quit()), actionCollection());

	
	// Mapper for the actions for the notes.
	QSignalMapper *noteMapper = new QSignalMapper(this);
	connect( noteMapper, SIGNAL(mapped(int)), scoreReadingTrainerView, SLOT(checkNote(int)));
	
	// Actions for the notes.
	noteActions.append(new KAction(i18n("The musical note, not the letter", "A"), "A", Qt::Key_K, noteMapper, SLOT(map()), actionCollection(), "a"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::A);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "A Sharp / B Flat"), "ASharp", Qt::Key_O, noteMapper, SLOT(map()), actionCollection(), "asharp"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::ASharp);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "B"), "B", Qt::Key_L, noteMapper, SLOT(map()), actionCollection(), "b"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::B);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "C"), "C", Qt::Key_D, noteMapper, SLOT(map()), actionCollection(), "c"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::C);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "C Sharp / D Flat"), "CSharp", Qt::Key_R, noteMapper, SLOT(map()), actionCollection(), "csharp"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::CSharp);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "D"), "D", Qt::Key_F, noteMapper, SLOT(map()), actionCollection(), "d"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::D);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "D Sharp / E Flat"), "DSharp", Qt::Key_T, noteMapper, SLOT(map()), actionCollection(), "dsharp"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::DSharp);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "E"), "E", Qt::Key_G, noteMapper, SLOT(map()), actionCollection(), "e"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::E);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "F"), "F", Qt::Key_H, noteMapper, SLOT(map()), actionCollection(), "f"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::F);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "F Sharp / G Flat"), "FSharp", Qt::Key_U, noteMapper, SLOT(map()), actionCollection(), "fsharp"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::FSharp);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "G"), "G", Qt::Key_J, noteMapper, SLOT(map()), actionCollection(), "g"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::G);
	noteActions.append(new KAction(i18n("The musical note, not the letter", "G Sharp / A Flat"), "GSharp", Qt::Key_I, noteMapper, SLOT(map()), actionCollection(), "gsharp"));
	noteMapper->setMapping( noteActions.last(), ScoreReadingTrainerView::GSharp);

	// Items for showing or hiding the main and the notes toolbar (or all the available toolbars in 3.2).
	#if KDE_IS_VERSION(3,2,0)
		setStandardToolBarMenuEnabled(true);
	#else
		mMainToolBarAction = KStdAction::showToolbar("mainToolBar", actionCollection(), "mainToolBarAction");
		mMainToolBarAction->setText(i18n("Show &Main Toolbar")); // FIXME: is there a better way to do this ?
		mNotesToolBarAction = KStdAction::showToolbar("notesToolBar", actionCollection(), "notesToolBarAction");
		mNotesToolBarAction->setText(i18n("Show &Notes Toolbar")); // FIXME: is there a better way to do this ?
	#endif

	// Items for configuring, the shortcuts, the toolbars and the whole preferences.
	KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
	KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
	KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

	// Let's rock and roll (this is needed to cal to generate all that was set by the above calls).
	createGUI();
}



void ScoreReadingTrainer::setSharpIcons(){
	noteActions[1]->setIcon("ASharp");
	noteActions[4]->setIcon("CSharp");
	noteActions[6]->setIcon("DSharp");
	noteActions[9]->setIcon("FSharp");
	noteActions[11]->setIcon("GSharp");
}

void ScoreReadingTrainer::setFlatIcons(){
	noteActions[1]->setIcon("BFlat");
	noteActions[4]->setIcon("DFlat");
	noteActions[6]->setIcon("EFlat");
	noteActions[9]->setIcon("GFlat");
	noteActions[11]->setIcon("AFlat");
}
#include "ScoreReadingTrainer.moc"
