/*******************************************************************************
 * Score Reading Trainer                                                       *
 * Copyright (C) 2004 by Jos Pablo Ezequiel Fernndez <pupeno@pupeno.com>     *
 *                                                                             *
 * This program is free software; you can redistribute it and/or               *
 * modify it under the terms of the GNU General Public License                 *
 * version 2 as published by the Free Software Foundation.                     *
 *                                                                             *
 * This program is distributed in the hope that it will be useful,             *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 * GNU General Public License for more details.                                *
 *                                                                             *
 * You should have received a copy of the GNU General Public License           *
 * along with this program; if not, write to the Free Software                 *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *******************************************************************************/


#ifndef SCORE_H
#define SCORE_H

#include <qcanvas.h>

// We should declare instead of include and include on the .cpp...
#include <qvaluelist.h>
#include <qwmatrix.h>

class QPoint;
class Note;
/*
 * Score Widget.
 *
 * @short Main class
 * @author Jos Pablo Ezequiel "Pupeno" Fernndez <pupeno@pupeno.com>
 */
class Score : public QCanvasView {
	Q_OBJECT
	public:
		/**
		 * Diferent tipes of cleff this widget can show.
		 */
		enum Clefs { ClefG2 = 0, ClefC3, ClefC4, ClefF4 };

		/**
		 * Diferent tipes of key signatures this widget can show.
		 */
		enum KeySignatures { KeySignatureCFlat = 100, KeySignatureGFlat, KeySignatureDFlat, KeySignatureAFlat, KeySignatureEFlat, KeySignatureBFlat, KeySignatureF,
		                     KeySignatureC, KeySignatureG, KeySignatureD, KeySignatureA, KeySignatureE, KeySignatureB, KeySignatureFSharp, KeySignatureCSharp};

		/**
		 * Diferent tipes of notes
		 */
		enum NoteValues { C = 200, D, E, F, G, A, B };

		/**
		 * Diferent types of Accidental
		 */
		enum Accidentals { None = 300, Natural, Sharp, DoubleSharp, Flat, DoubleFlat };

		/**
		 * Diferent drawings for the note
		 */
		enum NoteDrawings { Up = 400, Down };

		/**
		 * Default Constructor
		 */
		Score(QWidget *parent = 0, const char *name = 0, WFlags f = 0);

		/**
		 * Default Destructor
		 */
		~Score();

		/**
		 * Gets the current clef
		 */
		Clefs clef();

		/**
		 * Sets the clef
		 */
		void setClef(Clefs pClef);

		/**
		 * Get the current key signature
		 */
		KeySignatures keySignature();

		/**
		 * Sets the key signature
		 */
		void setKeySignature(KeySignatures pKeySignature);

		/**
		 * Enqueue a note
		 */
		unsigned int enqueueNote(Note note);

		/**
		 * Get the current note
		 */
		Note note();

	protected:
		virtual void resizeEvent(QResizeEvent *);

	public slots:
		/**
		 * Dequeue a note
		 */
		unsigned int dequeueNote();

		/**
		 * Clear all notes
		 */
		void clearNotes();

		/**
		 * Set the clef to be G2
		 */
		void setClefG2();

		/**
		 * Sets the clef to be C3
		 */
		void setClefC3();

		/**
		 * Sets the clef to be C4
		 */
		void setClefC4();

		/**
		 * Sets the clef to be F4
		 */
		void setClefF4();

		/**
		 * Sets the key signature to C
		 */
		void setKeySignatureC();

		/**
		 * Sets the key signature to G
		 */
		void setKeySignatureG();

		/**
		 * Sets the key signature to D
		 */
		void setKeySignatureD();

		/**
		 * Sets the key signature to A
		 */
		void setKeySignatureA();

		/**
		 * Sets the key signature to E
		 */
		void setKeySignatureE();

		/**
		 * Sets the key signature to B
		 */
		void setKeySignatureB();

		/**
		 * Sets the key signature to F Sharp
		 */
		void setKeySignatureFSharp();

		/**
		 * Sets the key signature to C Sharp
		 */
		void setKeySignatureCSharp();

		/**
		 * Sets the key signature to F
		 */
		void setKeySignatureF();

		/**
		 * Sets the key signature to B Flat
		 */
		void setKeySignatureBFlat();

		/**
		 * Sets the key signature to E Flat
		 */
		void setKeySignatureEFlat();

		/**
		 * Sets the key signature to A Flat
		 */
		void setKeySignatureAFlat();

		/**
		 * Sets the key signature to D Flat
		 */
		void setKeySignatureDFlat();

		/**
		 * Sets the key signature to G Flat
		 */
		void setKeySignatureGFlat();

		/**
		 * Sets the key signature to C Flat
		 */
		void setKeySignatureCFlat();


	private:
		/**
		 * Return the vertical value in pixels of a given score steps
		 */
		int score2pixels(int pPos);

		/**
		 * Given a G clef note position, compensate it to the current clef.
		 */
		int clefCompensate(int pPos);
		
		/**
		 * Clef
		 */
		Clefs mClef;

		/**
		 * Key signature
		 */
		KeySignatures mKeySignature;

		/**
		 * Queue of notes
		 */
		QValueList<Note> mNotes;

		/**
		 * Main canvas
		 */
		QCanvas *mMainCanvas;

		/**
		 * Main canvas view
		 */
		QCanvasView *mMainCanvasView;

		/**
		 * Canvas sprite for the clef
		 */
		QCanvasSprite *mClefSprite;

		/**
		 * Canvas sprite for the key signature
		 */
		QCanvasItemList mKeySignatureSprites;

		/**
		 * Canvases sprite for the notes
		 */
		QCanvasItemList mNotesSprites;

		/**
		 * Canvases sprite for the accidentals
		 */
		QCanvasItemList mAccidentalsSprites;

		/**
		 * Canvases that goes with the notes outside the five lines
		 */
		QCanvasItemList mLinesSprites;

		/**
		 * Map for the positions of the general objects in the canvas
		 */
		QMap<QString, int> generalPositioning;

		/**
		 * Positioning for key signatures
		 */
		int keySignaturePositioning;

		/**
		 * Note positioning
		 */
		QMap<int, int> notePositioning;

		/**
		 * Center (in pixels) of the graphical objects
		 */
		QMap<int, int> center;

		/**
		 * Position in Score steps of the graphical objects
		 */
		QMap<int, int> position;
};


/*
 * Note class
 *
 * @short Main class
 * @author Jos Pablo Ezequiel "Pupeno" Fernndez <pupeno@pupeno.com>
 */
class Note {
	public:
		/**
		 * Creates a note with the given (right) parameters or defaults
		 */
		Note(Score::NoteValues pNoteValue = Score::C, Score::Accidentals pAccidental = Score::None, int pOctave = 0);

		/**
		 * Get the note value
		 */
		Score::NoteValues noteValue();

		/**
		 * Set the note Value
		 */
		bool setNoteValue(Score::NoteValues pNoteValue);

		/**
		 * Get the accidental
		 */
		Score::Accidentals accidental();

		/**
		 * Set the accidental
		 */
		bool setAccidental(Score::Accidentals pAccidental);

		/**
		 * Get the octave
		 */
		int octave();

		/**
		 * Set the octave
		 */
		void setOctave(int pOctave);

	private:
		/**
		 * The note itself
		 */
		Score::NoteValues mNoteValue;

		/**
		 * The accidental of the note
		 */
		Score::Accidentals mAccidental;

		/**
		 * The octave of this note
		 */
		int mOctave;
};
#endif
