/***************************************************************************
 *   copyright           : (C) 2003 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "options.h"
#include "common.h"

#include <string.h>

void args_init (struct args_t* args) {
  args->port.device = NULL;
  args->port.baud = NULL;
  args->port.timeout = 100;
  args->port.ignorebits = 1;
  args->port.startdelay = 0;

  args->myparams.text=NULL;
  args->myparams.number=NULL;
  args->myparams.mem=NULL;
  args->myparams.slot=SCMXX_SLOT_UNDEFINED;

  args->mysmsopts.flash=0;
  args->mysmsopts.srr=0;
  args->mysmsopts.unicode=0;
  args->mysmsopts.direct=0;
  args->mysmsopts.sort_mode=NULL;

  args->scmxx_action=0;
  args->scmxx_ftype=0;

  args->myFILE=NULL;
  args->myPIPE=NULL;

  args->info = 0;
  args->meminfo = 0;
  args->lockinfo = 0;
  args->time = 0;
  args->lock = NULL;
  args->state_change = 0;
  args->smsc = 0;
  args->dial = 0;
  args->hangup = 0;
  args->reset = 0;
  args->help = 0;
  args->version = 0;

  args->verbose = 0;
  args->pin = NULL;
  args->system_charset = NULL;
}

#include "helper.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

void arg_fill_slot (char* argument, struct args_t* args) {
  if (is_number(argument)) {
    args->myparams.slot = atoi(argument);
  } else {
    if(!strcmp(argument,"all")) {
      args->myparams.slot = SCMXX_SLOT_ALL;
    } else if(!strcmp(argument,"sent")) {
      args->myparams.slot = SCMXX_SLOT_SMS_SENT;
    } else if(!strcmp(argument,"unsent")) {
      args->myparams.slot = SCMXX_SLOT_SMS_UNSENT;
    } else if(!strcmp(argument,"read")) {
      args->myparams.slot = SCMXX_SLOT_SMS_READ;
    } else if(!strcmp(argument,"unread")) {
      args->myparams.slot = SCMXX_SLOT_SMS_UNREAD;
    } else if (!strcmp(argument,"last")) {
      args->myparams.slot = SCMXX_SLOT_LAST;
    } else {
      args->myparams.slot = SCMXX_SLOT_UNDEFINED;
    }
  }
}

void arg_fill (int short_opt, const char* long_opt,
	       char* argument, struct args_t* args) {
  int temp;

  switch (short_opt) {
  case 'b': //--baud
    args->port.baud = argument;
    break;
  case 'd': //--device
    args->port.device = argument;
    break;
  case 'g':
    args->scmxx_action |= SCMXX_ACTION_GET;
    break;
  case 'h':
    args->help=1;
    break;
  case 'i':
    args->info=1;
    break;
  case 'l':
    arg_fill_slot(argument,args);
    break;
  case 'm':
    args->myparams.mem = argument;
    break;
  case 'n':
    if (str_len(argument)) args->myparams.number=argument;
    break;
  case 'o':
    args->myFILE=argument;
    break;
  case 'p':
    args->myPIPE=argument;
    break;
  case 'q':
    args->verbose--;
    break;
  case 'r':
    args->scmxx_action |= SCMXX_ACTION_REMOVE;
    break;
  case 's': 
    args->scmxx_action |= SCMXX_ACTION_SEND;
    break;
  case 't':
    if (str_len(argument)) args->myparams.text=argument;
    break;
  case 'v':
    args->verbose++;
    break;
  case 'N':
    args->scmxx_ftype = SCMXX_FTYPE_BINARY;
    break;
  case 'P':
    args->scmxx_ftype = SCMXX_FTYPE_PBOOK;
    break;
  case 'S':
    args->scmxx_ftype = SCMXX_FTYPE_SMS;
    break;
  case 'V':
    args->version=1;
    break;
  case 0:
    if (str_len(long_opt)) {
      if (!strcmp(long_opt,"pin")) {
	args->pin = argument;
      } else if (!strcmp(long_opt,"unicode")) {
	args->mysmsopts.unicode=1;
      } else if (!strcmp(long_opt,"direct")) {
	args->mysmsopts.direct=1;
      } else if (!strcmp(long_opt,"srr")) {
	args->mysmsopts.srr=1;
      } else if (!strcmp(long_opt,"flash")) {
	args->mysmsopts.flash=1;
      } else if (!strcmp(long_opt,"ignore-serial-bits")) {
	args->port.ignorebits=1;
      } else if (!strcmp(long_opt,"keep-serial-bits")) {
	args->port.ignorebits=0;
      } else if (!strcmp(long_opt,"device-timeout")) {
	if (argument) {
	  temp = atoi(argument);
	  if (temp < 1) {
	    args->port.timeout = 10;
	  } else if (1 <= temp && temp <= 25) {
	    args->port.timeout = 10*temp;
	  } else if (temp > 25) {
	    args->port.timeout = 0;
	  }
	}
      } else if (!strcmp(long_opt,"start-delay")) {
	args->port.startdelay = atoi(argument);
      } else if (!strcmp(long_opt,"reset")) {
	args->reset=1;
      } else if (!strcmp(long_opt,"set-time")) {
	args->time=1;
      } else if (!strcmp(long_opt,"set-smsc")) {
	args->smsc=1;
      } else if (!strcmp(long_opt,"lock")) {
	args->lock=argument;
      } else if (!strcmp(long_opt,"enable")) {
	args->state_change=1;
      } else if (!strcmp(long_opt,"disable")) {
	args->state_change=-1;
      } else if (!strcmp(long_opt,"dial")) {
	args->dial=1;
      } else if (!strcmp(long_opt,"hangup")) {
	args->hangup=1;
      } else if (!strcmp(long_opt,"mem-info")) {
	args->meminfo = 1;
      } else if (!strcmp(long_opt,"lock-info")) {
	args->lockinfo = 1;
      } else if (!strcmp(long_opt,"sort")) { //old, compat
	args->mysmsopts.sort_mode = argument;
      } else if (!strcmp(long_opt,"system-charset")) {
	args->system_charset = argument;
      }
    }
    break;
  }
}

#include "config.h"

void use_envvars (struct args_t* args) {
  //using env. var. SCMXX_BAUD
  if (str_len(args->port.baud) == 0 &&
      (args->port.baud=getenv("SCMXX_BAUD")) == NULL) {
    args->port.baud = TTYSPEED;
  }
  //using env. var. SCMXX_TTY
  if (str_len(args->port.device) == 0 &&
      (args->port.device=getenv("SCMXX_TTY")) == NULL) {
    args->port.device = TTYPORT;
  }
}

//gettext workaround
#ifdef _
#undef _
#endif
#define _(s) s
struct args_def_type args_type_list[] = {
  { 0, _("options"), 0 },
  { 1, _("actions"), 1 },
  { 2, _("memory types"), 0 },
  { 3, _("memory type options"), 0 },
  { 4, _("SMS options"), 0 },
  ARGS_DEF_TYPE_LAST
};

struct args_def args_list[] = {
  {"baud",'b',ARGS_PARAM_REQ,_("rate"),0,
   _("specify another than the compiled in baudrate")},
  {"binary",'N',ARGS_PARAM_NONE,NULL,2,
   _("binary file transfer")},
  {"device",'d',ARGS_PARAM_REQ,_("device"),0,
   _("specify another than the compiled in device")},
  {"device-timeout",0,ARGS_PARAM_REQ,_("seconds"),0,
   _("return after this time if there is no answer")},
  {"dial",0,ARGS_PARAM_NONE,NULL,1,
   _("dials a number (requires option -n)")}, //since 0.7.4
  {"direct",0,ARGS_PARAM_NONE,NULL,4,
   _("send/receive without storing in the phone")},
  {"disable",0,ARGS_PARAM_NONE,NULL,1,
   _("disable e.g. a lock")}, //since 0.8.0
  {"enable",0,ARGS_PARAM_NONE,NULL,1,
   _("enable e.g. a lock")}, //since 0.8.0
  {"flash",0,ARGS_PARAM_NONE,NULL,4,
   _("make the sms appear directly (if supported by the receiving entity)")},
  {"get",'g',ARGS_PARAM_NONE,NULL,1,
   _("get something from the phone")},
  {"hangup",0,ARGS_PARAM_NONE,NULL,1,
   _("hangup all active calls")}, //since 0.8.0
  {"help",'h',ARGS_PARAM_NONE,NULL,0,
   _("print this message")},
  {"ignore-serial-bits",0,ARGS_PARAM_NONE,NULL,0,
   _("do not use current serial port settings as base (default)")},
  {"info",'i',ARGS_PARAM_NONE,NULL,1,
   _("show various information")},
  {"keep-serial-bits",0,ARGS_PARAM_NONE,NULL,0,
   _("use current serial port settings as base")},
  {"lock",0,ARGS_PARAM_REQ,_("lock"),1,
   _("selects a lock to change (might need option --pin)")}, //since 0.8.0
  {"lock-info",0,ARGS_PARAM_NONE,NULL,1,
   _("show lock information")},
  {"mem",'m',ARGS_PARAM_REQ,_("mem"),3,
   _("select a memory")},
  {"mem-info",0,ARGS_PARAM_NONE,NULL,1,
   _("show memory information")},
  {"number",'n',ARGS_PARAM_REQ,_("number"),0,
   _("specify number on send")},
  {"out",'o',ARGS_PARAM_REQ,_("file"),0,
   _("specify a file when getting ('-' for stdout)")},
  {"pbook",'P',ARGS_PARAM_NONE,NULL,2,
   _("phonebook")},
  {"pin",0,ARGS_PARAM_REQ,"PIN",4,
   _("use this pin if one is needed")},
  {"pipe",'p',ARGS_PARAM_REQ,_("program"),0,
   _("specify a pipe to use")},
  {"quiet",'q',ARGS_PARAM_NONE,NULL,0,
   _("decreases verbosity by 1")},
  {"remove",'r',ARGS_PARAM_NONE,NULL,1,
   _("removes something from the phone")},
  {"reset",0,ARGS_PARAM_NONE,NULL,1,
   _("sends some special characters")},
  {"send",'s',ARGS_PARAM_NONE,NULL,1,
   _("sends something to the phone")},
  {"set-smsc",0,ARGS_PARAM_NONE,NULL,1,
   _("set the SMSC number (requires option -n)")},
  {"set-time",0,ARGS_PARAM_NONE,NULL,1,
   _("synchronize time on phone")},
  {"slot",'l',ARGS_PARAM_REQ,_("slot"),3,
   _("select a slot or slot group")},
  {"sms",'S',ARGS_PARAM_NONE,NULL,2,
   _("short messages")},
  {"sort",0,ARGS_PARAM_REQ,_("method"),4,
   _("sort short message on getting(type or slot)")}, //since 0.7.3
  {"srr",0,ARGS_PARAM_NONE,NULL,4,
   _("request a status report from the SMSC")},
  {"start-delay",0,ARGS_PARAM_REQ,_("seconds"),0,
   _("wait this time after opening device")},
  {"system-charset",0,ARGS_PARAM_REQ,_("charset"),0,
   _("use charset for input/output from/to system")}, //since 0.7.5
  {"text",'t',ARGS_PARAM_REQ,_("text"),0,
   _("specify content on send")},
  {"unicode",0,ARGS_PARAM_NONE,NULL,4,
   _("use UCS-2 (unicode) as charset")},
  {"verbose",'v',ARGS_PARAM_NONE,NULL,0,
   _("increases verbosity by 1")},
  {"version",'V',ARGS_PARAM_NONE,NULL,0,
   _("print the version number")},
  ARGS_DEF_LAST
};
