/* tree_manipulation.h - GtkTreeView manipulation code for sciteproj
 *
 *  Copyright 2006 Roy Wood, 2009 Andreas Ronnquist
 *
 * This file is part of Sciteproj.
 * 
 * Sciteproj is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sciteproj is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Sciteproj.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
#ifndef __TREE_MANIPULATION_HEADER_
#define __TREE_MANIPULATION_HEADER_

#include <gtk/gtk.h>


// Node relative-position indicators

enum NodePosition {
	ADD_BEFORE,
	ADD_AFTER,
	ADD_CHILD
};


// Mnemonic identifiers for the columns in the GtkTreeStore

enum { 
	COLUMN_ITEMTYPE = 0,
	COLUMN_FILEPATH, 
	COLUMN_FILENAME, 
	COLUMN_FILESIZE, 
	COLUMN_FONTWEIGHT,
	COLUMN_FONTWEIGHTSET,
	COLUMN_ICON,
	COLUMN_EXPANDED,
	COLUMN_EOL
};


// The type of each row in the tree datamodel

enum {
	ITEMTYPE_GROUP,
	ITEMTYPE_FILE
};



typedef gint(*StringCompareFunction)(gconstpointer,gconstpointer);


// Get the GTKTreeStore
GtkTreeStore* get_treestore(GError **err);

// Get the project file directory
const gchar* get_project_directory();
gchar* get_project_filepath();

// Clear the tree
extern void empty_tree(GtkTreeStore *treeStore);

// Mark the project as dirty or clean (i.e. modified and in need of saving)
void set_project_dirty_status(gboolean isDirty);

// Get the status of the project
gboolean project_is_dirty();

// Prompting the user to save the project if it is dirty
void prompt_user_to_save_project();

// Save the current project, or pop up an error dialog if something bad happened
gboolean save_project(gchar *proj_filepath,GError **err);

// Allow user to select a project file to load (save current project first, if appropriate)
gboolean load_project(gchar *projectPath, GError **err);

// Allow user to select and add files to the project
gboolean add_files_to_project(GtkTreeIter *parentIter, GError **err);

// Add a file node to a GtkTreeModel
extern gboolean add_tree_file(GtkTreeIter *currentIter, enum NodePosition position, const gchar* filepath, GtkTreeIter *newIter, gboolean makeRelative, GError **err);

//gboolean add_tree_filelist(GtkTreeIter *parentIter, GSList *fileList, GError **err);
gboolean add_tree_filelist(GtkTreeIter *parentIter, GSList *fileList, GError **err);

// Add a group node to a GtkTreeModel
extern gboolean add_tree_group(GtkTreeIter *parentIter, enum NodePosition position, const gchar* groupname, gboolean expanded, GtkTreeIter *newIter, GError **err);

// Remove a node from a GtkTreeModel
extern gboolean remove_tree_node(GtkTreeIter *iter, GError **err);

// Remove a node from a GtkTreeModel
extern gboolean set_tree_node_name(GtkTreeIter *iter, const gchar *newContents, GError **err);

extern gboolean set_tree_node_expanded(GtkTreeIter *iter, gboolean expanded, GError **err);

extern gboolean set_tree_node_icon(GtkTreeIter *iter, GdkPixbuf *pixbuf, GError **err);

// Copy a node in the tree (including children)
gboolean copy_tree_node(GtkTreeIter *srcIter, GtkTreeIter *dstIter, enum NodePosition position, GtkTreeIter *newIter, GError **err);

gchar *get_path_string(GtkTreeIter *iter);

gint compare_strings_smaller(gconstpointer a,gconstpointer b);
gint compare_strings_bigger(gconstpointer a,gconstpointer b);

void sort_children(GtkTreeIter *node,GError **err,StringCompareFunction compare_func);

gboolean tree_contains(gchar *value);

void unref_tree_graphics();

#endif /*__TREE_MANIPULATION_HEADER_*/
