/* rename.c - rename code for sciteproj
 *
 *  Copyright 2009,2010 Andreas Ronnquist
 *
 * This file is part of Sciteproj.
 * 
 * Sciteproj is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sciteproj is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Sciteproj.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "tree_manipulation.h"
#include "gui.h"

#include "rename.h"



/**
 *	is_name_valid
 * returns TRUE if string isn't NULL, and the string isn't just filled with spaces.
 * This code doesn't give an error for the "\0" string, that is given when pressing cancel
 */
gboolean is_name_valid(gchar *instring)
{
	gboolean result=TRUE;
	
	// Check for NULL
	if (instring==NULL) return FALSE;
	
	// Check if the string is just filled with spaces
	gchar *nullstring=g_strdup(instring);
	nullstring=g_strchug(g_strchomp(nullstring));
	if (g_strcmp0(nullstring,"")==0) result=FALSE;
	
	g_free(nullstring);
	
	return result;
}


/**
 *
 */
gboolean dialog_response_is_exit(gint test)
{
	gboolean result=FALSE;
	
	if ((test==GTK_RESPONSE_REJECT) || (test==GTK_RESPONSE_CANCEL) || 
		 (test==GTK_RESPONSE_DELETE_EVENT) || (test==GTK_RESPONSE_NONE)) {
		result=TRUE;
	}
	
	return result;
}




/**
 *
 */
void rename_key_pressed_cb()
{
	GError *err = NULL;
	GtkWidget *dialog = NULL;
	//gint dialogResponse;
	//gchar *nodename = NULL;
	GtkTreeIter nodeIter;
	
	gint selected_rows=0;
	
	gboolean multiple_selected=FALSE;
	
	GtkTreeSelection *treeSelect;
	
	treeSelect=gtk_tree_view_get_selection(GTK_TREE_VIEW(sTreeView));
	GtkTreeModel *tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(sTreeView));
	
	
	selected_rows=gtk_tree_selection_count_selected_rows(treeSelect);
	if (selected_rows>1) {
		multiple_selected=TRUE;
	}
	
	// Confirm removal from project
	
	if (!multiple_selected) {
		// Remove the node
	
		GList *list_of_selected=gtk_tree_selection_get_selected_rows(treeSelect,NULL);
		
		if (list_of_selected) {
			list_of_selected=g_list_first(list_of_selected);
			
			GtkTreePath *path=(GtkTreePath*)list_of_selected->data;
			
			gtk_tree_model_get_iter (tree_model, &nodeIter, path);
		
		}
		
		int nodeType=-1;
		
		gchar *name;
		gtk_tree_model_get(tree_model, &nodeIter, COLUMN_ITEMTYPE, &nodeType, -1);
		gtk_tree_model_get(tree_model, &nodeIter, COLUMN_FILENAME, &name,-1);
		
		if (nodeType==ITEMTYPE_FILE) {
			goto EXITPOINT;
		}
		
			
		gboolean cancel_result=FALSE;
		gchar *newGroupName=rename_dialog(name,&cancel_result);
		
		if (!cancel_result) {
			if (is_name_valid(newGroupName)) {
				// Rename the group
				
				if (!set_tree_node_name(&nodeIter, newGroupName, &err)) {
					GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while renaming the group: %s", err->message);
					
					gtk_dialog_run(GTK_DIALOG(errDialog));
					
					gtk_widget_destroy(errDialog);
				}
			} else {
				GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Invalid name!");
				
				gtk_dialog_run(GTK_DIALOG(errDialog));
				
				gtk_widget_destroy(errDialog);
			}
		}
				
		g_list_foreach (list_of_selected, (GFunc)(gtk_tree_path_free), NULL);
		g_list_free (list_of_selected);
	}
	
EXITPOINT:
	
	if (err) g_error_free(err);
	if (dialog) gtk_widget_destroy(dialog);		
}



/**
 * Callback to handle "activation" message from a GtkEntry widget.  This code allows a return/enter
 * keystroke to trigger dismissal of the dialog box containing the GtkEntry.
 *
 * @param entry is the GtkEntry widget
 * @param dialog is the dialog that contains the GtkEntry widget
 */
void entry_widget_activated_cb(GtkEntry *entry, gpointer dialog)
{
	if (dialog != NULL) {
		gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);
	}
}


/**
 *	rename_dialog
 * opens a dialog that is used for the rename fucntionality - returns the entered name.
 */
gchar *rename_dialog(gchar *source_name,gboolean *cancel_result)
{
	gchar *newGroupName=NULL;
	GtkWidget *dialog = NULL;
	GtkWidget* gtkEntry = NULL;
	GtkWidget* gtkLabel = NULL;
	GtkWidget* table = NULL;
	GtkAttachOptions options = (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL);
	gint dialogResponse;
	
	*cancel_result=FALSE;
// Create a dialog box with a nicely-centered text entry widget
	
	dialog = gtk_dialog_new_with_buttons("Choose New Group Name", NULL, GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	
	g_signal_connect(dialog, "response",  G_CALLBACK(gtk_widget_hide), dialog);
	
	gtk_container_set_border_width(GTK_CONTAINER(dialog), 0);
	
	
	table = gtk_table_new(1, 2, FALSE);
	
	// set the text to the current folder name
	
	gtkLabel = gtk_label_new("Enter new name of group:");
	gtk_table_attach(GTK_TABLE(table), gtkLabel, 0, 1, 0, 1, options, options, 5, 5);
	
	gtkEntry = gtk_entry_new();
	if (source_name==NULL) {
		source_name=g_strdup(clicked_node.name);
	}
	
	gtk_entry_set_text(GTK_ENTRY(gtkEntry),source_name);
	
	g_signal_connect(GTK_OBJECT(gtkEntry), "activate", G_CALLBACK(entry_widget_activated_cb), dialog);
	gtk_table_attach(GTK_TABLE(table), gtkEntry, 1, 2, 0, 1, options, options, 5, 5);
	
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
	
	gtk_widget_show_all (dialog);
	
	
	// Let the user enter a name or cancel the whole thing
	gchar *result=NULL;
		
	
	dialogResponse=gtk_dialog_run(GTK_DIALOG(dialog));
	if (dialog_response_is_exit(dialogResponse)) {
		*cancel_result=TRUE;
		goto EXITPOINT;
	}
	
	newGroupName = (gchar*)gtk_entry_get_text(GTK_ENTRY(gtkEntry));
	
	if (newGroupName == NULL || *newGroupName == '\0') {
		GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Invalid group name");
		
		gtk_dialog_run(GTK_DIALOG(errDialog));
		
		gtk_widget_destroy(errDialog);
		
		*cancel_result=TRUE;
		
		goto EXITPOINT;
	}
	
	result=g_strdup(newGroupName);
	
	
EXITPOINT:	
	if (dialog) gtk_widget_destroy(dialog);
	
		
	return result;
}

/**
 * Rename a group
 */
void popup_rename_group_cb()
{
	GError *err = NULL;
	gchar *newGroupName=NULL;
	
	// We can only rename groups!
	
	if (!clicked_node.valid || clicked_node.type != ITEMTYPE_GROUP) {
		goto EXITPOINT;
	}
	
	gboolean cancel_result=FALSE;
	
	newGroupName=rename_dialog(NULL,&cancel_result);
	
	// Rename the group
	if (!cancel_result) {
		if (is_name_valid(newGroupName)) {
		
			if (!set_tree_node_name(&(clicked_node.iter), newGroupName, &err)) {
				GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while renaming the group: %s", err->message);
				
				gtk_dialog_run(GTK_DIALOG(errDialog));
				
				gtk_widget_destroy(errDialog);
			}
		} else {
				GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Invalid name!");
				
				gtk_dialog_run(GTK_DIALOG(errDialog));
				
				gtk_widget_destroy(errDialog);			
		}
	}
	
	
EXITPOINT:
	
	// Destroying the dialog should also destroy the table, label, and entry widgets
	
	if (err) g_error_free(err);
	
	// Do NOT free newGroupName, since that is owned by the GtkEntry widget!
}


