/* gui.c - GUI code for sciteproj
 *
 *  Copyright 2006 Roy Wood, 2009,2010 Andreas Ronnquist
 *
 * This file is part of Sciteproj.
 * 
 * Sciteproj is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sciteproj is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Sciteproj.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdlib.h>

#include "gui.h"
#include "drag_drop.h"
#include "tree_manipulation.h"
#include "scite_utils.h"
#include "string_utils.h"
#include "prefs.h"
#include "statusbar.h"
#include "graphics.h"
#include "about.h"
#include "properties_dialog.h"
#include "file_utils.h"

#include <string.h>
#include <sys/stat.h>
#include <glib.h>
#include <gtk/gtk.h>

#include <gdk/gdkkeysyms.h>

#include "search.h"
#include "rename.h"


#define APP_SCITEPROJ_ERROR g_quark_from_static_string("APP_GUI_ERROR")


// Forward-declare static functions

gboolean dialog_response_is_exit(gint test);

static gint window_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data);
static void tree_row_activated_cb(GtkTreeView *treeView, GtkTreePath *path, GtkTreeViewColumn *column, gpointer userData);
static gboolean mouse_button_pressed_cb(GtkWidget *treeView, GdkEventButton *event, gpointer userData);

static void ask_name_add_group(GtkTreeIter *nodeIter);

static void row_expand_or_collapse_cb(GtkTreeView *treeview, GtkTreeIter *arg1, GtkTreePath *arg2, gpointer user_data);

static void menu_add_widget_cb(GtkUIManager *ui, GtkWidget *widget, GtkContainer *container);

static void quit_menu_cb();
static void about_menu_cb();
static void usage_menu_cb();
static void saveproject_menu_cb();
static void saveproject_as_menu_cb();
static void openproject_menu_cb();
static void addfile_menu_cb();
static void creategroup_menu_cb();

static void popup_add_files_cb();
static void popup_open_file_cb();
static void popup_add_group_cb();
static void popup_remove_node_cb();

static void expand_all_items_cb();
static void collapse_all_items_cb();

static void removeitem_menu_cb();

static void sort_ascending_cb();
static void sort_descending_cb();

static void edit_options_cb();

gboolean is_name_valid(gchar *instring);

gboolean key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer userData);

gchar *window_saved_title=NULL;


// Menu definitions

static gchar *sMenuDefXML = (gchar*)\
	"<ui> \
		<menubar> \
			<menu name=\"FileMenu\" action=\"FileMenuAction\"> \
				<menuitem name=\"OpenProjectItem\" action=\"OpenProjectAction\" /> \
				<menuitem name=\"SaveProjectItem\" action=\"SaveProjectAction\" /> \
				<menuitem name=\"SaveProjectAsItem\" action=\"SaveProjectAsAction\" /> \
				<separator/> \
				<menuitem name=\"ExitItem\" action=\"ExitAction\" /> \
			</menu> \
			<menu name=\"EditMenu\" action=\"EditMenuAction\"> \
				<menuitem name=\"CreateGroupItem\" action=\"CreateGroupAction\" /> \
				<menuitem name=\"AddFileItem\" action=\"AddFileAction\" /> \
				<menuitem name=\"RemoveFileItem\" action=\"RemoveFileAction\"/> \
				<separator/> \
				<menuitem name=\"ExpandAllGroupsItem\" action=\"ExpandAllGroupsAction\" /> \
				<menuitem name=\"CollapseAllGroupsItem\" action=\"CollapseAllGroupsAction\" /> \
				<separator/> \
				<menuitem name=\"Edit Options\" action=\"EditOptionsAction\" /> \
			</menu> \
			<menu name=\"HelpMenu\" action=\"HelpMenuAction\"> \
				<menuitem name=\"UsageItem\" action=\"UsageAction\"/> \
				<menuitem name=\"AboutItem\" action=\"AboutAction\"/> \
			</menu> \
		</menubar> \
		<popup name=\"GeneralPopup\" action=\"GeneralPopupAction\"> \
			<menuitem name=\"AddFilesPopupItem\" action=\"AddFilesPopupAction\"/> \
			<menuitem name=\"AddGroupPopupItem\" action=\"AddGroupPopupAction\"/> \
		</popup> \
		<popup name=\"FilePopup\" action=\"FilePopupAction\"> \
			<menuitem name=\"OpenFilePopupItem\" action=\"OpenFilePopupAction\"/> \
			<menuitem name=\"RemoveFilePopupItem\" action=\"RemoveFilePopupAction\"/> \
			<separator/> \
			<menuitem name=\"PropertiesPopupItem\" action=\"PropertiesPopupAction\"/> \
		</popup> \
		<popup name=\"GroupPopup\" action=\"GroupPopupAction\"> \
			<menuitem name=\"AddFilesToGroupPopupItem\" action=\"AddFilestoGroupPopupAction\"/> \
			<menuitem name=\"AddSubgroupPopupItem\" action=\"AddSubgroupPopupAction\"/> \
			<menuitem name=\"RenameGroupPopupItem\" action=\"RenameGroupPopupAction\"/> \
			<menuitem name=\"RemoveGroupPopupItem\" action=\"RemoveGroupPopupAction\"/> \
			<separator/> \
			<menuitem name=\"SortAscendingItem\" action=\"SortAscendingAction\"/> \
			<menuitem name=\"SortDescendingItem\" action=\"SortDescendingAction\"/> \
			<separator/> \
			<menuitem name=\"ProperiesGroupPopupItem\" action=\"PropertiesGroupPopupAction\"/> \
		</popup> \
	</ui>";



static GtkActionEntry sMenuActions[] = 
{
	{ "FileMenuAction", NULL, "_File" },
	{ "EditMenuAction", NULL, "_Edit" },
	{ "HelpMenuAction", NULL, "_Help" },
	
	{ "OpenProjectAction", GTK_STOCK_OPEN, "_Open Project", "<control>O", "Open Project", G_CALLBACK(openproject_menu_cb) },
	{ "SaveProjectAction", GTK_STOCK_SAVE, "_Save Project", "<control>S", "Save Project", G_CALLBACK(saveproject_menu_cb) },
	{ "SaveProjectAsAction", GTK_STOCK_SAVE_AS, "Save Project As...", "<control><shift>S", "Save Proeject As", G_CALLBACK(saveproject_as_menu_cb) },
	{ "ExitAction", GTK_STOCK_QUIT, "_Exit", "<control>Q", "Exit", G_CALLBACK(quit_menu_cb) },
	
	{ "CreateGroupAction", GTK_STOCK_DIRECTORY, "Create _Group", "", "Create a group node in the project", G_CALLBACK(creategroup_menu_cb) },
	{ "AddFileAction", GTK_STOCK_FILE, "Add _File", "", "Add a file to the project", G_CALLBACK(addfile_menu_cb) },
	{ "RemoveFileAction", GTK_STOCK_DELETE, "Remove File(s)", "", "Remove selected files from the project", G_CALLBACK(removeitem_menu_cb) },
	
	{ "ExpandAllGroupsAction", NULL, "Expand All Groups", "<control><shift>E", "Expand All Groups", G_CALLBACK(expand_all_items_cb) },
	{ "CollapseAllGroupsAction", NULL, "Collapse All Groups", "<control><shift>C", "Collapse All Groups", G_CALLBACK(collapse_all_items_cb) },
	
	{ "UsageAction", GTK_STOCK_HELP, "_Usage", "", "Show command line usage", G_CALLBACK(usage_menu_cb) },
	{ "AboutAction", GTK_STOCK_ABOUT, "_About", "", "Show information about this application", G_CALLBACK(about_menu_cb) },
	
	{ "AddFilesPopupAction", GTK_STOCK_FILE, "Add Files", "", "Add files to the project", G_CALLBACK(popup_add_files_cb) },
	{ "AddGroupPopupAction", GTK_STOCK_DIRECTORY, "Add Group", "", "Add a group to the project", G_CALLBACK(popup_add_group_cb) },
	
	{ "AddFilestoGroupPopupAction", GTK_STOCK_FILE, "Add Files to Group", "", "Add files to an existing group", G_CALLBACK(popup_add_files_cb) },
	{ "AddSubgroupPopupAction", GTK_STOCK_DIRECTORY, "Add Subgroup to Group", "", "Add a subgroup to an existing group", G_CALLBACK(popup_add_group_cb) },
	{ "RenameGroupPopupAction", GTK_STOCK_EDIT, "Rename Group", "", "Rename a group", G_CALLBACK(popup_rename_group_cb) },
	{ "RemoveGroupPopupAction", GTK_STOCK_DELETE, "Remove Group From Project", "", "Remove a group and its children from the project", G_CALLBACK(popup_remove_node_cb) },
	{ "SortAscendingAction", GTK_STOCK_SORT_ASCENDING, "Sort Group Ascending","","Sort the filenames ascending",G_CALLBACK(sort_ascending_cb) },
	{ "SortDescendingAction", GTK_STOCK_SORT_DESCENDING, "Sort Group Descending","","Sort the filenames descending",G_CALLBACK(sort_descending_cb) },
	{ "PropertiesGroupPopupAction", GTK_STOCK_PROPERTIES, "Group Properties", "", "Show group properties", G_CALLBACK(group_properties_cb) },
	{ "EditOptionsAction", GTK_STOCK_PROPERTIES, "Edit Options", "", "Edit Program Options", G_CALLBACK(edit_options_cb) },
	
	{ "OpenFilePopupAction", GTK_STOCK_OPEN, "Open File in SciTE", "", "Open a file in SciTE", G_CALLBACK(popup_open_file_cb) },
	{ "RemoveFilePopupAction", GTK_STOCK_DELETE, "Remove File From Project", "", "Remove a file from the project", G_CALLBACK(popup_remove_node_cb) },
	{ "PropertiesPopupAction", GTK_STOCK_PROPERTIES, "File Properties", "", "Show file properties", G_CALLBACK(file_properties_cb) }

};

static guint sNumMenuActions = G_N_ELEMENTS(sMenuActions);



static TreeViewDragStruct sDragStruct;

static GtkWidget *sMainWindow = NULL;
GtkWidget *sTreeView = NULL;

static GtkWidget *sGroupPopupMenu = NULL;
static GtkWidget *sFilePopupMenu = NULL;
static GtkWidget *sGeneralPopupMenu = NULL;

ClickedNode clicked_node;

static GtkActionGroup *sActionGroup = NULL;
static GtkUIManager *sGtkUIManager = NULL;

GtkCellRenderer *textCellRenderer = NULL;
GtkCellRenderer *pixbuffCellRenderer = NULL;

GtkWidget *scrolledWindow = NULL;

/**
 * Initialize globals (i.e. create the main window and populate it).  This is a long chunk of code.
 *
 * @return TRUE on success, FALSE on failure (further details returned in err)
 *
 * @param err returns any errors
 */
gboolean setup_gui(GError **err)
{
	gboolean resultCode = FALSE;
	GtkTreeSelection *selection = NULL;
	GtkWidget *vbox=NULL;
	GtkTreeViewColumn *column1 = NULL;
	GtkTreeViewColumn *column2 = NULL;

	GtkTargetEntry dragTargets[] = { { (gchar*)"text/uri-list", 0, DND_URI_TYPE } };
	GtkTreeStore *treeStore = NULL;
	GtkAccelGroup* accelgroup = NULL;
	GError *tempErr = NULL;
	
	clicked_node.valid=FALSE;
	clicked_node.name=NULL;
	clicked_node.type=-1;
	
	window_saved_title=g_strdup_printf("[UNTITLED]");
	
	// Create top-level window, configure it
	
	if (!(sMainWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL))) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create main window, gtk_window_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	// TODO: call these in load_graphics instead
	
	if (!(load_graphics(err))) {
		goto EXITPOINT;
	}
	
	gtk_window_set_icon(GTK_WINDOW(sMainWindow),program_icon_pixbuf);
	gtk_window_set_default_icon(program_icon_pixbuf);
	
	gtk_window_set_title(GTK_WINDOW(sMainWindow), window_saved_title);
	
	gtk_container_set_border_width(GTK_CONTAINER(sMainWindow), 0);	//3
	g_signal_connect(G_OBJECT(sMainWindow), "delete_event", G_CALLBACK(window_delete_event_cb), NULL);
	
	
	// Main content of the window is a vbox
	
	if (!(vbox = gtk_vbox_new(FALSE, 0))) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create main vbox, gtk_vbox_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	
	gtk_container_add(GTK_CONTAINER(sMainWindow), vbox);
	
	
	
	// Create menus
	
	if (!(sActionGroup = gtk_action_group_new("SciteprojActions"))) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkActionGroup, gtk_action_group_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
		
	if (!(sGtkUIManager = gtk_ui_manager_new())) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkUIManager, gtk_ui_manager_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
	

	g_signal_connect(sGtkUIManager, "add_widget", G_CALLBACK(menu_add_widget_cb), vbox);
	
	gtk_action_group_add_actions(sActionGroup, sMenuActions, sNumMenuActions, NULL);
	
	gtk_ui_manager_insert_action_group(sGtkUIManager, sActionGroup, 0);
	
	if (gtk_ui_manager_add_ui_from_string(sGtkUIManager, sMenuDefXML, strlen(sMenuDefXML), &tempErr) == 0) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create create menus from XML, gtk_ui_manager_add_ui_from_string() = %s", __func__, tempErr->message);
		
		goto EXITPOINT;
	}
	
	gtk_ui_manager_ensure_update(sGtkUIManager);
	
	
	// Activate the keyboard accelerators
	
	accelgroup = gtk_ui_manager_get_accel_group(sGtkUIManager);
	gtk_window_add_accel_group(GTK_WINDOW(sMainWindow), accelgroup);
	
	
	// Create popup menus (shown when user right-clicks in gui elements)
	
	sGeneralPopupMenu = gtk_ui_manager_get_widget(sGtkUIManager, "/ui/GeneralPopup");
	sGroupPopupMenu = gtk_ui_manager_get_widget(sGtkUIManager, "/ui/GroupPopup");
	sFilePopupMenu = gtk_ui_manager_get_widget(sGtkUIManager, "/ui/FilePopup");
	
	
	// Add a scrolled window to the main window
	
	if (!(scrolledWindow = gtk_scrolled_window_new(NULL, NULL))) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create main scrolled window, gtk_scrolled_window_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledWindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	
	
	gtk_box_pack_start(GTK_BOX(vbox), scrolledWindow, TRUE, TRUE, 0);

	GtkWidget *hbox;
	
	hbox=gtk_hbox_new(FALSE,0);
	
	if (!init_statusbar(hbox,err)) {
		goto EXITPOINT;
	}
	
	gtk_widget_show(hbox);
	
	gtk_box_pack_end(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
	
	
	// Create the tree datastore
	
	if ((treeStore = get_treestore(err)) == NULL) {
		goto EXITPOINT;
	}
	
	
	// Create the treeview, set it up to render the tree datastore, and add it to the hbox
	
	if (!(sTreeView = gtk_tree_view_new_with_model(GTK_TREE_MODEL(treeStore)))) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkTreeView, gtk_tree_view_new_with_model() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	if (!(textCellRenderer = gtk_cell_renderer_text_new())) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkCellRenderer, gtk_cell_renderer_text_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	if (!(pixbuffCellRenderer = gtk_cell_renderer_pixbuf_new())) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkCellRenderer, gtk_cell_renderer_pixbuf_new() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	if (!(column1 = gtk_tree_view_column_new())) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkTreeViewColumn, gtk_tree_view_column_new() = NULL", __func__);
		
		goto EXITPOINT;
	}

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(sTreeView), FALSE);
	
	gtk_tree_view_column_set_resizable(column1, TRUE);
	gtk_tree_view_column_set_min_width(column1, (int)(gPrefs.width*.75));
	
	
	gtk_tree_view_column_pack_start(column1, pixbuffCellRenderer, FALSE);
	gtk_tree_view_column_add_attribute(column1, pixbuffCellRenderer, "pixbuf", COLUMN_ICON);
	
	
	gtk_tree_view_column_pack_start(column1, textCellRenderer, TRUE);
	gtk_tree_view_column_add_attribute(column1, textCellRenderer, "text", COLUMN_FILENAME);
	gtk_tree_view_column_add_attribute(column1, textCellRenderer, "weight", COLUMN_FONTWEIGHT);
	gtk_tree_view_column_add_attribute(column1, textCellRenderer, "weight-set", COLUMN_FONTWEIGHTSET);
	
	gtk_tree_view_append_column(GTK_TREE_VIEW(sTreeView), column1);
	
	
	if (!(column2 = gtk_tree_view_column_new_with_attributes("", textCellRenderer, "text", COLUMN_FILESIZE, NULL))) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Could not create GtkTreeViewColumn, gtk_tree_view_column_new_with_attributes() = NULL", __func__);
		
		goto EXITPOINT;
	}
	
	// Stoopid gtk always expands the last column
	
	gtk_tree_view_append_column(GTK_TREE_VIEW(sTreeView), column2);
	
	gtk_container_add(GTK_CONTAINER(scrolledWindow), sTreeView);
	
	
	// Get tree events
	
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(sTreeView));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
	
	g_signal_connect(G_OBJECT(sTreeView), "row-activated", G_CALLBACK(tree_row_activated_cb), NULL);
	
	
	gtk_tree_view_enable_model_drag_source(GTK_TREE_VIEW(sTreeView), GDK_BUTTON1_MASK, dragTargets, 1, GDK_ACTION_MOVE);
	gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(sTreeView), dragTargets, 1, GDK_ACTION_MOVE);
	
	
	sDragStruct.treeView = GTK_TREE_VIEW(sTreeView);
	sDragStruct.treeStore = treeStore;
	sDragStruct.isLocalDrag = FALSE;
	sDragStruct.dragNodes = NULL;
	
	g_signal_connect(G_OBJECT(sTreeView), "drag-data-received", G_CALLBACK(drag_data_received_cb), &sDragStruct);
	g_signal_connect(G_OBJECT(sTreeView), "drag-data-get", G_CALLBACK(drag_data_get_cb), &sDragStruct);
	g_signal_connect(G_OBJECT(sTreeView), "drag-motion", G_CALLBACK(drag_motion_cb), &sDragStruct);
	g_signal_connect(G_OBJECT(sTreeView), "row-expanded", G_CALLBACK(row_expand_or_collapse_cb), NULL);
	g_signal_connect(G_OBJECT(sTreeView), "row-collapsed", G_CALLBACK(row_expand_or_collapse_cb), NULL);
	
	g_signal_connect(G_OBJECT(sTreeView), "button-press-event", G_CALLBACK(mouse_button_pressed_cb), sTreeView);
 	g_signal_connect(G_OBJECT(sTreeView), "key-press-event", G_CALLBACK(key_press_cb), sTreeView);
	
	
	
		
	gtk_window_resize(GTK_WINDOW(sMainWindow), gPrefs.width, gPrefs.height);
	gtk_window_move(GTK_WINDOW(sMainWindow),gPrefs.xpos,gPrefs.ypos);
	
	// Show it all....
	
	gtk_widget_show(sTreeView);
	gtk_widget_show(scrolledWindow);
	gtk_widget_show(vbox);
	gtk_widget_show(sMainWindow);
	
	resultCode = TRUE;
	
EXITPOINT:
	
	if (tempErr) g_error_free(tempErr);
	
	return resultCode;
}


/**
 *
 */
void gui_close()
{
	if (window_saved_title) g_free(window_saved_title);
	
	if (scrolledWindow) gtk_widget_destroy(scrolledWindow);
	
	unref_tree_graphics();
	
	unload_graphics();
	
	done_statusbar();

	if (sMainWindow) gtk_widget_destroy(sMainWindow);
}
	

/**
 *
 */
void get_dimensions(gint *left, gint *top, gint *width, gint *height) {
	gtk_window_get_position(GTK_WINDOW(sMainWindow), left, top); 
	gtk_window_get_size(GTK_WINDOW(sMainWindow), width, height);
	return;
}


/**
 * Determine whether a specified row in the tree is expanded.
 *
 * @return TRUE if the row is expanded; FALSE otherwise
 *
 * @param path is the GtkTreePath referencing the row
 */
gboolean tree_row_is_expanded(GtkTreePath *path)
{
	g_assert(path != NULL);
	
	return gtk_tree_view_row_expanded(GTK_TREE_VIEW(sTreeView), path);
}



/**
 * Expand a row in the tree.
 *
 * @param path is the GtkTreePath referencing the row
 * @param expandChildren indicates whether all children should be expanded
 */
void expand_tree_row(GtkTreePath *path, gboolean expandChildren)
{
	if (path!=NULL) {
		gtk_tree_view_expand_row(GTK_TREE_VIEW(sTreeView), path, FALSE);
	}
}


/**
 * Expand a row in the tree.
 *
 * @param path is the GtkTreePath referencing the row
 * @param expandChildren indicates whether all children should be expanded
 */
void collapse_tree_row(GtkTreePath *path)
{
	gtk_tree_view_collapse_row(GTK_TREE_VIEW(sTreeView), path);
}



/**
 * Enable/disable the "Save Project" button.
 *
 * @param enabled indicates whether the button should be enabled or disabled
 */
void set_save_button_sensitivity(gboolean enabled)
{
//~ 	if (sSaveProjectButton) {
//~ 		gtk_widget_set_sensitive(GTK_WIDGET(sSaveProjectButton), enabled);
//~ 	}
}






/**
 * Callback for Gtk "delete_event" message for the top-level application window.
 *
 * @param widget is not used
 * @param event is not used
 * @param data is not used
 */
static gint window_delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gboolean eventHandled = TRUE;
	
	prompt_user_to_save_project();
	
	if (!project_is_dirty()) {
		gtk_main_quit();
	}
	
	return eventHandled;
}


/**
 *
 */
static void switch_folder_icon(GtkTreeView *treeView,GtkTreePath *path)
{
	GtkTreeModel *treeModel = NULL;
	GtkTreeIter iter;
	gint nodeItemType;
	
	gchar *relFilePath = NULL;

	treeModel = gtk_tree_view_get_model(treeView);
	gtk_tree_model_get_iter(treeModel, &iter, path);
	gtk_tree_model_get(treeModel, &iter, COLUMN_ITEMTYPE, &nodeItemType, COLUMN_FILEPATH, &relFilePath, -1);
	
	GdkPixbuf *pixbuf;
	gtk_tree_model_get(treeModel, &iter, COLUMN_ICON, &pixbuf,-1);
	
	gboolean res=gtk_tree_view_row_expanded(treeView,path);
	
	if (res) {
		gtk_tree_view_collapse_row(treeView,path);
	} else {
	
		gtk_tree_view_expand_row(treeView,path,FALSE);
	}
	
	g_free(relFilePath);
	
	
	
}

/**
 * Callback handler for Gtk "row-activated" event.
 *
 * @param treeView is the GtkTreeView
 * @param path is the GtkTreePath of the activated row
 * @param column is not used
 * @param userData is not used
 */
static void tree_row_activated_cb(GtkTreeView *treeView, GtkTreePath *path, GtkTreeViewColumn *column, gpointer userData)
{
	GtkTreeModel *treeModel = NULL;
	GtkTreeIter iter;
	gchar *relFilePath = NULL;
	gchar *absFilePath = NULL;
	gchar *command = NULL;
	GError *err = NULL;
	GtkWidget *dialog = NULL;
	gint nodeItemType;
	gchar *fixed=NULL;
	
	
	// Get the data from the row that was activated
	
	treeModel = gtk_tree_view_get_model(treeView);
	gtk_tree_model_get_iter(treeModel, &iter, path);
	gtk_tree_model_get(treeModel, &iter, COLUMN_ITEMTYPE, &nodeItemType, COLUMN_FILEPATH, &relFilePath, -1);
	
	
	// We can only open files
	
	if (nodeItemType != ITEMTYPE_FILE) {
		switch_folder_icon(treeView,path);
		if (gPrefs.dirty_on_folder_change) {
			set_project_dirty_status(TRUE);
		}
		goto EXITPOINT;
	}
	
	absFilePath=fix_path((gchar*)get_project_directory(),relFilePath);
	
	debug_printf("absFilePath:%s\n",absFilePath);
	debug_printf("relFilePath:%s\n",relFilePath);

	fixed=fix_path((gchar*)get_project_directory(),relFilePath);

	debug_printf("fixed:%s\n",fixed);
	debug_printf("relFilePath:%s\n",relFilePath);
	
	if ((command = g_strdup_printf("open:%s\n", fixed)) == NULL) {
		g_set_error(&err, APP_SCITEPROJ_ERROR, -1, "%s: Error formatting Scite director command, g_strdup_printf() = NULL", __func__);
	}
	else {
		if (send_scite_command(command, &err)) {
			// Try to activate SciTE; ignore errors
			
			activate_scite(NULL);
			
			if (gPrefs.give_scite_focus==TRUE) {
				send_scite_command((gchar*)"focus:0",NULL);
			}
			
			gchar *statusbar_text=NULL;
			
			statusbar_text=g_strdup_printf("Opened %s",remove_newline(get_filename_from_full_path(command)));
			
			set_statusbar_text(statusbar_text);
			
			g_free(statusbar_text);
		}
	}
	
EXITPOINT:
	
	if (err != NULL) {
		dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Could not open selected file: \n\n%s", err->message);
		
		gtk_dialog_run(GTK_DIALOG (dialog));
	}
	
	if (relFilePath) g_free(relFilePath);
	if (absFilePath) g_free(absFilePath);
	if (command) g_free(command);
	if (err) g_error_free(err);
	if (dialog) gtk_widget_destroy(dialog);
	if (fixed) g_free(fixed);
}



/**
 * Respond to a Gtk "button-press-event" message.
 *
 * @param treeView is the GTKTreeView widget in which the mouse-button event occurred
 * @param event is the GdkEventButton event object
 * @param userData is not currently used
 */
static gboolean mouse_button_pressed_cb(GtkWidget *treeView, GdkEventButton *event, gpointer userData)
{
	gboolean eventHandled = FALSE;
	GtkTreePath *path = NULL;
	GtkTreeModel *treeModel = NULL;
	gchar *nodeName = NULL;
	gint nodeItemType;
	GtkTreeIter iter;
	
	
	g_assert(treeView != NULL);
	g_assert(event != NULL);
	
	
	// Until we know for sure, assume that the user has not clicked on a node
	
	clicked_node.valid=FALSE;
	
	
	// If it is not a right-click, then ignore it
	
	if (event->type != GDK_BUTTON_PRESS || event->button != 3) {
		goto EXITPOINT;
	}
	
	
	// Find if the user has clicked on a node
	
	if (!gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(treeView), (gint) event->x, (gint) event->y, &path, NULL, NULL, NULL)) {
		// Nope-- user clicked in the GtkTreeView, but not on a node
		
		gtk_menu_popup(GTK_MENU(sGeneralPopupMenu), NULL, NULL, NULL, NULL, event->button, gdk_event_get_time((GdkEvent*) event));
		
		goto EXITPOINT;
	}
	
	
	// User clicked on a node, so retrieve the particulars
	
	treeModel = gtk_tree_view_get_model(GTK_TREE_VIEW(treeView));
	
	if (!gtk_tree_model_get_iter(treeModel, &iter, path)) {
		goto EXITPOINT;
	}
	
	gtk_tree_model_get(treeModel, &iter, COLUMN_ITEMTYPE, &nodeItemType, COLUMN_FILEPATH, &nodeName, -1);
	
	
	// Save the node info for use by the popup menu callbacks
	
	if (clicked_node.name) g_free(clicked_node.name);
	clicked_node.name=NULL;
	
	clicked_node.valid=TRUE;
	clicked_node.iter=iter;
	clicked_node.type=nodeItemType;
	clicked_node.name=nodeName;
	nodeName = NULL;
	
	// Check if something is selected
	GtkTreeSelection *tree_selection=NULL;
	
	tree_selection=gtk_tree_view_get_selection(GTK_TREE_VIEW(treeView));
	
	if (tree_selection!=NULL) {
		// Check if clicked on something in the selection, otherwise make the clicked one the selection.
			
		if (gtk_tree_selection_path_is_selected(tree_selection,path)==FALSE) {
			// clear selection and make current line selected
				
			gtk_tree_selection_unselect_all(tree_selection);				
				
			gtk_tree_selection_select_path (tree_selection,path);
		}
	}
	
	// Pop up the appropriate menu for the node type
	
	if (nodeItemType == ITEMTYPE_FILE) {
		gtk_menu_popup(GTK_MENU(sFilePopupMenu), NULL, NULL, NULL, NULL, event->button, gdk_event_get_time((GdkEvent*) event));
	}
	else if (nodeItemType == ITEMTYPE_GROUP) {
		gtk_menu_popup(GTK_MENU(sGroupPopupMenu), NULL, NULL, NULL, NULL, event->button, gdk_event_get_time((GdkEvent*) event));
	}
	
	
	// We took care of the event, so no need to propogate it
	
	eventHandled = TRUE;
	
	
EXITPOINT:
	
	if (path) gtk_tree_path_free(path);
	if (nodeName) g_free(nodeName);
	
	return eventHandled;
}


/**
 *
 */
static void sort_ascending_cb()
{
	GError *err = NULL;	
	GtkTreeIter *nodeIter=NULL;
	
	if (clicked_node.valid && clicked_node.type==ITEMTYPE_FILE) {
		goto EXITPOINT;
	}
	
	if (clicked_node.valid) {
		nodeIter=&(clicked_node.iter);
	}
	
	sort_children(&(clicked_node.iter),&err,compare_strings_smaller);
	
	
EXITPOINT:
	//
	if (err) g_error_free(err);
}

/**
 *
 */
static void sort_descending_cb()
{
	GError *err = NULL;	
	GtkTreeIter *nodeIter=NULL;
	
	if (clicked_node.valid && clicked_node.type==ITEMTYPE_FILE) {
		goto EXITPOINT;
	}
	
	if (clicked_node.valid) {
		nodeIter=&(clicked_node.iter);
	}
	
	sort_children(&clicked_node.iter,&err,compare_strings_bigger);
	
	
EXITPOINT:
	//
	if (err) g_error_free(err);
}

/**
 * Add files to a group or the root of the tree
 */
static void popup_add_files_cb()
{
	GtkWidget *dialog = NULL;
	GError *err = NULL;
	GtkTreeIter *nodeIter = NULL;

	debug_printf("%s!\n",__func__);
	
	// Files cannot be added to files!
	
	if (clicked_node.valid && clicked_node.type == ITEMTYPE_FILE) {
		goto EXITPOINT;
	}
	
	
	// Add to the root or to a group?
	
	if (clicked_node.valid) {
		nodeIter = &(clicked_node.iter);
	}
	
	if (!add_files_to_project(nodeIter, &err)) {
		dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while trying add files to the project: %s", err->message);
		
		gtk_dialog_run(GTK_DIALOG(dialog));
	} else {

	}
	
	
EXITPOINT:
	
	if (err) g_error_free(err);
	if (dialog) gtk_widget_destroy(dialog);
}

/**
 *
 */
gboolean open_filename(gchar *filename,GError **err)
{
	gchar *command=NULL;
	
	if (!relative_path_to_abs_path(filename, &filename, get_project_directory(), err)) {
		return FALSE;
	}
	
	// It's a file, so try to open it
	
	if ((command = g_strdup_printf("open:%s\n", filename)) == NULL) {
		g_set_error(err, APP_SCITEPROJ_ERROR, -1, "%s: Error formatting Scite director command, g_strdup_printf() = NULL", __func__);
	}
	else {
		if (send_scite_command(command, err)) {
			// Try to activate SciTE; ignore errors
			
			activate_scite(NULL);
			
			if (gPrefs.give_scite_focus==TRUE) {
				send_scite_command((gchar*)"focus:0",NULL);
			}
		}
	}
	
	return TRUE;
}


/**
 *
 */
static void _selected_helper (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gint nodeItemType;
	gchar *relFilePath;
	
	GError **err=NULL;
		
	gtk_tree_model_get(model, iter, COLUMN_ITEMTYPE, &nodeItemType, COLUMN_FILEPATH, &relFilePath, -1);
	
	if (nodeItemType==ITEMTYPE_FILE) {
		
		open_filename(relFilePath,err);
	}
}


/**
 * Open the selected file.
 *	This is called when a file is rightclicked and open is selected in the menu
 */
static void popup_open_file_cb()
{
	gchar *command = NULL;
	GError *err = NULL;
	GtkWidget *dialog = NULL;
	gchar *absFilePath = NULL;
	
	// several files in selection?
	
	GtkTreeSelection *tree_selection=NULL;
	
	tree_selection=gtk_tree_view_get_selection(GTK_TREE_VIEW(sTreeView));
	
	if (tree_selection!=NULL) {
		
		gtk_tree_selection_selected_foreach (tree_selection, _selected_helper, NULL);
	}
	
	
	// We can only open files
	
	if (!clicked_node.valid || clicked_node.type != ITEMTYPE_FILE) {
		goto EXITPOINT;
	}
	
	if (!open_filename(clicked_node.name,&err)) {
		goto EXITPOINT;
	}
	
	
EXITPOINT:
	
	if (err != NULL) {
		dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Could not open selected file: \n\n%s", err->message);
		
		gtk_dialog_run(GTK_DIALOG (dialog));
	}
	
	if (command) g_free(command);
	if (absFilePath) g_free(absFilePath);
	if (err) g_error_free(err);
	if (dialog) gtk_widget_destroy(dialog);	
}

/**
 *		Actually remove selected nodes
 */
void remove_selected_items ( GtkTreeView *treeview )
{
	GtkTreeSelection *selection = gtk_tree_view_get_selection ( treeview );
	GtkTreeModel *model;
	GtkTreeIter iter;

	GtkTreeStore *store;
	
	GError *error=NULL;
   
	if (gtk_tree_selection_count_selected_rows(selection) == 0)
		return;
   
	GList *list = gtk_tree_selection_get_selected_rows( selection, &model );

	store = get_treestore(NULL);

	model=GTK_TREE_MODEL(store);

	int nRemoved = 0;
	
	// Begin at the end and go to the start
	
	list=g_list_last(list);
	while(list) {

		int ipath=(int)(strtoll(gtk_tree_path_to_string((GtkTreePath*)(list->data)),NULL,10));

		ipath-=nRemoved;
		GString *fixed_path = g_string_new("");
		g_string_printf(fixed_path, "%s", gtk_tree_path_to_string((GtkTreePath*)list->data)/*ipath*/);

		GtkTreePath *path = gtk_tree_path_new_from_string(fixed_path->str);
		g_string_free(fixed_path, TRUE);

		if (path) {
			if ( gtk_tree_model_get_iter ( model, &iter, path) ) { // get iter from specified path
					
				remove_tree_node(&iter,&error);
				nRemoved++;   
			}
			else { // invalid path
				g_error("Error!!!\n");
			}
			gtk_tree_path_free (path);
		}
		else {
			g_error("Error!!!\n");
		}
		list=list->prev;
	}
	
	g_list_foreach (list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free (list);
}


/**
 *
 */
static void do_remove_node(gboolean ignore_clicked_node)
{
	GError *err = NULL;
	GtkWidget *dialog = NULL;
	gint dialogResponse;
	gchar *nodename = NULL;
	
	gint selected_rows=0;
	
	gboolean multiple_selected=FALSE;
	
	// Make sure a node has been selected
	if (!ignore_clicked_node) {
		if (!clicked_node.valid) {
			goto EXITPOINT;
		}
	}
	
	GtkTreeSelection *treeSelect;
	
	treeSelect=gtk_tree_view_get_selection(GTK_TREE_VIEW(sTreeView));
	
	selected_rows=gtk_tree_selection_count_selected_rows(treeSelect);
	if (selected_rows>1) {
		multiple_selected=TRUE;
	}
	
	if (!ignore_clicked_node) {
		// Figure out the node name
		nodename = strrchr(clicked_node.name, '/');
		
		if (nodename != NULL) {
			++nodename;
		}
		else {
			nodename = clicked_node.name;
		}
	} else {
		// if there is only one selected, get its name
		nodename=NULL;
		multiple_selected=TRUE;
	}
		
	// Confirm removal from project
	
	if (multiple_selected==TRUE) {
		
		dialog=gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, "Remove all selected items?" /*, nodename*/);
		
		dialogResponse = gtk_dialog_run(GTK_DIALOG (dialog));
		
		if (dialog_response_is_exit(dialogResponse)) {
			goto EXITPOINT;
		}
		
		// remove them!
		remove_selected_items(GTK_TREE_VIEW(sTreeView));

	} else {
		
		if (clicked_node.type == ITEMTYPE_FILE) {
			if (nodename!=NULL) {
				dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, "Remove file '%s' from project?", nodename);
			} else {
				dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, "Remove file from project?");
			}
		}
		else {
			if (nodename!=NULL) {
				dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, "Remove group '%s' and any contained files from project?", nodename);
			} else {
				dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL, "Remove group and any contained files from project?");
			}
		}
		
		dialogResponse = gtk_dialog_run(GTK_DIALOG (dialog));
		if (dialog_response_is_exit(dialogResponse)) {
			goto EXITPOINT;
		}
		
		
		// Remove the node
		
		if (!remove_tree_node(&(clicked_node.iter), &err)) {
			GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Could not remove the selected node: \n\n%s", err->message);
			
			gtk_dialog_run(GTK_DIALOG (errDialog));
			
			gtk_widget_destroy(errDialog);	
		}
	}
	
EXITPOINT:
	
	if (err) g_error_free(err);
	if (dialog) gtk_widget_destroy(dialog);	
}


/**
 * Remove the selected file/group.
 */
static void popup_remove_node_cb()
{
	do_remove_node(FALSE);
}



/**
 * Add a group to the root of the tree, or to an existing group
 */
static void popup_add_group_cb()
{
	GtkTreeIter *nodeIter = NULL;
	
	
	// Groups cannot be added to files!
	
	if (clicked_node.valid && clicked_node.type == ITEMTYPE_FILE) {
		goto EXITPOINT;
	}
	
	
	// Are we adding this to the root of the tree, or to an existing group?
	
	if (clicked_node.valid) {
		nodeIter = &(clicked_node.iter);
	}
	
	ask_name_add_group(nodeIter);
	
	
EXITPOINT:
	
	return;
}






/**
 * Set the title of the main window.
 *
 * @param newName is the desired new name of the window.
 */
void set_window_title(const gchar *newName)
{
	g_assert(newName != NULL);
	
	gchar *temp_string=g_new(gchar,512);
	g_snprintf(temp_string,512,"%s",newName);
	
	gtk_window_set_title(GTK_WINDOW(sMainWindow), temp_string);
	
	g_free(window_saved_title);
	
	window_saved_title=g_strdup_printf("%s",newName);
	
	
	g_free(temp_string);
}


/**
 *
 */
void update_project_is_dirty(gboolean dirty)
{
	gchar *temp_string=g_new(gchar,512);
	
	if ((int)strlen((char*)(window_saved_title))==0) {
		g_snprintf(window_saved_title,512,"[UNTITLED]");
	}
	
	if (!dirty) {
		g_snprintf(temp_string,512,"%s",window_saved_title);
	} else {
		g_snprintf(temp_string,512,"%s *",window_saved_title);
	}
	
	gtk_window_set_title(GTK_WINDOW(sMainWindow), temp_string);
	
	g_free(temp_string);	
}


 
/**
 * Ask the user for the name of a group and add it to the tree
 *
 * @param nodeIter is the node the group will be added to as a child; if NULL, then the group is added to the root of the tree
 */
static void ask_name_add_group(GtkTreeIter *nodeIter)
{
	GError *err = NULL;
	GtkWidget *dialog = NULL;
	GtkWidget* gtkEntry = NULL;
	GtkWidget* gtkLabel = NULL;
	GtkWidget* table = NULL;
	GtkAttachOptions options = (GtkAttachOptions) (GTK_EXPAND | GTK_SHRINK | GTK_FILL);
	const gchar *groupName = NULL;
	gint dialogResponse;
	
	
	// Create a dialog box with a nicely-centered text entry widget
	
	dialog = gtk_dialog_new_with_buttons("Choose Group Name", NULL, GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	
	g_signal_connect(dialog, "response",  G_CALLBACK(gtk_widget_hide), dialog);
	
	gtk_container_set_border_width(GTK_CONTAINER(dialog), 0);
	
	table = gtk_table_new(1, 2, FALSE);
	
	gtkLabel = gtk_label_new("Enter name of new group:");
	gtk_table_attach(GTK_TABLE(table), gtkLabel, 0, 1, 0, 1, options, options, 5, 5);
	
	gtkEntry = gtk_entry_new();
	g_signal_connect(GTK_OBJECT(gtkEntry), "activate", G_CALLBACK(entry_widget_activated_cb), dialog);
	gtk_table_attach(GTK_TABLE(table), gtkEntry, 1, 2, 0, 1, options, options, 5, 5);
	
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
	
	gtk_widget_show_all(dialog);
	
	
	// Let the user enter a name or cancel the whole thing
	
	dialogResponse=gtk_dialog_run(GTK_DIALOG(dialog));
	if (dialog_response_is_exit(dialogResponse)) {
		goto EXITPOINT;
	}
	
	groupName = gtk_entry_get_text(GTK_ENTRY(gtkEntry));
	
	if (groupName == NULL || *groupName == '\0') {
		GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Invalid group name");
		
		gtk_dialog_run(GTK_DIALOG(errDialog));
		
		gtk_widget_destroy(errDialog);
		
		goto EXITPOINT;
	}
	
	
	// Add the group
	
	if (!add_tree_group(nodeIter, ADD_CHILD, groupName, TRUE, NULL, &err)) {
		GtkWidget *errDialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while adding the group: %s", err->message);
		
		gtk_dialog_run(GTK_DIALOG(errDialog));
		
		gtk_widget_destroy(errDialog);
	}
	
	
EXITPOINT:
	
	// Destroying the dialog should also destroy the table, label, and entry widgets
	if (gtkLabel) gtk_widget_destroy(GTK_WIDGET(gtkLabel));
	if (table) gtk_widget_destroy(GTK_WIDGET(table));

	if (dialog) gtk_widget_destroy(GTK_WIDGET(dialog));
	
	if (err) g_error_free(err);
	
	// Do NOT free groupName, since that is owned by the GtkEntry widget!
}



static void fix_folders_step_through(GtkTreeView *tree_view, GtkTreeIter newiter,GtkTreePath *tree_path)
{
	//g_printf("HEJ!\n");
	GtkTreeModel *tree_model = gtk_tree_view_get_model(tree_view);
	
	gchar *relFilePath;
	
	GError *error;
	gint nodeItemType;
	
	GtkTreeIter iter=newiter;

	do {
		
		gtk_tree_model_get(tree_model, &iter, COLUMN_ITEMTYPE, &nodeItemType, -1);
		

		if (nodeItemType==ITEMTYPE_GROUP) {

			GtkTreePath *srcPath = gtk_tree_model_get_path(tree_model, &iter);
			gboolean groupIsExpanded = tree_row_is_expanded(srcPath);
			
			if (groupIsExpanded) {
				set_tree_node_icon(&iter,directory_open_pixbuf,&error);
			} else {
				set_tree_node_icon(&iter,directory_closed_pixbuf,&error);
			}
			
			gtk_tree_model_get(tree_model, &iter, COLUMN_FILEPATH, &relFilePath, -1);
			
			if (gtk_tree_model_iter_has_child(tree_model,&iter)) {
				
				GtkTreeIter newIter;
				gtk_tree_model_iter_children(tree_model,&newIter,&iter);
				fix_folders_step_through(tree_view,newIter,tree_path);
			}
			
			g_free(relFilePath);
			gtk_tree_path_free(srcPath);
		
		} else {
			
		}
	

	} while(gtk_tree_model_iter_next(tree_model,&iter));
}


/**
 * Callback for expand/collapse event of GtkTreeView
 *
 * @param treeView is not used
 * @param arg1 is not used
 * @param arg2 is not used
 * @param user_data is not used
 */
static void row_expand_or_collapse_cb(GtkTreeView *tree_view, GtkTreeIter *iter, GtkTreePath *tree_path, gpointer user_data)
{
	/* Switch the folder icon open/closed*/
	
	GtkTreeModel *tree_model = NULL;
	tree_model = gtk_tree_view_get_model(tree_view);
	
	
	// make sure all icons the folder (and folders inside it) are set to a correct icon.
	fix_folders_step_through(tree_view,*iter,tree_path);
}



/**
 * Callback for "Quit" menu item
 */
static void quit_menu_cb()
{
	prompt_user_to_save_project();
	
	if (!project_is_dirty()) {
		gtk_main_quit();
	}
}


/**
 * Callback for "About" menu item
 */
static void about_menu_cb()
{
	show_about_dialog();	
}


/**
 * Callback for "Usage" menu item
 */
static void usage_menu_cb()
{
	show_usage_dialog();
}


/**
 * Callback for "Save Project As..." menu item
 */
static void saveproject_as_menu_cb()
{
	GError *err = NULL;
	
	if (!save_project(NULL,&err)) {
		GtkWidget *dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while saving the project: %s", err->message);
		
		if (dialog) {
			gtk_dialog_run(GTK_DIALOG(dialog));
			
			gtk_widget_destroy(dialog);
		}
	}
	
	if (err) g_error_free(err);
}

/**
 * Callback for "Save Project" menu item
 */
static void saveproject_menu_cb()
{
	GError *err = NULL;
	
	gchar *temp_filepath=get_project_filepath();
	
	if (!save_project(temp_filepath,&err)) {
		GtkWidget *dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while saving the project: %s", err->message);
		
		if (dialog) {
			gtk_dialog_run(GTK_DIALOG(dialog));
			
			gtk_widget_destroy(dialog);
		}
	}
	
	if (err) g_error_free(err);
}



/**
 * Callback for "Open Project" menu item
 */
static void openproject_menu_cb()
{
	GError *err = NULL;
	
	if (!load_project(NULL, &err)) {
		GtkWidget *dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "An error occurred while saving the project: %s", err->message);
		
		if (dialog) {
			gtk_dialog_run(GTK_DIALOG(dialog));
			
			gtk_widget_destroy(dialog);
		}
	}
	
	if (err) g_error_free(err);
}



/**
 * Callback for "Create Group" menu item
 */
static void creategroup_menu_cb()
{
	ask_name_add_group(NULL);
}


/**
 * Callback for Remove Item(s) menu item
 */
static void removeitem_menu_cb()
{
	do_remove_node(TRUE);
}



/**
 * Callback for "Add File" menu item
 */
static void addfile_menu_cb()
{
	GError *err = NULL;
	
	// Get the selction, and add to that group
	
	GtkTreeIter *parentIter=NULL;
	
	GtkTreeIter nodeIter;
	
	gboolean nodeValid=FALSE;
	
	GList *list=NULL;
	
	GtkTreeSelection *treeSelection = gtk_tree_view_get_selection(GTK_TREE_VIEW(sTreeView));
	
	GtkTreeModel *tree_model=GTK_TREE_MODEL(get_treestore(&err));
	
	list = gtk_tree_selection_get_selected_rows(treeSelection, NULL);
	
	// is the list empty?
	if (list) {
		list=g_list_first(list);
		
		GtkTreePath *path=(GtkTreePath*)list->data;

		gtk_tree_model_get_iter (tree_model, &nodeIter, path);
	
		nodeValid=TRUE;
			
		g_list_foreach (list, (GFunc)(gtk_tree_path_free), NULL);
		g_list_free (list);

	}
	
	if (nodeValid) {
		
		int nodeType=-1;
		gtk_tree_model_get(tree_model, &nodeIter, COLUMN_ITEMTYPE, &nodeType, -1);
		
		if (nodeType==ITEMTYPE_GROUP) {
			parentIter=&nodeIter;
		} else {
			GtkTreeIter newIter;
			if (gtk_tree_model_iter_parent(tree_model,&newIter,&nodeIter)) {
				parentIter=&newIter;
			}				
			
		}
	}
	
	
	if (parentIter!=NULL) {
		gchar *nodeContents=NULL;
		gtk_tree_model_get(tree_model, parentIter, COLUMN_FILENAME, &nodeContents, -1);
		
		g_free(nodeContents);
	} else {

	}
	
	if (!add_files_to_project(parentIter, &err)) {
		GtkWidget *dialog = gtk_message_dialog_new(NULL, 
					GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
					"An error occurred while trying add files to the project: %s", err->message);
		
		if (dialog) {
			gtk_dialog_run(GTK_DIALOG(dialog));
			
			gtk_widget_destroy(dialog);
		}
	} 
	
	if (err) g_error_free(err);
}



/**
 * Callback for menu manager to populate GUI widgets
 *
 * @param ui is the GtkUIManager
 * @param widget is the GtkWidget to add to the UI
 * @param container is the container to add widget to
 */
static void menu_add_widget_cb(GtkUIManager *ui, GtkWidget *widget, GtkContainer *container)
{
	gtk_box_pack_start(GTK_BOX(container), widget, FALSE, FALSE, 0);
	gtk_widget_show(widget);
}

/**
 *
 */
gboolean key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer userData)
{
	switch (event->keyval) 
	{
		case GDK_BackSpace: 
		{
			debug_printf((gchar*)"key_press_cb: keyval = %d = GDK_BackSpace, hardware_keycode = %d\n", event->keyval, event->hardware_keycode);
			break;
		}
		
		case GDK_Delete: 
		{
			do_remove_node(TRUE);
			break;
		}
		case GDK_Insert: 
		{
			break;
		}
		case GDK_F2:
		{
			rename_key_pressed_cb();
			break;
		}
		default: 
		{
			debug_printf("key_press_cb: keyval = %d = '%c', hardware_keycode = %d\n", event->keyval, (char) event->keyval, event->hardware_keycode);
			return FALSE;
			
			break;
		}
	}
	
	if (event->state & GDK_SHIFT_MASK) debug_printf(", GDK_SHIFT_MASK");
	if (event->state & GDK_CONTROL_MASK) debug_printf(", GDK_CONTROL_MASK");
	if (event->state & GDK_MOD1_MASK) debug_printf(", GDK_MOD1_MASK");
	if (event->state & GDK_MOD2_MASK) debug_printf(", GDK_MOD2_MASK");
	if (event->state & GDK_MOD3_MASK) debug_printf(", GDK_MOD3_MASK");
	if (event->state & GDK_MOD4_MASK) debug_printf(", GDK_MOD4_MASK");
	if (event->state & GDK_MOD5_MASK) debug_printf(", GDK_MOD5_MASK");
	
	debug_printf("\n");
	
	return FALSE;
}


/**
 *
 */
static gboolean foreach_expand(GtkTreeModel *model,GtkTreePath *path,GtkTreeIter *iter,gpointer data)
{
	expand_tree_row(path,TRUE);
	return FALSE;
}

/**
 *
 */
static gboolean foreach_collapse(GtkTreeModel *model,GtkTreePath *path,GtkTreeIter *iter,gpointer data)
{
	collapse_tree_row(path);
	return FALSE;
}

/**
 *
 */
static void expand_all_items_cb()
{
	GtkTreeStore *tree_store=get_treestore(NULL);
	
	gtk_tree_model_foreach(GTK_TREE_MODEL(tree_store),foreach_expand,NULL);
}


/**
 *
 */
static void collapse_all_items_cb()
{
	GtkTreeStore *tree_store=get_treestore(NULL);
	
	gtk_tree_model_foreach(GTK_TREE_MODEL(tree_store),foreach_collapse,NULL);
}


/**
 *		edit_options_cb
 *			opens the user-specific options-file ($HOME/.sciteproj) in SciTE. 
 */
void edit_options_cb()
{
	gchar *absFilePath=NULL;
	GError *err=NULL;
	gchar *command=NULL;
	
	const char *homedir=g_get_home_dir();
	
	debug_printf("Home:%s\n",homedir);
	
	absFilePath=g_strdup_printf("%s/.sciteproj",homedir);
	
	debug_printf("absFilePath:%s\n",absFilePath);
	
	if ((command = g_strdup_printf("open:%s\n", absFilePath)) == NULL) {
		g_set_error(&err, APP_SCITEPROJ_ERROR, -1, "%s: Error formatting Scite director command, g_strdup_printf() = NULL", __func__);
	}
	else {
		if (send_scite_command(command, &err)) {
			// Try to activate SciTE; ignore errors
			
			activate_scite(NULL);
			
			if (gPrefs.give_scite_focus==TRUE) {
				send_scite_command((gchar*)"focus:0",NULL);
			}
		}
	}	
}


/**
 *
 */
GtkWidget *get_main_window()
{
	return sMainWindow;
}
