/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.ruler.graduations;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

/**
 * @author Pierre Lando
 */
public abstract class AbstractGraduations implements Graduations {

    /** The left bracket used by {@link #toString()} */
    private static final String LEFT_BRACKET = "[";

    /** The right bracket used by {@link #toString()} */
    private static final String RIGHT_BRACKET = "]";

    /** True if the lower bound is included in the graduation interval. */
    private final boolean isLowerBoundIncluded;

    /** True if the upper bound is included in the graduation interval. */
    private final boolean isUpperBoundIncluded;

    /** Interval lower bound. */
    private final double lowerBound;

    /** Interval upper bound. */
    private final double upperBound;

    private final Graduations parentGraduations;
    private DecimalFormat numberFormat;

    /**
     * Constructor from parent graduations.
     * This constructor copy information from given {@link Graduations} and set it as is parent.
     * @param parentGraduations the parent graduations to copy.
     */
    public AbstractGraduations(Graduations parentGraduations) {
        this.parentGraduations = parentGraduations;
        this.isLowerBoundIncluded = parentGraduations.isLowerBoundIncluded();
        this.isUpperBoundIncluded = parentGraduations.isUpperBoundIncluded();
        this.lowerBound = parentGraduations.getLowerBound();
        this.upperBound = parentGraduations.getUpperBound();
    }

    /**
     * Root constructor.
     * Graduations made this way don't have a parent.
     * @param lowerBound the actual lower bounds.
     * @param lowerBoundIncluded the actual lower bounds included status.
     * @param upperBound the actual upper bounds.
     * @param upperBoundIncluded the actual upper bounds included status.
     */
    public AbstractGraduations(double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        this.parentGraduations = null;
        this.isLowerBoundIncluded = lowerBoundIncluded;
        this.isUpperBoundIncluded = upperBoundIncluded;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }

    /**
     * Root constructor.
     * Graduations made this way don't have a parent.
     * There bounds are included.
     * @param lowerBound the actual lower bounds included status.
     * @param upperBound the actual upper bounds included status.
     */
    public AbstractGraduations(double lowerBound, double upperBound) {
        this.parentGraduations = null;
        this.isLowerBoundIncluded = true;
        this.isUpperBoundIncluded = true;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }

    /**
     * Child constructor.
     * @param parentGraduations the parent graduation.
     * @param lowerBound the actual lower bounds.
     * @param lowerBoundIncluded the actual lower bounds included status.
     * @param upperBound the actual upper bounds.
     * @param upperBoundIncluded the actual upper bounds included status.
     */
    public AbstractGraduations(Graduations parentGraduations, double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        this.parentGraduations = parentGraduations;
        this.isLowerBoundIncluded = lowerBoundIncluded;
        this.isUpperBoundIncluded = upperBoundIncluded;
        this.lowerBound = lowerBound;
        this.upperBound = upperBound;
    }



    @Override
    public final double getLowerBound() {
        return lowerBound;
    }

    @Override
    public final boolean isLowerBoundIncluded() {
        return isLowerBoundIncluded;
    }

    @Override
    public final double getUpperBound() {
        return upperBound;
    }

    @Override
    public final boolean isUpperBoundIncluded() {
        return isUpperBoundIncluded;
    }

    @Override
    public final Graduations getParentGraduations() {
        return parentGraduations;
    }

    @Override
    public final boolean contain(double value) {
        if (value == lowerBound) {
            return isLowerBoundIncluded;
        }
        if (value == upperBound) {
            return isUpperBoundIncluded;
        }
        return (lowerBound < value) && (value < upperBound);
    }

    @Override
    public final DecimalFormat getFormat() {
        if (numberFormat == null) {
            int maxDisplayedExponent = (int) Math.log10(Math.max(Math.abs(lowerBound), Math.abs(upperBound)));
            if ((maxDisplayedExponent <= -5) || (maxDisplayedExponent >= 6)) {
                numberFormat = new DecimalFormat("0.#E00");
            } else {
                numberFormat = new DecimalFormat();
            }

            DecimalFormatSymbols decimalFormatSymbols = numberFormat.getDecimalFormatSymbols();
            decimalFormatSymbols.setExponentSeparator("e");
            numberFormat.setDecimalFormatSymbols(decimalFormatSymbols);
        }
        return numberFormat;
    }

    @Override
    public final String toString() {
        String lowerBoundBracket;
        String upperBoundBracket;

        if (isLowerBoundIncluded) {
            lowerBoundBracket = LEFT_BRACKET;
        } else {
            lowerBoundBracket = RIGHT_BRACKET;
        }

        if (isUpperBoundIncluded) {
            upperBoundBracket = RIGHT_BRACKET;
        } else {
            upperBoundBracket = LEFT_BRACKET;
        }
        return "Graduation " + lowerBoundBracket
                + getFormat().format(lowerBound) + ", "
                + getFormat().format(upperBound) + upperBoundBracket;
    }
}
