/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl;

import org.scilab.forge.scirenderer.Canvas;
import org.scilab.forge.scirenderer.Drawer;
import org.scilab.forge.scirenderer.depthtest.DepthTestManager;
import org.scilab.forge.scirenderer.implementation.jogl.buffers.JoGLBuffersManager;
import org.scilab.forge.scirenderer.implementation.jogl.depthtest.JoGLDepthBufferManager;
import org.scilab.forge.scirenderer.implementation.jogl.picking.JoGLPickingManager;
import org.scilab.forge.scirenderer.implementation.jogl.renderer.JoGLRendererManager;
import org.scilab.forge.scirenderer.implementation.jogl.texture.JoGLTextureManager;
import org.scilab.forge.scirenderer.picking.PickingManager;

import com.jogamp.opengl.util.awt.Screenshot;
import java.awt.Dimension;
import java.awt.image.BufferedImage;
import javax.media.opengl.DebugGL2;
import javax.media.opengl.GL2;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLCapabilities;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLDrawableFactory;
import javax.media.opengl.GLEventListener;
import javax.media.opengl.GLPbuffer;
import javax.media.opengl.GLProfile;

/**
 * JoGL implementation of a Canvas.
 *
 * @author Pierre Lando
 */
public final class JoGLCanvas implements Canvas, GLEventListener {

    private final GLAutoDrawable autoDrawable;

    private final JoGLDrawingTools drawingTools;
    private final JoGLParameters parameters;
    private final JoGLBuffersManager buffersManager;
    private final JoGLRendererManager rendererManager;
    private final JoGLDepthBufferManager depthBufferManager;
    private final JoGLPickingManager pickingManager;
    private final JoGLTextureManager textureManager;
    private boolean isOffscreen;
    private DebugGL2 debug;


    /**
     * The current mainDrawer.
     */
    private Drawer mainDrawer;

    /**
     * Default constructor.
     * @param autoDrawable the JoGL autoDrawable this canvas depend on.
     */
    JoGLCanvas(GLAutoDrawable autoDrawable) {
        this.autoDrawable = autoDrawable;
        parameters = new JoGLParameters();
        buffersManager = new JoGLBuffersManager();
        rendererManager = new JoGLRendererManager();
        depthBufferManager = new JoGLDepthBufferManager();
        drawingTools = new JoGLDrawingTools(this);
        pickingManager = new JoGLPickingManager(this);
        textureManager = new JoGLTextureManager(this);

        autoDrawable.addGLEventListener(this);
    }

    /**
     * Constructor for offscreen rendering
     * @param width the width
     * @param height the height
     */
    JoGLCanvas(int width, int height) {
        this(getOffscreenDrawable(width, height));
        isOffscreen = true;
    }

    public void setDebugMode(boolean debug) {
        if (debug) {
            this.debug = new DebugGL2(autoDrawable.getGL().getGL2());
        } else {
            this.debug = null;
        }
    }

    // Implementation of getter & setter from Canvas.

    @Override
    public void setMainDrawer(Drawer mainDrawer) {
        this.mainDrawer = mainDrawer;
    }

    @Override
    public Drawer getMainDrawer() {
        return mainDrawer;
    }

    @Override
    public JoGLRendererManager getRendererManager() {
        return rendererManager;
    }

    @Override
    public JoGLBuffersManager getBuffersManager() {
        return buffersManager;
    }

    @Override
    public DepthTestManager getDepthTestManager() {
        return depthBufferManager;
    }

    @Override
    public PickingManager getPickingManager() {
        return pickingManager;
    }

    @Override
    public JoGLTextureManager getTextureManager() {
        return textureManager;
    }

    @Override
    public int getWidth() {
        return autoDrawable.getWidth();
    }

    @Override
    public int getHeight() {
        return autoDrawable.getHeight();
    }

    @Override
    public Dimension getDimension() {
        return new Dimension(autoDrawable.getWidth(), autoDrawable.getHeight());
    }

    @Override
    public void redraw() {
        autoDrawable.display();
    }

    // JoGLCanvas specific getter.

    /**
     * Return the OpenGl context.
     * @return the OpenGl context.
     */
    public GL2 getGl() {
        if (debug == null) {
            return autoDrawable.getGL().getGL2();
        } else {
            return debug;
        }
    }

    /**
     * Return the rendering parameters.
     * @return the rendering parameters.
     */
    public JoGLParameters getJoGLParameters() {
        return parameters;
    }

    /**
     * Get an image from the autoDrawable
     * @return an image
     */
    public BufferedImage getImage() {
        GLContext context = autoDrawable.getContext();
        context.makeCurrent();
        BufferedImage image = Screenshot.readToBufferedImage(autoDrawable.getWidth(), autoDrawable.getHeight());
        context.release();

        return image;
    }

    /**
     * Destroy the GLPbuffer
     */
    public void destroy() {
        if (isOffscreen) {
            ((GLPbuffer) autoDrawable).destroy();
        }
    }

    /**
     * Creates a GLPbuffer for an offscreen rendering
     * @param width the width
     * @param height the height
     * @return a GLPbuffer
     */
    private static GLAutoDrawable getOffscreenDrawable(int width, int height) {
        GLDrawableFactory factory = GLDrawableFactory.getDesktopFactory();
        GLCapabilities capabilities = new GLCapabilities(GLProfile.getDefault());

        return factory.createGLPbuffer(null, capabilities, null, width, height, null);
    }

    // Implementation of function from GLEventListener.

    @Override
    public void display(GLAutoDrawable glAutoDrawable) {
        GL2 gl = getGl().getGL2();
        drawingTools.glSynchronize(gl);
        buffersManager.glSynchronize(gl);
        rendererManager.glSynchronize(gl);

        if (mainDrawer != null) {
            gl.glEnable(GL2.GL_DEPTH_TEST); // TODO add possibility to (de-)activate depth test.
            gl.glDepthFunc(GL2.GL_LEQUAL); // Set to less equal to allow last drawn object to be on the top.
            if (depthBufferManager.isAutoClearDepthBufferEnable()) {
                gl.glClear(GL2.GL_DEPTH_BUFFER_BIT);
            }

            mainDrawer.draw(drawingTools);
        }
        pickingManager.glConsume(drawingTools);
    }

    @Override
    public void init(GLAutoDrawable glAutoDrawable) {
        textureManager.glReload();
        buffersManager.glReload();
        rendererManager.glReload();
    }

    @Override
    public void reshape(GLAutoDrawable glAutoDrawable, int x, int y, int width, int height) {
    }

    //@Override
    public void displayChanged(GLAutoDrawable glAutoDrawable, boolean b, boolean b1) {
    }

    @Override
    public void dispose(GLAutoDrawable drawable) { }
}
