/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.ruler.graduations;

import java.util.LinkedList;
import java.util.List;

/**
 * @author Pierre Lando
 */
public class LinearGraduations extends AbstractGraduations implements Graduations {

    protected final int stepExponent;
    protected final int stepMantissa;


    private LinearGraduations moreGraduation;
    private LinearGraduations alternativeGraduation;
    private Graduations subGraduation;
    private Double stepValue;
    private List<Double> allValues;
    private List<Double> newValues;

    public static LinearGraduations create(double lowerBound, double upperBound) {
        return create(lowerBound, true, upperBound, true);
    }

    public static LinearGraduations create(Graduations parentGraduations, double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        if (lowerBound < upperBound) {
            return new LinearGraduations(parentGraduations, lowerBound, lowerBoundIncluded, upperBound, upperBoundIncluded);
        } else {
            return new LinearGraduations(parentGraduations, upperBound, upperBoundIncluded, lowerBound, lowerBoundIncluded);
        }
    }

    public static LinearGraduations create(double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        if (lowerBound < upperBound) {
            return new LinearGraduations(null, lowerBound, lowerBoundIncluded, upperBound, upperBoundIncluded);
        } else {
            return new LinearGraduations(null, upperBound, upperBoundIncluded, lowerBound, lowerBoundIncluded);
        }
    }

    private LinearGraduations() {
        super(null);
        stepExponent = 0;
        stepMantissa = 0;
    }

    private LinearGraduations(Graduations parentGraduations, double lowerBound, boolean lowerBoundIncluded, double upperBound, boolean upperBoundIncluded) {
        super(parentGraduations, lowerBound, lowerBoundIncluded, upperBound, upperBoundIncluded);
        if (lowerBound != upperBound) {
            double size = upperBound - lowerBound;
            stepExponent = (int) Math.ceil(Math.log10(size));
            stepMantissa = 1;
        } else {
            stepExponent = 0;
            stepMantissa = 0;
            newValues = allValues = new LinkedList<Double>();
            allValues.add(lowerBound);
        }
    }

    private LinearGraduations(Graduations parentGraduations, int stepExponent, int stepMantissa) {
        super(parentGraduations);
        this.stepExponent = stepExponent;
        this.stepMantissa = stepMantissa;
    }

    private Double getStepValue() {
        if (stepValue == null) {
            stepValue = stepMantissa * Math.pow(10, stepExponent);
        }
        return stepValue;
    }

    private boolean isNewIndex(long index) {
        /* We are now searching for value look like
         * index * (stepMantissa * 10^n) and we don't want (previousStrepMantissa * 10^k) value.
         */

        if (stepMantissa == 1) {
            return ((5 * index * stepMantissa) % 10 != 0);
        } else if (stepMantissa == 2) {
            return ((2 * index * stepMantissa) % 10 != 0);
        } else {
            return ((5 * index * stepMantissa) % 10 != 0);
        }
    }

    @Override
    public List<Double> getNewValues() {
        if (getParentGraduations() == null) {
            return getAllValues();
        }

        if (newValues == null) {
            newValues = new LinkedList<Double>();
            long currentIndex = (long) Math.ceil(lowerBound / getStepValue());
            double currentValue = getStepValue() * currentIndex;

            if ((currentValue == lowerBound) && (!contain(currentValue))) {
                currentValue += getStepValue();
                currentIndex ++;
            }

            while (contain(currentValue)) {
                if (isNewIndex(currentIndex)) {
                    newValues.add(currentValue);
                }
                currentValue += getStepValue();
                currentIndex ++;
            }
        }

        return newValues;
    }

    @Override
    public List<Double> getAllValues() {
        if (allValues == null) {
            allValues = new LinkedList<Double>();
            double currentValue = getStepValue() * Math.ceil(lowerBound / getStepValue());

            if ((currentValue == lowerBound) && (!contain(currentValue))) {
                currentValue += getStepValue();
            }

            while (contain(currentValue)) {
                allValues.add(currentValue);
                currentValue += getStepValue();
            }
        }
        return allValues;
    }

    @Override
    public LinearGraduations getMore() {
        if (stepMantissa != 5) {
            if (moreGraduation == null) {
                if (stepMantissa == 1) {
                    moreGraduation = new LinearGraduations(this, stepExponent - 1, 2);
                } else {
                    moreGraduation = new LinearGraduations(this, stepExponent, 1);
                }
            }
            return moreGraduation;
        } else {
            return null;
        }
    }

    @Override
    public LinearGraduations getAlternative() {
        if (stepMantissa == 2) {
            if (alternativeGraduation == null) {
                if (getParentGraduations() == null) {
                    alternativeGraduation =  new LinearGraduations(null, lowerBound, true, lowerBound, true);
                } else {
                    alternativeGraduation = new LinearGraduations(getParentGraduations(), stepExponent, 5);
                }
            }
            return alternativeGraduation;
        } else {
            return null;
        }
    }

    @Override
    public Graduations getSubGraduations() {
        if (subGraduation == null) {
            switch (stepMantissa) {
                case 1:
                    subGraduation = new LinearGraduations(this, stepExponent - 1, 5);
                    break;
                case 2:
                    subGraduation = new LinearGraduations(this, stepExponent, 1);
                    break;
                case 5:
                    subGraduation = new LinearGraduations(getParentGraduations(), stepExponent, 1);
                    break;
            }
        }
        return subGraduation;
    }

    @Override
    public int getSubDensity() {
        if (stepMantissa == 5) {
            return 5;
        } else {
            return 2;
        }
    }
}
