/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2012 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.ruler;

import org.scilab.forge.scirenderer.DrawingTools;
import org.scilab.forge.scirenderer.SciRendererException;
import org.scilab.forge.scirenderer.buffers.BuffersManager;
import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.ruler.graduations.Graduations;
import org.scilab.forge.scirenderer.shapes.appearance.Appearance;
import org.scilab.forge.scirenderer.shapes.geometry.Geometry;
import org.scilab.forge.scirenderer.shapes.geometry.DefaultGeometry;
import org.scilab.forge.scirenderer.sprite.Sprite;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.sprite.SpriteManager;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

import java.awt.geom.Rectangle2D;
import java.nio.FloatBuffer;
import java.text.DecimalFormat;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Pierre Lando
 */
public class RulerDrawer {

    /**
     * Sprite map.
     */
    private final Map<Double, Sprite> spriteMap = new ConcurrentHashMap<Double, Sprite>();

    /**
     * The current {@link SpriteManager}.
     */
    private final SpriteManager spriteManager;

    /**
     * The used {@link RulerSpriteFactory}.
     */
    private RulerSpriteFactory spriteFactory;

    /**
     * Ruler drawer constructor.
     * @param spriteManager the {@link SpriteManager} of the canvas where the ruler will be drawn.
     */
    public RulerDrawer(SpriteManager spriteManager) {
        this.spriteManager = spriteManager;
        this.spriteFactory = new DefaultRulerSpriteFactory();
    }

    /**
     * Ruler drawing method.
     * @param drawingTools the {@link DrawingTools} of the canvas where the ruler will be drawn.
     * @param model the {@link RulerModel} of the drawn ruler.
     * @return the {@link RulerDrawingResult} give information about how the ruler have been drawn.
     */
    public RulerDrawingResult draw(DrawingTools drawingTools, RulerModel model) {
        return new OneShotRulerDrawer(drawingTools, model).getDrawingResult();
    }

    /**
     * Set the current {@link RulerSpriteFactory}.
     * All existing sprite will be cleared.
     * This ruler drawer will use the new {@link RulerSpriteFactory}.
     * @param spriteFactory the new {@link RulerSpriteFactory}.
     */
    public void setSpriteFactory(RulerSpriteFactory spriteFactory) {
        disposeResources();
        this.spriteFactory = spriteFactory;
    }

    /**
     * Dispose all used resources.
     */
    public void disposeResources() {
        spriteManager.dispose(spriteMap.values());
        spriteMap.clear();
    }


    /**
     * This class actually perform all the rendering of one ruler.
     */
    private class OneShotRulerDrawer {

        private final RulerDrawingResult rulerDrawingResult;
        private final Transformation canvasProjection;
        private final RulerModel rulerModel;
        private final Vector3d windowSubTicksDelta;
        private final Vector3d windowTicksDelta;

        private List<PositionedSprite> spritesList = new LinkedList<PositionedSprite>();

        /**
         * The maximum distance corresponding to the actually displayed sprites.
         */
        private double maximalSpritesDistance = 0.0;

        /**
         * Deepest possible {@see Graduations}
         */
        private Graduations graduations;

        private List<Double> subTicksValue;
        private List<Double> ticksValue;
        private int density;

        /**
         * Constructor.
         * @param drawingTools the {@link DrawingTools} of the canvas where the ruler will be drawn.
         * @param rulerModel the {@link RulerModel} of the drawn ruler.
         */
        public OneShotRulerDrawer (DrawingTools drawingTools, RulerModel rulerModel) {
            this.rulerModel =rulerModel;
            canvasProjection = drawingTools.getTransformationManager().getCanvasProjection();

            Vector3d windowTicksDirection = canvasProjection.projectDirection(rulerModel.getTicksDirection());
            windowTicksDirection = windowTicksDirection.setZ(0);

            Vector3d normalizedProjectedTicksDirection = windowTicksDirection.getNormalized();
            windowSubTicksDelta = normalizedProjectedTicksDirection.times(rulerModel.getSubTicksLength());
            windowTicksDelta = normalizedProjectedTicksDirection.times(rulerModel.getSpriteDistance());

            if (rulerModel.isAutoTicks()) {
                computeAutoGraduation();
            } else {
                computeUserGraduation();
            }
            computeTicksData();
            draw(drawingTools);
            double distRatio = computeTicksDistanceRatio(windowTicksDirection.getNorm());

            rulerDrawingResult = new RulerDrawingResult(ticksValue, subTicksValue, density, distRatio, normalizedProjectedTicksDirection);
        }

        /**
         * Return the drawing result.
         * @return the drawing result.
         */
        public RulerDrawingResult getDrawingResult() {
            return rulerDrawingResult;
        }

        /**
         * Compute the ratio between windows ticks norm and the sprite distance.
         * @param windowTicksNorm the windows tics norm.
         * @return the ratio between windows ticks norm and the sprite distance.
         */
        private double computeTicksDistanceRatio(double windowTicksNorm) {
            double distRatio;
            if (windowTicksNorm == 0) {
                distRatio = 1.0;
            } else if (maximalSpritesDistance == 0) {
                distRatio = rulerModel.getSpriteDistance() / windowTicksNorm;
            } else {
                distRatio = maximalSpritesDistance / windowTicksNorm;
            }
            return distRatio;
        }

        /**
         * Actually perform the ruler drawing.
         * @param drawingTools {@link DrawingTools} used to perform the ruler drawing.
         */
        private void draw(DrawingTools drawingTools) {
            BuffersManager bufferManager = drawingTools.getCanvas().getBuffersManager();
            ElementsBuffer vertices = bufferManager.createElementsBuffer();
            fillVertices(vertices, rulerModel, ticksValue, subTicksValue, canvasProjection);
            DefaultGeometry geometry = new DefaultGeometry();
            geometry.setFillDrawingMode(Geometry.FillDrawingMode.NONE);
            geometry.setLineDrawingMode(Geometry.LineDrawingMode.SEGMENTS);
            geometry.setVertices(vertices);

            Appearance appearance = new Appearance();
            appearance.setLineColor(rulerModel.getColor());
            appearance.setLineWidth(1);

            drawingTools.getTransformationManager().useWindowCoordinate();
            for (PositionedSprite positionedSprite : spritesList) {
                drawingTools.draw(positionedSprite.getSprite(), SpriteAnchorPosition.CENTER, positionedSprite.getWindowPosition());
            }
            try {
                drawingTools.draw(geometry, appearance);
            } catch (SciRendererException ignored) {
            }
            drawingTools.getTransformationManager().useSceneCoordinate();
            bufferManager.dispose(vertices);
        }

        /**
         * Compute the {@link Graduations} used to the ruler drawing in auto-ticks mode..
         */
        private void computeAutoGraduation() {
            /* The maximum distance corresponding to the actually displayed sprites. */
            double maximalSpritesDistance = 0.0;

            Graduations currentGraduations = rulerModel.getGraduations();
            Graduations ticksGraduation = currentGraduations;

            boolean canGetMore = true;
            while (currentGraduations != null) {
                /* The maximum distance to any of the sprites' farthest sides at a given iteration. */
                double currentMaximalSpritesDistance = 0;

                List<PositionedSprite> newSpritesList = new LinkedList<PositionedSprite>();
                List<Double> ticks = currentGraduations.getNewValues();
                for (double value : ticks) {
                    Sprite sprite = computeSprite(value, currentGraduations.getFormat());
                    Vector3d windowPosition = canvasProjection.project(rulerModel.getPosition(value));

                    Vector3d delta = projectCenterToEdge(sprite, windowTicksDelta);
                    newSpritesList.add(new PositionedSprite(sprite, windowPosition.plus(windowTicksDelta.plus(delta))));

                    Vector3d farDelta = windowTicksDelta.plus(delta.times(2.0));
                    currentMaximalSpritesDistance = Math.max(currentMaximalSpritesDistance, farDelta.getNorm());
                }

                if (collide(newSpritesList, rulerModel.getMargin()) || collide(spritesList, newSpritesList, rulerModel.getMargin())) {
                    currentGraduations = currentGraduations.getAlternative();
                    canGetMore = false;
                } else {
                    maximalSpritesDistance = Math.max(maximalSpritesDistance, currentMaximalSpritesDistance);
                    spritesList.addAll(newSpritesList);
                    ticksGraduation = currentGraduations;
                    if (canGetMore) {
                        currentGraduations = currentGraduations.getMore();
                    } else {
                        currentGraduations = null;
                    }
                }
            }

            this.graduations = ticksGraduation;
            this.maximalSpritesDistance = maximalSpritesDistance;
        }

        /**
         * Compute the {@link Graduations} used to the ruler drawing in auto-ticks mode..
         */
        private void computeUserGraduation() {
            /* The maximum distance corresponding to the actually displayed sprites. */
            double maximalSpritesDistance = 0.0;
            Graduations currentGraduations = rulerModel.getGraduations();

            List<Double> ticks = currentGraduations.getNewValues();
            for (double value : ticks) {
                Sprite sprite = computeSprite(value, currentGraduations.getFormat());
                if (sprite != null) {
                    Vector3d windowPosition = canvasProjection.project(rulerModel.getPosition(value));

                    Vector3d delta = projectCenterToEdge(sprite, windowTicksDelta);
                    spritesList.add(new PositionedSprite(sprite, windowPosition.plus(windowTicksDelta.plus(delta))));

                    Vector3d farDelta = windowTicksDelta.plus(delta.times(2.0));
                    maximalSpritesDistance = Math.max(maximalSpritesDistance, farDelta.getNorm());
                }
            }

            this.graduations = currentGraduations;
            this.maximalSpritesDistance = maximalSpritesDistance;
        }

        /**
         * Compute the ticks, sub-ticks data and the sub-ticks density.
         */
        private void computeTicksData() {
            if (graduations != null) {
                density = graduations.getSubDensity();
                ticksValue = graduations.getAllValues();

                Graduations subGraduation = graduations.getSubGraduations();

                while ((subGraduation != null) && (computeTicksDistance(subGraduation) < rulerModel.getMinimalSubTicksDistance())) {
                    subGraduation = subGraduation.getAlternative();
                }

                if (subGraduation != null) {
                    subTicksValue = subGraduation.getAllValues();
                } else {
                    subTicksValue = new LinkedList<Double>();
                }
            } else {
                subTicksValue = new LinkedList<Double>();
                ticksValue = new LinkedList<Double>();
                density = 0;
            }
        }

        /**
         * Compute and return the minimal screen distance between two successive ticks of the given {@link Graduations}.
         * If the given {@link Graduations} is <code>null</code>, the returned value is {@link Double#MAX_VALUE}.
         * @param graduations the given {@link Graduations}.
         * @return the minimal screen distance between two successive ticks of the given {@link Graduations}.
         */
        private double computeTicksDistance(Graduations graduations) {
            double minimalDistance = Double.MAX_VALUE;
            if (graduations != null) {
                Vector3d previousProjection = null;
                for (double currentValue : graduations.getAllValues()) {
                    Vector3d currentProjection = canvasProjection.project(rulerModel.getPosition(currentValue));
                    if (previousProjection != null) {
                        minimalDistance = Math.min(minimalDistance, currentProjection.minus(previousProjection).getNorm2());
                    }
                    previousProjection = currentProjection;
                }
                minimalDistance = Math.sqrt(minimalDistance);
            }
            return minimalDistance;
        }

        /**
         * Return true if at leas two element of {@see spritesList} collide.
         * @param spritesList the list of sprite to be tested.
         * @param margin the collision margin.
         * @return true if at leas two element of {@see spritesList} collide.
         */
        private boolean collide(List<PositionedSprite> spritesList, double margin) {
            for (int i = 0; i < spritesList.size(); i++) {
                for (int j = i + 1; j <spritesList.size(); j++) {
                    if (collide(spritesList.get(i).getWindowBounds(), spritesList.get(j).getWindowBounds(), margin)) {
                        return true;
                    }
                }
            }
            return false;
        }

        /**
         * Return true if the at least one element of <code>newSpritesList</code> collides with one element of <code>spritesList</code>.
         * @param spritesList the list of reference sprites.
         * @param newSpritesList the list of new sprites.
         * @param margin the collision margin.
         * @return true if the at least one element of <code>newSpritesList</code> collides with one element of <code>spritesList</code>.
         */
        private boolean collide(List<PositionedSprite> spritesList, List<PositionedSprite> newSpritesList, double margin) {
            for (PositionedSprite sprite1 : newSpritesList) {
                for (PositionedSprite sprite2 : spritesList) {
                    if (collide(sprite1.getWindowBounds(), sprite2.getWindowBounds(), margin)) {
                        return true;
                    }
                }
            }
            return false;
        }

        /**
         * Return true if the givens rectangles collide.
         * @param rectangle1 first rectangle.
         * @param rectangle2 second rectangle.
         * @param margin the margin.
         * @return true if the givens rectangles collide.
         */
        private boolean collide(Rectangle2D rectangle1, Rectangle2D rectangle2, double margin) {
            return  ((rectangle2.getMinX() - rectangle1.getMaxX()) < margin)
                &&  ((rectangle1.getMinX() - rectangle2.getMaxX()) < margin)
                &&  ((rectangle2.getMinY() - rectangle1.getMaxY()) < margin)
                &&  ((rectangle1.getMinY() - rectangle2.getMaxY()) < margin);
        }

        /**
         * Compute and return a translation along the given <code>direction</code>.
         * This translation is the vector between the <code>sprite</code> center and is projection along the given
         * {@see direction} to the sprite edges.
         * @param sprite the given {@link Sprite}
         * @param direction the given direction.
         * @return the vector between the sprite center and is projection to the sprite edges.
         */
        private Vector3d projectCenterToEdge(Sprite sprite, Vector3d direction) {
            if ((direction == null) || (direction.isZero())) {
                direction = new Vector3d(1, 0, 0);
            }
            double ratioX = sprite.getWidth() / Math.abs(direction.getX());
            double ratioY = sprite.getHeight() / Math.abs(direction.getY());
            double ratio = Math.min(ratioY, ratioX) / 2;
            return direction.times(ratio);
        }


        /**
         * Fill a vertices buffer with the needed data to draw a ruler.
         * @param verticesBuffer the {@link ElementsBuffer} to fill.
         * @param rulerModel the {@link RulerModel} to draw.
         * @param ticksValue the list of ticks.
         * @param subTicksValue the list of sub-ticks.
         * @param canvasProjection the used canvas projection.
         */
        private void fillVertices(ElementsBuffer verticesBuffer, RulerModel rulerModel, List<Double> ticksValue, List<Double> subTicksValue, Transformation canvasProjection) {
            Vector3d a = rulerModel.getFirstPoint();
            Vector3d b = rulerModel.getSecondPoint();

            if ((a != null) && (b != null)) {
                int bufferSize = 2 * ticksValue.size() + 2 * subTicksValue.size();
                if (rulerModel.isLineVisible()) {
                    bufferSize += 2;
                }
                FloatBuffer data = FloatBuffer.allocate(4 * bufferSize);
                data.rewind();

                for (double value : ticksValue) {
                    Vector3d p = canvasProjection.project(rulerModel.getPosition(value));
                    data.put(p.getDataAsFloatArray(4));
                    data.put(p.plus(windowTicksDelta).getDataAsFloatArray(4));
                }

                for (double value : subTicksValue) {
                    Vector3d p = canvasProjection.project(rulerModel.getPosition(value));
                    data.put(p.getDataAsFloatArray(4));
                    data.put(p.plus(windowSubTicksDelta).getDataAsFloatArray(4));
                }

                if (rulerModel.isLineVisible()) {
                    data.put(canvasProjection.project(a).getDataAsFloatArray(4));
                    data.put(canvasProjection.project(b).getDataAsFloatArray(4));
                }

                data.rewind();
                verticesBuffer.setData(data, 4);
            } else {
                verticesBuffer.setData(new float[0], 4);
            }
        }

        /**
         * Compute the {@link Sprite} for a given value.
         * The {@link Sprite} is made once using the current {@link RulerSpriteFactory}.
         * @param value the given value.
         * @param format the format to use.
         * @return A {@link Sprite} for the label at the given value.
         */
        private Sprite computeSprite(double value, DecimalFormat format) {
            Sprite sprite = spriteMap.get(value);
            if (sprite == null) {
                sprite = spriteFactory.create(value, format, spriteManager);
                spriteMap.put(value, sprite);
            }
            return sprite;
        }

        /**
         * This class is a basic container for a {@link Sprite} and an associated window position.
         *
         * @author Pierre Lando
         */
        private class PositionedSprite {

            private final Sprite sprite;
            private final Vector3d windowPosition;
            private final Rectangle2D windowsBounds;

            /**
             * Default constructor.
             * @param sprite the {@link Sprite}.
             * @param windowPosition the window position.
             */
            public PositionedSprite(Sprite sprite,  Vector3d windowPosition) {
                this.windowPosition = windowPosition;
                this.sprite = sprite;

                windowsBounds = new Rectangle2D.Double(
                        windowPosition.getX(),
                        windowPosition.getY(),
                        sprite.getWidth(),
                        sprite.getHeight()
                );
            }

            /**
             * Return the {@link Sprite}
             * @return the {@link Sprite}
             */
            public Sprite getSprite() {
                return sprite;
            }

            /**
             * Return the window position of the {@link Sprite}.
             * @return the window position of the {@link Sprite}.
             */
            public Vector3d getWindowPosition() {
                return windowPosition;
            }

            /**
             * Return the window bounds of the {@link Sprite}
             * @return the window bounds of the {@link Sprite}
             */
            public Rectangle2D getWindowBounds() {
                return windowsBounds;
            }
        }
    }
}