/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 * Copyright (C) 2011 - DIGITEO - Manuel Juliachs
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import java.nio.FloatBuffer;

import javax.media.opengl.GL;

import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.TransformationManager;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

/**
 * @author Pierre Lando
 * @author Manuel Juliachs
 *
 * Performs the rendering of a rotatable 2D sprite using textured quads.
 * To do:
 * -implement correct screen-space aligning relative to the anchor point,
 *  as its apparent position may vary when changing the dimensions of the texture and/or
 *  rendered image, or its coordinates (possibly caused by subpixel accuracy and other
 *  related rasterization issues).
 * -refactoring (e.g. draw's initialization calls).
 */
public class RotatableQuadsTextureRenderer extends AbstractRotatableTextureRenderer implements RotatableRenderer {

    /**
     * Default constructor.
     *
     * @param sprite the sprite this renderer draws.
     */
    RotatableQuadsTextureRenderer(JoGLSprite sprite) {
        super(sprite, false);
    }

    @Override
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, ElementsBuffer positions) {
        draw(drawingTools, anchor, positions, 0.0);
    }

    @Override
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, ElementsBuffer positions, double rotationAngle) {
        GL gl = drawingTools.getGl();
        update(gl);

        gl.glEnable(GL.GL_TEXTURE_2D);
        bindTexture(gl);

        gl.glEnable(GL.GL_BLEND);
        gl.glBlendFunc(GL.GL_SRC_ALPHA, GL.GL_ONE_MINUS_SRC_ALPHA);
        gl.glEnable(GL.GL_ALPHA_TEST);
        gl.glAlphaFunc(GL.GL_GREATER, 0);

        /*
         * Draw the sprite.
         */
        TransformationManager transformationManager = drawingTools.getTransformationManager();
        Transformation canvasProjection = transformationManager.getCanvasProjection();

        boolean needProjection = drawingTools.getTransformationManager().isUsingSceneCoordinate();
        drawingTools.getTransformationManager().useWindowCoordinate();

        FloatBuffer data = positions.getData();
        data.rewind();
        while (data.remaining() >= positions.getElementsSize()) {
            Vector3d position = new Vector3d(data.get(), data.get(), data.get());
            if (positions.getElementsSize() == 4) {
                data.get();
            }

            Vector3d projected;
            if (needProjection) {
                projected = canvasProjection.project(position);
            } else {
                projected = position;
            }

            int x = (int) (projected.getX());
            int y = (int) (projected.getY());
            double z = projected.getZ();

            gl.glPushMatrix();

            /* Translate to the anchor point's position then rotate about the anchor point */
            gl.glTranslatef((float) x, (float) y, 0.0f);
            gl.glRotatef((float) rotationAngle, 0.0f, 0.0f, 1.0f);

            /* Get the anchor-dependent x and y coordinates of the quad's edges */
            float[] coordsX = getXCoordinates(anchor);
            float[] coordsY = getYCoordinates(anchor);

            gl.glBegin(GL.GL_QUADS);
                gl.glTexCoord2f(0, 1);
                gl.glVertex3d(coordsX[0], coordsY[0], z);
                gl.glTexCoord2f(0, 0);
                gl.glVertex3d(coordsX[0],  coordsY[1], z);
                gl.glTexCoord2f(1, 0);
                gl.glVertex3d(coordsX[1], coordsY[1], z);
                gl.glTexCoord2f(1, 1);
                gl.glVertex3d(coordsX[1], coordsY[0], z);
            gl.glEnd();

            gl.glPopMatrix();
        }

        if (needProjection) {
            drawingTools.getTransformationManager().useSceneCoordinate();
        }

        gl.glDisable(GL.GL_TEXTURE_2D);
        gl.glDisable(GL.GL_BLEND);
        gl.glDisable(GL.GL_ALPHA_TEST);
    }

    /**
     * Returns the x coordinates of the quad's left and right edges depending on the given anchor.
     * The difference between the right and left coordinates is equal to the image width.
     * @param anchor the given anchor.
     * @return the x coordinates of the quad's left and right edges.
     */
    private float[] getXCoordinates(SpriteAnchorPosition anchor) {
        float[] coords = new float[2];
        int spriteWidth = getSprite().getWidth();
        int imageWidth = getImage().getWidth();

        float deltaW = (float)(imageWidth - spriteWidth) / 2.0f;

        /*
         * This offset ensures that the difference between the two coordinates
         * is equal to the image width and that they are aligned on
         * integer values.
         */
        float Woffset = 0.0f;

        if (((imageWidth - spriteWidth) % 2) != 0) {
            Woffset = 1.0f;
        }

        switch (anchor) {
            case LEFT:
            case LOWER_LEFT:
            case UPPER_LEFT:
                coords[0] = -deltaW;
                coords[1] = (float)spriteWidth + deltaW + Woffset;
            break;
            case UP:
            case CENTER:
            case DOWN:
                coords[0] = (float)-imageWidth/2.0f;
                coords[1] = (float)+imageWidth/2.0f;
            break;
            case RIGHT:
            case LOWER_RIGHT:
            case UPPER_RIGHT:
                coords[0] = -((float)spriteWidth+deltaW+Woffset);
                coords[1] = +deltaW;
            break;
            default:
                coords[0] = (float)-imageWidth/2.0f;
                coords[1] = (float)+imageWidth/2.0f;
            break;
        }

        return coords;
    }

    /**
     * Returns the y coordinates of the quad's bottom and top edges depending on the given anchor.
     * The difference between the top and bottom coordinates is equal to the image height.
     * @param anchor the given anchor.
     * @return the y coordinates of the quad's bottom and top edges.
     */
    private float[] getYCoordinates(SpriteAnchorPosition anchor) {
        float[] coords = new float[2];
        int spriteHeight = getSprite().getHeight();
        int imageHeight = getImage().getHeight();

        float deltaH = (float) (imageHeight - spriteHeight) / 2.0f;

        /*
         * This offset ensures that the difference between the two coordinates
         * is equal to the image height and that they are aligned on
         * integer values.
         */
        float Hoffset = 0.0f;

        if (((imageHeight - spriteHeight) % 2) != 0) {
            Hoffset = 1.0f;
        }

        switch (anchor) {
            case UPPER_LEFT:
            case UP:
            case UPPER_RIGHT:
                coords[0] = -((float)spriteHeight+deltaH+Hoffset);
                coords[1] = deltaH;
            break;
            case LEFT:
            case CENTER:
            case RIGHT:
                coords[0] = (float)-imageHeight/2.0f;
                coords[1] = (float)+imageHeight/2.0f;
            break;
            case LOWER_LEFT:
            case DOWN:
            case LOWER_RIGHT:
                coords[0] = -deltaH;
                coords[1] = (float)spriteHeight+deltaH+Hoffset;
            break;
            default:
                coords[0] = (float)-imageHeight/2.0f;
                coords[1] = (float)+imageHeight/2.0f;
            break;
        }

        return coords;
    }

}
