/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.HardwareFailException;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLCanvas;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.sprite.Sprite;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.sprite.SpriteDrawer;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

/**
 * Abstract JoGL implementation of a sprite
 *
 * @author Pierre Lando
 */
class JoGLSprite implements Sprite {

    /**
     * Current sprite drawer.
     */
    private SpriteDrawer spriteDrawer;

    /**
     * The sprite renderer.
     */
    private JoGLSpriteRenderer spriteRenderer;

    /**
     * Current up to date status of this sprite.
     */
    private boolean upToDate;

    /**
     * Current sprite width.
     */
    private final int width;

    /**
     * Current sprite height.
     */
    private final int height;

    /**
     * Default constructor.
     * The constructor is package : only {@see JoGLSpriteManager} can instantiate this object.
     * @param canvas the current canvas.
     * @param width  sprite width.
     * @param height sprite height.
     */
    JoGLSprite(JoGLCanvas canvas, int width, int height) {
        this.width = width;
        this.height = height;
        upToDate = false;
    }

    /**
     * Draw this sprite with the given drawing tools at the given position.
     * @param drawingTools the drawing tools to use.
     * @param anchor anchor position.
     * @param positions the position.
     */
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, ElementsBuffer positions) {
        getRenderer(drawingTools).draw(drawingTools, anchor, positions);
    }

    /**
     * Draw this sprite with the given drawing tools at the given position.
     * @param drawingTools the drawing tools to use.
     * @param anchor anchor position.
     * @param position the position.
     */
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, Vector3d position) {
        getRenderer(drawingTools).draw(drawingTools, anchor, position);
    }

    /**
     * Ask this sprite to dispose.
     * @param drawingTools the drawing tools to use.
     */
    void dispose(JoGLDrawingTools drawingTools) {
        getRenderer(drawingTools).dispose(drawingTools);
    }


    @Override
    public void setDrawer(SpriteDrawer drawer) {
        spriteDrawer = drawer;
        upToDate = false;
    }

    @Override
    public SpriteDrawer getDrawer() {
        return spriteDrawer;
    }

    @Override
    public void redraw() {
        upToDate = false;
    }

    @Override
    public int getWidth() {
        return width;
    }

    @Override
    public int getHeight() {
        return height;
    }

    /**
     * Return the up to date status.
     * @return the up to date status.
     */
    boolean isUpToDate() {
        return upToDate;
    }

    /**
     * Set the up to date status.
     * @param upToDate the new up to date status.
     */
    void setUpToDate(boolean upToDate) {
        this.upToDate = upToDate;
    }

    /**
     * Ask the sprite to reset the OpenGl status.
     * @param drawingTools the current drawing tools.
     */
    public void reload(JoGLDrawingTools drawingTools) {
        getRenderer(drawingTools).reload();
    }

    /**
     * Return the sprite implementation.
     * @param drawingTools the drawing tools.
     * @return the sprite implementation.
     */
    private JoGLSpriteRenderer getRenderer(JoGLDrawingTools drawingTools) {
        if (spriteRenderer == null) {
            if (drawingTools.getCanvas().getJoGLParameters().usePointSprite()) {
                try {
                    spriteRenderer = new PointRenderer(drawingTools, this);
                } catch (HardwareFailException e) {
                    // TODO : verbose this in debug mode
                    // System.out.println(e.getMessage() + " | Now use QuadsTextureRenderer.");
                    spriteRenderer = new QuadsTextureRenderer(this);
                }
            } else {
                spriteRenderer = new QuadsTextureRenderer(this);
            }
        }
        return spriteRenderer;
    }
}
