/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import java.nio.Buffer;
import java.nio.ByteBuffer;

import javax.media.opengl.GL;

import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.implementation.jogl.utils.texture.TextureBufferedImage;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

/**
 * @author Pierre Lando
 */
abstract class AbstractTextureRenderer implements JoGLSpriteRenderer {

    /**
     * The sprite this object render.
     */
    private final JoGLSprite sprite;

    /**
     * The current buffered drawingTools where the sprite is drawn.
     */
    private final TextureDrawingTools drawingTools;


    /**
     * The current OpenGl texture name.
     */
    private Integer textureId;

    /**
     * Default constructor.
     * @param sprite the sprite this renderer draw.
     * @param squareTexture true if the texture should be a square.
     */
    AbstractTextureRenderer(JoGLSprite sprite, boolean squareTexture) {
        this.sprite = sprite;
        drawingTools = new TextureDrawingTools(sprite.getWidth(), sprite.getHeight(), squareTexture);
        textureId = null;
    }

    @Override
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, Vector3d position) {
        ElementsBuffer positions = drawingTools.getCanvas().getBuffersManager().createElementsBuffer();
        positions.setData(position.getDataAsFloatArray(), 3);
        draw(drawingTools, anchor, positions);
        drawingTools.getCanvas().getBuffersManager().dispose(positions);
    }

    @Override
    public void dispose(JoGLDrawingTools drawingTools) {
        GL gl = drawingTools.getGl();
        if ((textureId != null) && gl.glIsTexture(textureId)) {
            int[] t = new int[] {textureId};
            gl.glDeleteTextures(1, t, 0);
            textureId = null;
        }
    }

    @Override
    public void reload() {
        textureId = null;
        sprite.setUpToDate(false);
    }

    /**
     * Update the texture.
     * @param gl the current OpenGl context.
     */
    protected void update(GL gl) {
        if (!sprite.isUpToDate()) {
            if (sprite.getDrawer() != null) {
                drawingTools.accept(sprite.getDrawer());
                Buffer textureData = ByteBuffer.wrap(getImage().getRGBAData());
                textureData.rewind();
                bindTexture(gl);
                gl.glTexImage2D(
                        GL.GL_TEXTURE_2D,
                        0,          // level 0, no MIP mapping.
                        GL.GL_RGBA,
                        getImage().getWidth(),
                        getImage().getHeight(),
                        0,          // no border.
                        GL.GL_RGBA,
                        GL.GL_UNSIGNED_BYTE,
                        textureData
                );
            }
            sprite.setUpToDate(true);
        }
    }

    /**
     * Bind the texture. Create it if it wasn't.
     * @param gl the OpenGl context.
     */
    protected void bindTexture(GL gl) {
        if ((textureId == null) || (!gl.glIsTexture(textureId))) {
            int[] t = new int[1];
            gl.glGenTextures(1, t, 0);
            textureId = t[0];
        }
        gl.glBindTexture(GL.GL_TEXTURE_2D, textureId);
        gl.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MIN_FILTER, GL.GL_NEAREST);
        gl.glTexParameteri(GL.GL_TEXTURE_2D, GL.GL_TEXTURE_MAG_FILTER, GL.GL_NEAREST);
        gl.glTexEnvi(GL.GL_TEXTURE_ENV, GL.GL_TEXTURE_ENV_MODE, GL.GL_REPLACE);
    }

    /**
     * Return the image.
     * @return the image.
     */
    protected TextureBufferedImage getImage() {
        return drawingTools.getImage();
    }

    /**
     * Return the sprite.
     * @return the sprite.
     */
    protected JoGLSprite getSprite() {
        return sprite;
    }

    /**
     * Return the deltaX to apply to the sprite in regards to the given anchor.
     * @param anchor the given anchor.
     * @return the deltaX to apply to the sprite in regards to the given anchor.
     */
    protected float getAnchorDeltaX(SpriteAnchorPosition anchor) {
        int spriteWidth = getSprite().getWidth();
        switch (anchor) {
            case LEFT:
            case LOWER_LEFT:
            case UPPER_LEFT:
                return spriteWidth / 2f;
            case UP:
            case CENTER:
            case DOWN:
                return 0;
            case RIGHT:
            case LOWER_RIGHT:
            case UPPER_RIGHT:
                return -spriteWidth / 2f;
            default:
                return 0;
        }
    }

    /**
     * Return the deltaY to apply to the sprite in regards to the given anchor.
     * @param anchor the given anchor.
     * @return the deltaY to apply to the sprite in regards to the given anchor.
     */
    protected float getAnchorDeltaY(SpriteAnchorPosition anchor) {
        int spriteHeight = getSprite().getHeight();
        switch (anchor) {
            case UPPER_LEFT:
            case UP:
            case UPPER_RIGHT:
                return -spriteHeight / 2f;
            case LEFT:
            case CENTER:
            case RIGHT:
                return 0;
            case LOWER_LEFT:
            case DOWN:
            case LOWER_RIGHT:
                return spriteHeight / 2f;
            default:
                return 0;
        }
    }
}
