/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.examples.lineRendering;

import org.scilab.forge.scirenderer.Canvas;
import org.scilab.forge.scirenderer.Drawer;
import org.scilab.forge.scirenderer.DrawingTools;
import org.scilab.forge.scirenderer.SciRendererException;
import org.scilab.forge.scirenderer.examples.utils.MouseRotationAdapter;
import org.scilab.forge.scirenderer.shapes.appearance.Appearance;
import org.scilab.forge.scirenderer.shapes.appearance.Color;
import org.scilab.forge.scirenderer.shapes.geometry.Geometry;
import org.scilab.forge.scirenderer.shapes.geometry.DefaultGeometry;
import org.scilab.forge.scirenderer.sprite.Sprite;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.sprite.SpriteDrawer;
import org.scilab.forge.scirenderer.sprite.SpriteDrawingTools;
import org.scilab.forge.scirenderer.sprite.TextEntity;
import org.scilab.forge.scirenderer.tranformations.DegenerateMatrixException;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.TransformationFactory;
import org.scilab.forge.scirenderer.tranformations.Vector3d;
import org.scilab.forge.scirenderer.utils.shapes.geometry.TetrahedronFactory;

import java.awt.Dimension;

/**
 * @author Pierre Lando
 */
public class LineDrawer implements Drawer {
    private final MouseRotationAdapter mra;
    private final DefaultGeometry tetrahedron;
    private final Sprite message;

    public LineDrawer(Canvas canvas, MouseRotationAdapter mra) {
        this.mra = mra;
        this.tetrahedron = TetrahedronFactory.createTetrahedron(canvas);
        tetrahedron.setFaceCullingMode(Geometry.FaceCullingMode.CCW);
        message = createMessage(canvas);
    }

    /**
     * Switch culled face.
     */
    public void switchFace() {
        Geometry.FaceCullingMode newCullingMode =
                (tetrahedron.getFaceCullingMode() == Geometry.FaceCullingMode.CW) ?
                Geometry.FaceCullingMode.CCW:
                Geometry.FaceCullingMode.CW;
        tetrahedron.setFaceCullingMode(newCullingMode);
    }

    @Override
    public void draw(DrawingTools dt) {
        dt.clear(new Color(1f, 1f, 1f));

        try {
            Transformation projection = TransformationFactory.getPreferredAspectRatioTransformation(dt.getCanvas().getDimension(), 1f);
            dt.getTransformationManager().getProjectionStack().push(projection);
        } catch (DegenerateMatrixException ignored) {
            // Should not occur.
        }

        dt.draw(message, SpriteAnchorPosition.UPPER_LEFT, new Vector3d(-1, 1, 0));

        try {
            dt.getTransformationManager().getModelViewStack().pushRightMultiply(TransformationFactory.getScaleTransformation(.1, .1, .1));

            dt.getTransformationManager().getModelViewStack().pushRightMultiply(TransformationFactory.getRotationTransformation(mra.getRotation()));

            Appearance appearance = new Appearance();
            appearance.setLineColor(new Color(.2f, .2f, .2f));
            appearance.setLineWidth(3);
            dt.draw(tetrahedron, appearance);
        } catch (SciRendererException ignored) {
            // Should not occur.
        }
    }


    /**
     * Create a help message.
     * @param canvas the canvas where the message sprite is created.
     * @return a sprite that draws the message.
     */
    private Sprite createMessage(Canvas canvas) {
        final TextEntity text = new TextEntity("Press 'F' to switch culling mode");
        Dimension dimension = canvas.getSpriteManager().getSize(text);
        Sprite message = canvas.getSpriteManager().createSprite(dimension.width, dimension.height);
        message.setDrawer(new SpriteDrawer() {

            @Override
            public void draw(SpriteDrawingTools drawingTools) {
                drawingTools.draw(text, 0, 0);
            }

            @Override
            public OriginPosition getOriginPosition() {
                return OriginPosition.UPPER_LEFT;
            }
        });

        return message;
    }
}
