#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/wait.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <string>

#include "scim-bridge-agent.h"
#include "scim-bridge-agent-kernel.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-agent-exception.h"
#include "scim-bridge-environment.h"

using std::cerr;
using std::cout;
using std::endl;
using std::ifstream;
using std::ofstream;
using std::string;
using std::stringstream;

static const int ACCEPT_RETRY_SLEEP_MICROSEC = 5000;
static const int MAX_CLIENT_COUNT = 5;

static int input_socket_fd;
static int output_socket_fd;

static bool exit_with_no_client;
static bool enable_daemon;

static ScimBridgeAgentKernel *kernel = NULL;

static sem_t shutdown_lock;
static pthread_t shutdown_thread;
static bool shutdown_triggered;

/* Functions */
static void init_sockets ();
static void wait_for_connection ();
static void connected (const int input_fd, const int output_fd);
static void close_sockets ();
static void sig_shutdown (int sig_id);
static void setup_environments (int argc, char **argv);
static bool str_to_int (int *value, const char *str);
static void initialize_kernel ();
static void finalize_kernel ();
static void create_lockfile ();
static void destroy_lockfile ();

static void *run_shutdown (void *arg);

/* Implementations */
bool str_to_int (int *value, const char *str)
{
    int i;
    for (i = 0; str[i] != '\0'; ++i) {
        switch (str[i]) {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
                break;
            default:
                return true;
        }
    }

    *value = atoi (str);
    return false;
}


void init_sockets ()
{
    const char *input_socket_path = scim_bridge_environment_get_client_to_agent_socket_path ();
    input_socket_fd = socket (PF_UNIX, SOCK_STREAM, 0);
    if (input_socket_fd < 0) {
        scim_bridge_perrorln ("Error occured while making the input socket: %s", strerror (errno));
        exit (-1);
    }

    struct sockaddr_un input_socket_addr;
    memset (&input_socket_addr, 0, sizeof (struct sockaddr_un));
    input_socket_addr.sun_family = AF_UNIX;
    strcpy (input_socket_addr.sun_path, input_socket_path);

    unlink (input_socket_path);
    if (bind (input_socket_fd, (struct sockaddr*)&input_socket_addr, sizeof (struct sockaddr_un)) != 0) {
        scim_bridge_perrorln ("Error occured while binding the input socket: %s", strerror (errno));
        exit (-1);
    }

    if (listen (input_socket_fd, MAX_CLIENT_COUNT) != 0) {
        scim_bridge_perrorln ("Error occured while starting the input socket: %s", strerror (errno));
        exit (-1);
    }

    const char *output_socket_path = scim_bridge_environment_get_agent_to_client_socket_path ();
    output_socket_fd = socket (PF_UNIX, SOCK_STREAM, 0);
    if (output_socket_fd < 0) {
        scim_bridge_perrorln ("Error occured while making the output socket: %s", strerror (errno));
        exit (-1);
    }

    struct sockaddr_un output_socket_addr;
    memset (&output_socket_addr, 0, sizeof (struct sockaddr_un));
    output_socket_addr.sun_family = AF_UNIX;
    strcpy (output_socket_addr.sun_path, output_socket_path);

    unlink (output_socket_path);
    if (bind (output_socket_fd, (struct sockaddr*)&output_socket_addr, sizeof (struct sockaddr_un)) != 0) {
        scim_bridge_perrorln ("Error occured while binding the output socket: %s", strerror (errno));
        exit (-1);
    }

    if (listen (output_socket_fd, MAX_CLIENT_COUNT) != 0) {
        scim_bridge_perrorln ("Error occured while starting the output socket: %s", strerror (errno));
        exit (-1);
    }

    if (enable_daemon) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Daemonizing...");
        if (daemon (0, 0)) {
            scim_bridge_perrorln ("Cannot daemonize myself: %s", strerror (errno));
            exit (-1);
        }

        scim_bridge_agent_enable_syslog ();
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Daemonize done");

        update_lockfile ();
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Stand alone mode, deamonization is canceled...");
        scim_bridge_agent_enable_syslog ();
    }
}


void wait_for_connection ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Waiting for connection...");
    int input_fd = -1;
    while (input_socket_fd > 0 && input_fd < 0) {
        struct sockaddr_un empty_socket_addr;
        socklen_t emtpy_socket_size = sizeof (empty_socket_addr);
        input_fd = accept (input_socket_fd, (sockaddr*)&empty_socket_addr, &emtpy_socket_size);
        if (input_fd < 0) {
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 1, "Connection opps...");
            usleep (ACCEPT_RETRY_SLEEP_MICROSEC);
        }
    }

    int output_fd = -1;
    while (output_socket_fd > 0 && output_fd < 0) {
        struct sockaddr_un empty_socket_addr;
        socklen_t emtpy_socket_size = sizeof (empty_socket_addr);
        output_fd = accept (output_socket_fd, (sockaddr*)&empty_socket_addr, &emtpy_socket_size);
        if (output_fd < 0) {
            scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 1, "Connection opps...");
            usleep (ACCEPT_RETRY_SLEEP_MICROSEC);
        }
    }

    if (input_fd > 0 && output_fd > 0) connected (input_fd, output_fd);
}


void connected (const int input_fd, const int output_fd)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Connected...");

    int retval = 0;
    try {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 1, "Establishing the connection...");
        kernel->connect (input_fd, output_fd);
    } catch (ScimBridgeAgentException ex) {
        scim_bridge_perrorln ("Exception occured inside of the kernel: %s", ex.what ());
        retval = -1;
    }
}


void close_sockets ()
{
    int fd;

    fd = input_socket_fd;
    input_socket_fd = -1;
    shutdown (fd, SHUT_RDWR);
    close (fd);

    const char *input_socket_path = scim_bridge_environment_get_client_to_agent_socket_path ();
    unlink (input_socket_path);

    fd = output_socket_fd;
    output_socket_fd = -1;
    shutdown (fd, SHUT_RDWR);
    close (fd);

    const char *output_socket_path = scim_bridge_environment_get_agent_to_client_socket_path ();
    unlink (output_socket_path);
}


void initialize_kernel ()
{
    try {
        kernel = ScimBridgeAgentKernel::create (exit_with_no_client);
    } catch (ScimBridgeAgentException ex) {
        cerr << ex.what () << endl;
        exit (-1);
    }
}


void finalize_kernel ()
{
    if (kernel != NULL) {
        delete kernel;
        kernel = NULL;
    }
}


void setup_environments (int argc, char **argv)
{
    int debug_level = 0;

    bool debug_scim = false;
    bool debug_messenger = false;
    bool debug_imcontext = false;
    bool debug_agent = false;
    bool debug_client = false;

    exit_with_no_client = true;
    enable_daemon = true;

    const struct option long_options[] = {
        {"help", 0, NULL, 'h'},
        {"verbose", 0, NULL, 'v'},
        {"quiet", 0, NULL, 'q'},
        {"debuglevel", 1, NULL, 'l'},
        {"debugflags", 1, NULL, 'b'},
        {"noexit", 0, NULL, 'n'},
        {"standalone", 0, NULL, 's'},
        {0, 0, NULL, 0}
    };

    char short_options[] = "vhqdls:b:";

    int option = 0;
    while (option != EOF) {
        option = getopt_long (argc, argv, short_options, long_options, NULL);
        switch (option) {
            case 'v':
                debug_level = 9;
                debug_scim = true;
                debug_messenger = true;
                debug_agent = true;
                debug_client = true;
                debug_imcontext = true;
                break;
            case 'q':
                debug_level = 0;
                break;
            case 'b':
                if (strcasecmp (optarg, "all") == 0) {
                    debug_scim = true;
                    debug_messenger = true;
                    debug_agent = true;
                    debug_client = true;
                    debug_imcontext = true;
                } else if (strcasecmp (optarg, "none") == 0) {
                    debug_scim = false;
                    debug_messenger = false;
                    debug_agent = false;
                    debug_client = false;
                    debug_imcontext = false;
                } else if (strcasecmp (optarg, "agent") == 0) {
                    debug_agent = true;
                } else if (strcasecmp (optarg, "client") == 0) {
                    debug_client = true;
                } else if (strcasecmp (optarg, "messenger") == 0) {
                    debug_messenger = true;
                } else if (strcasecmp (optarg, "scim") == 0) {
                    debug_scim = true;
                } else if (strcasecmp (optarg, "imcontext") == 0) {
                    debug_imcontext = true;
                } else {
                    cout << "Unknwon debug flag: " << optarg << endl;
                }
                break;
            case 'l':
                if (str_to_int (&debug_level, optarg) || debug_level < 0) {
                    cerr << "Invalid debug level: " << optarg << endl;
                    exit (-1);
                }
                break;
            case 'n':
                exit_with_no_client = false;
                break;
            case 's':
                enable_daemon = false;
                break;
            case 'h':
                cout << "Usage: scim-bridge-agent [options]" << endl;
                cout << " Options" << endl << endl;
                cout << " -h, --help\tGive this help list" << endl;
                cout << " -v, --verbose\tVerbosely print out the debug message into standard output.This option equals to '--debuglevel=9 --debugflags=all'" << endl;
                cout << " -q, --quiet\tMake it print no debug message at all.This option equals to '--debuglevel=0 --debugflags=none'" << endl;
                cout << " -b, --debugflags\tSet which category of debug output do you need.Select one or more from 'all', 'none', 'agent', 'messenger', 'imcontext', and 'scim'." << endl;
                cout << " -l, --debuglevel\tSet how verbosely should it print debug output.'--debuglevel=0' equals to '--queit', and '--debuglevel=9' equals to '--verbose'" << endl;
                cout << " -standalone, --standalone\tGiven this, scim-brige-agent won't daemonize itself." << endl;
                cout << " -n, --noexit\tGiven this, scim-brige-agent won't exit when there is no client." << endl;
                exit (0);
                break;
            case ':':
            case '?':
                exit (-1);
            default:
                break;
        }
    }

    if (optind < argc) {
        cerr << "Invalid argument: " << argv[optind] << endl;
        exit (-1);
    }

    stringstream sstream;
    sstream.str ("");
    if (debug_imcontext) sstream << SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_IMCONTEXT << " ";
    if (debug_messenger) sstream << SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_MESSENGER << " ";
    if (debug_scim) sstream << SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_SCIM  << " ";
    if (debug_agent) sstream << SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_AGENT << " ";
    if (debug_client) sstream << SCIM_BRIDGE_ENV_VALUE_DEBUG_FLAGS_CLIENT << " ";
    setenv (SCIM_BRIDGE_ENV_NAME_DEBUG_FLAGS, sstream.str ().c_str (), 1);

    if (!debug_imcontext && !debug_messenger && !debug_scim && !debug_agent && !debug_client) debug_level = false;

    if (debug_level >= 0) {
        sstream.str ("");
        sstream << debug_level;
        setenv (SCIM_BRIDGE_ENV_NAME_DEBUG_LEVEL, sstream.str ().c_str (), 1);
    }
}


void create_lockfile ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Trying to create the lockfile...");

    const char *lockfile_path = scim_bridge_environment_get_lockfile_path ();
    ifstream input_stream (lockfile_path);

    pid_t pid;
    input_stream >> pid;

    input_stream.close ();

    if (input_stream.good ()) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Pinging for another process: pid = %d", pid);
        if (kill (pid, 0) == 0) {
            scim_bridge_perrorln ("Another agent is running.I'll exit.");
            exit (0);
        }
    }

    if (update_lockfile ()) exit (-1);
}


void destroy_lockfile ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Destroying the lockfile...");

    const char *lockfile_path = scim_bridge_environment_get_lockfile_path ();
    unlink (lockfile_path);
}


bool update_lockfile ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 1, "Updating the lockfile...");

    const char *lockfile_path = scim_bridge_environment_get_lockfile_path ();
    ofstream output_stream (lockfile_path);

    output_stream << getpid () << endl;
    output_stream.close ();

    if (!output_stream.good ()) {
        scim_bridge_perrorln ("Cannot update the lockfile");
        unlink (lockfile_path);
        return 1;
    } else {
        return 0;
    }
}


void shutdown ()
{
    shutdown_triggered = true;
    sem_post (&shutdown_lock);
}


void do_shutdown ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Exiting...");

    close_sockets ();
    destroy_lockfile ();
    finalize_kernel ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Cleanup done");
}


void sig_shutdown (int sig_id)
{
    shutdown_triggered = true;
    sem_post (&shutdown_lock);
}


void *run_shutdown (void *arg)
{
    while (!shutdown_triggered) sem_wait (&shutdown_lock);

    do_shutdown ();

    return NULL;
}


int main (int argc, char **argv)
{
    setup_environments (argc, argv);

    create_lockfile ();
    init_sockets ();

    shutdown_triggered = false;
    if (sem_init (&shutdown_lock, 0, 0) || pthread_create (&shutdown_thread, NULL, run_shutdown, NULL)) {
        scim_bridge_perrorln ("Cannot setup a shutdown hook: %s", strerror (errno));
        do_shutdown ();
        exit (-1);
    }

    initialize_kernel ();

    signal (SIGTERM, sig_shutdown);
    signal (SIGINT, sig_shutdown);
    signal (SIGQUIT, sig_shutdown);
    signal (SIGHUP, sig_shutdown);
    while (!shutdown_triggered) {
        wait_for_connection ();
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Exited, waiting for the shutdown hook...");
    scim_bridge_agent_disable_syslog ();

    while (pthread_join (shutdown_thread, NULL) && errno == EINTR);
    sem_destroy (&shutdown_lock);

    pthread_exit (NULL);
}
