//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [par] = CL_defParam(text,val,id,units,dim,typ,accv,valid)
// Definition of parameter structure for CL_inputParam
//
// Calling Sequence
// par = CL_defParam(text,val,id,units,dim,typ,accv,valid)
//
// Description
// <itemizedlist>
// <listitem>
// <p>Defines a parameter structure. </p>
// <p>Note: All the arguments are optional! </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// text: Text defining the parameter. Empty string by default. 
// val: Initial value (given in internal unit). 0 or ' ' (empty string) by default. 
// id: Identifier (string) useable in the 'valid' expression. Empty string by default. 
// units: [unit1, [unit2]] (strings), where unit1 is the internal unit (unit of val), and unit2 the unit used for input (unit2is the same as unit1 by default). By default: no units. If unit2 is defined, CL_unitConvert is used to convert from unit1 to unit2. 
// dim: Number of expected values. Default is 1. A value of -1 means 'any number including 0' (vector).
// typ: Type of parameter: 'r' (real) only is handled (default). 
// accv: Vector (1xN) of accepted values. By default: [], meaning that everything is accepted. NB: if units are defined, values in accv must be expressed in 'input' units (i.e. unit2). 
// valid: Expression (string) used to check the validity of the value. The expression used should be compatible with 'find' if many values are expected (i.e. if dim <> 1). NB: if units are defined, the values are checked while expressed in 'input' unit(i.e unit2), not internal unit. 
// par: Parameter structure. 
//
// Authors
// CNES - DCT/SB (AL)
//
// See also
// CL_inputParam
//
// Examples
// par=CL_defParam("param 1", val=1, accv=1:10);
//
// par=CL_defParam("param 2", val=[1,2], ..
// units=['m', 'km'], valid='$x == round($x)', dim=-1);
//
// par=CL_defParam("param 3",val=2000,units=['m', 'km']);

// Declarations:


// Code:

  // remove spaces inside strings
  function [str2] = strip(str)
    str2 = str;
    for i = 1:length(length(str))
       str2(i) = strcat(tokens(str(i), ' '), '');
    end
  endfunction

  if ~exists("text", "local")
    text = '';
  end

  if ~exists("typ", "local")
    typ = 'r';
  end

  if typ <> 'r'
    CL__error('Type not handled');
  end

  if ~exists("val", "local")
    if (typ == 'r'); val = 0; end
    if (typ == 's'); val = ""; end  // for the future...
  end

  unitf = 1; // unit conversion factor
  if ~exists("units", "local")
    units = ["", ""];  // no units
  elseif (typeof(units) <> 'string' | size(units,1) <> 1 | size(units,2) > 2 )
    CL__error('Invalid units (wrong type or size)');
  elseif (size(units,2) == 0 )
    units = ["", ""];
  elseif (size(units,2) == 1)
    units = strip([units(1), units(1)]);
  else
    units = strip(units); // remove spaces
    unitf = CL_unitConvert(1, units(2), units(1));
    if isempty(unitf)
       CL__error('Units (' + strcat(units,',') + ') not handled');
    end
  end

  if ~exists("id", "local")
    id = ""; // undefined name
  end
  if (typeof(id) <> 'string' | size(id,'*') <> 1)
     CL__error('Invalid id (wrong type or size)');
  end

  if ~exists("dim", "local")
    dim = 1; // one value
  end
  if (typeof(dim) <> 'constant' | length(dim) <> 1)
     CL__error('Invalid dimension (wrong type or size)');
  end
  if (dim <> round(dim) | dim < -1)
     CL__error('Invalid dimension (wrong value)');
  end

  if ~exists("accv", "local") // accepted values
    accv = [];
  end

  if ~exists("valid", "local") // validity condition
    valid = "";
  end

  // check type and dimension of value
  if (typ == 'r' & typeof(dim) <> 'constant' )
     CL__error('Invalid value (wrong type)');
  end
  if (size(val,1) > 1)
     CL__error('Invalid value (wrong size)');
  end
  if (dim <> -1 & size(val,2) <> dim)
     CL__error('Invalid value (wrong size)');
  end

  par = tlist(["", "id", "typ", "val", "libel", "units", ...
               "unitf", "dim", "accv", "valid"]);

  par.id     = id;
  par.typ    = typ;
  par.val    = val;
  par.libel  = text;
  par.units  = units;
  par.unitf  = unitf;
  par.dim    = dim;
  par.accv   = accv;
  par.valid  = valid;

endfunction
