//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [ires] = CL_intervInters(varargin)
// Intersection of sets of intervals
//
// Calling Sequence
// [ires] = CL_intervInters(i1,i2 [,i3,...,ip])
//
// Description
// <itemizedlist><listitem>
// This function computes the intersection of sets of intervals.
// <para> The intervals are gathered in different sets (i1, i2,...,ip). </para>
// <para> A real number x belongs to the intersection of (i1, i2,...,ip) if there exists at least one interval in each set which x belongs to.</para>
// <inlinemediaobject><imageobject><imagedata fileref="intersection.gif"/></imageobject></inlinemediaobject>
// </listitem>
// <listitem>
// Notes: 
// <para> - The intervals in each set should have empty intersections (use CL_intervUnion if needed). </para>
// <para> - Sets of intervals may be empty: CL_intervInters(i1,[]) = []. </para>
// <para> - Resulting set of intervals are sorted in ascending order (first row) </para>
// </listitem>
// </itemizedlist>
//
// Parameters
// i1: Set of intervals [start ; end] (2xN)
// i2: Set of intervals [start ; end] (2xN2)
// ip: Set of intervals [start ; end] (2xNp)
// ires: Intesection of i1,i2...,iN (2xM)
//
// See also
// CL_intervUnion
// CL_intervInv
//
// Authors
// CNES - DCT/SB
//
// Examples
// i1=[ [1;3] , [5;6] , [10;12]];
// i2=[ [2;4] , [5.5;5.7] , [5.8;15]];
// ires = CL_intervInters(i1,i2);
//
// i1=[ [1;3] , [5;6] , [10;12]];
// i2=[ [2;4] , [5.5;5.7] , [5.8;15]];
// i3=[ [1.1;1.2] , [3.5;7] , [11;20]];
// ires = CL_intervInters(i1,i2,i3);

// Declarations:


// Code:

if(size(varargin) < 2)
  CL__error('2 intervals minimum');
end

for i=1:size(varargin)
  if(varargin(i) ~= [])
    if(size(varargin(i),1)~=2)
      CL__error('Interval '+string(i)+' must be of size 2xN');
    end
  end
end

// 2 intervalles
if(size(varargin) ==2)
  if(varargin(1) == [] | varargin(2) == [])
    ires = [];
  else
  // tri par ordre croissant
  i1 = CL_matSort(varargin(1),1,'r');
  i2 = CL_matSort(varargin(2),1,'r');
  // i1 est le plus petit premier intervalle
  if(i1(1,1) > i2(1,1))
    i1_tmp = i1;
    i1 = i2;
    i2 = i1_tmp;
   clear i1_tmp;
  end

  N1 = size(i1,2);
  N2 = size(i2,2);

  ires = [];
  //i1(:,1);

  k = 1;
  // on parcourt les intervalles de i1 :
  while(k <= N1)
  // cas ou i1 =       [      ]
  // et     i2 =     [    ]
  // ---> ires =       [  ]
    ii = find(i2(2,:) <= i1(2,k) & i2(2,:) >= i1(1,k) & i2(1,:) <= i1(1,k)) ;
  if(ii ~= [])
    ires = [ires,[i1(1,k);i2(2,ii(1))]] // on rajoute aux resultats l'intervalle [deb i1 ; fin i2]
  else

  // cas ou i1 =       [              ]
  // et     i2 =          [    ]    [    ]
  // ---> ires =          [    ]    [ ]
  ii = find(i2(1,:) <= i1(2,k) & i2(1,:) >= i1(1,k)) ;
  if(ii ~= [])
    for j=1:length(ii)
      ires = [ires,[i2(1,ii(j));min(i2(2,ii(j)),i1(2,k))]] // on rajoute aux resultats tous les intervalles [deb i2 ; min(fin i2, fin i1)]
    end
  else

    // cas ou i1 =       [        ]
    // et     i2 =     [             ]
    // ---> ires =       [        ]
  ii = find(i2(1,:) <= i1(1,k) & i2(2,:) >= i1(2,k)) ;
  if(ii ~= [])
    for j=1:length(ii)
      ires = [ires,i1(:,k)] // on rajoute aux resultats l'intervalle i1
    end
  end
  end
  end

  k=k+1;
  end
  end
// plus que 2 intervalles : on fait les intersections une a une
elseif(size(varargin) > 2)
  ires = CL_intervInters(varargin(1),varargin(2));
  for k=3:size(varargin)
    ires = CL_intervInters(ires,varargin(k))
  end
end


endfunction
