//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [delta_v,dv1,dv2,dv3,anv1,anv2,anv3]=CL_man_biElliptic(ai,af,rt,mu)
// Delta V for a bi-elliptic transfer
//
// Calling Sequence
// [delta_v,dv1,dv2,dv3,anv1,anv2,anv3]=CL_man_biElliptic(ai,af,rt [,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the maneuvers of a bi-elliptical transfer from a circular 
// orbit with semi-major axis <emphasis role="bold">ai</emphasis> 
// to a circular orbit with semi-major axis <emphasis role="bold">af</emphasis>. 
// <para> The apogee radius of the elliptical transfer orbit is 
// <emphasis role="bold">rt</emphasis>. </para>
// <para><emphasis role="bold">delta-v</emphasis> is the sum of the norms 
// of the velocity increments.</para>
// <para>Velocity increments are expressed in spherical coordinates in the QSW frame: 
// [lambda; phi; dv], where lambda is the in-plane angle (+%pi: towards planet and +%pi/2: 
// ~along velocity), phi is the out-of-plane angle, positive towards the angular momentum vector 
// (the angular momentum vector is perpendicular to the orbit plane and oriented according to 
// right hand rule), dv is the norm of the velocity increment. </para>
// <para><inlinemediaobject><imageobject><imagedata fileref="bielliptic.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// ai : Semi-major axis of initial circular orbit  [m] (1xN)
// af : Semi-major axis of final circular orbit [m] (1xN)
// rt : Radius at the position of the second maneuver [m] (1xN)
// mu : (optionnal) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// delta_v : Total |delta-V| (=|dv1|+|dv2|+|dv3]) [m/s] (1xN)
// dv1: First delta-V, in spherical coordinates in the QSW frame [lambda;phi;dv] [rad,rad,m/s] (3xN)
// dv2: Second delta-V, in spherical coordinates in the QSW frame [lambda;phi;dv] [rad,rad,m/s] (3xN)
// dv3: Third delta-V, in spherical coordinates in the QSW frame [lambda;phi;dv] [rad,rad,m/s] (3xN)
// anv1: True anomaly at the position of the 1st maneuver : initial orbit is circular so this is an arbitrary value of 0  (1xN)
// anv2: True anomaly at the position of the 2nd maneuver (either 0 or %pi) [rad] (1xN)
// anv3: True anomaly at the position of the 3rd maneuver (either 0 or %pi) [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// Orbital Mechanics for engineering students, H D Curtis, Chapter 6 (equation 6.4a)
//
// See also
// CL_man_hohmann
// CL_man_hohmannG
//
// Examples
// // 7000 km to 98 000km through a 280 000 transfer orbit:
// ai = 7000.e3;
// af = 98000.e3;
// rt = 280000.e3;
// [delta_v,dv1,dv2,dv3,anv1,anv2,anv3]=CL_man_biElliptic(ai,af,rt)
// // Check results :
// kep = [ai ; 0 ; %pi/2 ; 0 ; 0 ; anv1];
// kep1 = CL_man_applyDv(kep,dv1);
// kep1(6) = anv2;
// kep2 = CL_man_applyDv(kep1,dv2);
// kep2(6) = anv3;
// kep3 = CL_man_applyDv(kep2,dv3)
//
// // Same example with a Hohmann transfer:
// // more expensive !
// [delta_v,dv1,dv2,anv1,anv2] = CL_man_hohmann(ai,af) 

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end


// Compute norms of DVs :
a1 = (ai+rt) / 2;
a2 = (af+rt) / 2;
dv1 = sqrt(2.0.*mu./ai - mu./a1) - sqrt(mu./ai) ;
dv2 = sqrt(2.0.*mu./rt - mu./a2) - sqrt(2.0.*mu./rt - mu./a1) ;
dv3 = sqrt(mu./af) - sqrt(2.0.*mu./af - mu./a2)  ;

// Compute DVs in spherical coordinates in QSW frame :
ii = find(dv1 <0);
dv1 = [ %pi/2*ones(dv1) ; zeros(dv1) ; abs(dv1) ];
if(ii~=[])
  dv1(1,ii) = 3*%pi/2;
end

ii = find(dv2 <0);
dv2 = [ %pi/2*ones(dv2) ; zeros(dv2) ; abs(dv2) ];
if(ii~=[])
  dv2(1,ii) = 3*%pi/2;
end

ii = find(dv3 <0);
dv3 = [ %pi/2*ones(dv3) ; zeros(dv3) ; abs(dv3) ];
if(ii~=[])
  dv3(1,ii) = 3*%pi/2;
end


// Compute True anomalies of maneuvers :
anv1 = zeros(ai);
anv2 = zeros(ai);
anv3 = zeros(ai);

ii = find(ai < rt);
if(ii~=[])
  anv2(ii) = %pi;
end
ii = find(rt < af);
if(ii~=[])
  anv3(ii) = %pi;
end

// Total DV :
delta_v = dv1(3,:) + dv2(3,:) + dv3(3,:);

endfunction
