//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [ires] = CL_intervInv(i1,i2)
// Difference of sets of intervals
//
// Calling Sequence
// [ires] = CL_intervInv(i1,i2)
//
// Description
// <itemizedlist><listitem>
// This function computes the difference of sets of intervals.
// <para> The intervals are gathered in two sets (i1 and i2). </para>
// <para> A real number x belongs to the difference of i1 and i2 if it belongs to an interval from i1, and does not belong to any interval from i2.  </para>
// <inlinemediaobject><imageobject><imagedata fileref="invInterval.gif"/></imageobject></inlinemediaobject>
// </listitem>
// <listitem>
// Notes: 
// <para> - The intervals in each set should have empty intersections (use CL_intervUnion if needed). </para>
// <para> - Sets of intervals may be empty. CL_intervInv(i1,[]) = i1. CL_intervInv([],i2) = []. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// i1: Set of intervals [start ; end] (2xN)
// i2: Set of intervals [start ; end] (2xN2)
// ires: Difference of i1 and i2 (i1 minus i2) (2xM)
//
// See also
// CL_intervInters
// CL_intervUnion
//
// Authors
// CNES - DCT/SB
//
// Examples
// i1=[ 0 ; 20 ];
// i2=[ [2;4] , [5.5;5.7] , [6.6;15]];
// ires = CL_intervInv(i1,i2);

// Declarations:


// Code:

if(i1 ~= [])
  if(size(i1,1)~=2)
    CL__error('Interval i1 must be of size 2xN');
  end
else
  CL__error('Interval i1 must not be []');
end

if(i2 ~= [])
  if(size(i2,1)~=2)
    CL__error('Interval i2 must be of size 2xN');
  end
else
  ires = i1;
  return;
end

isort = CL_matSort(i2,1,'r');
N = size(isort,2);

// "Inversion" of intervals :
ires = [ [i1(1) ;isort(1,1)], [isort(2,1:$-1) ; isort(1,2:$)] , [isort(2,$) ; i1(2) ] ] ;


endfunction
