//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [output]=CL_man_thrustDuration(type_output,par1,par2,par3,g0)
// Thrust duration as function of mass consumed
//
// Calling Sequence
// [output]=CL_man_thrustDuration(type_output,par1,par2,par3 [,g0])
// [dt]=CL_man_thrustDuration('dt',dm,F,isp,[g0])
// [dm]=CL_man_thrustDuration('dm',dt,F,isp,[g0])
// [F]=CL_man_thrustDuration('F',dt,dm,isp,[g0])
// [isp]=CL_man_thrustDuration('isp',dt,dm,F,[g0])
//
// Description
// <itemizedlist><listitem>
// The thrust duration can be computed as a function of mass consumed, thrust and propellant
// specific impulse by the following equation:   
// <para><inlinemediaobject><imageobject><imagedata fileref="propulsion.gif"/></imageobject></inlinemediaobject></para>
// <para> Given 3 of the 4 variables that appear in the equation (thrust duration, 
// mass consumed, thrust value and Isp), the function computes the 4th one. </para> 
// <para> The output argument <emphasis role="bold">type_output</emphasis> defines the parameter to be computed. It can be among the following: </para>
// <para> - <emphasis role="bold">'dt'</emphasis> : thrust duration.</para>
// <para> - <emphasis role="bold">'dm'</emphasis> : mass consumed</para>
// <para> - <emphasis role="bold">'F'</emphasis> : thrust value.</para>
// <para> - <emphasis role="bold">'isp'</emphasis> : propellant specific impulse. </para>
// <para> Notes: </para>
// <para> - Input arguments are always in the same order: dt, dm, F, isp </para>
// <para> - The flow rate (mass consumed per unit of time) can be computed by : flow = dm./dt</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// type_output: String defining the parameter to be computed. It can be 'dt,'dm','F' or 'isp'. (1x1)
// par1: First input parameter: dt, dm, F or isp [s kg N or s] (1xN)
// par2: Second input parameter: dm, F or isp [kg N or s] (1xN)
// par3: Third input parameter: F or isp [N or s] (1xN)
// g0: (optional) value of gravity acceleration. [m/s^2] (default value is %CL_g0) (1x1)
// output: Value of computed parameter: dt, dm, F or isp [s kg N or s] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Equations 11.10, 11.12, 11.22, 11.30
// 2 Mecanique spatiale, CNES - Cepadues 1995, Tome I, section 4.8.6
//
// See also
// CL_man_consumption
//
// Examples
// // mass consumed in kg
// isp = 220; // seconds
// dv = 1; // m/s
// m = 180; // kg
// dm = CL_man_consumption('dm',dv,isp,m); 
//
// // thrust duration in seconds
// F = 1 ; // N
// [dt] = CL_man_thrustDuration('dt',isp,F,dm) 

// Declarations:
global %CL_g0;

// Code:

Npar1 = size(par1,2)
Npar2 = size(par2,2)
Npar3 = size(par3,2)
N = max(Npar1,Npar2,Npar3)
coherence = (Npar1==N|Npar1==1) & (Npar2==N|Npar2==1) & (Npar2==N|Npar2==1)
if ~coherence then CL__error('bad size of input arguments'); end
if N~=1
  if Npar1==1 then par1=par1*ones(1,N); end
  if Npar2==1 then par2=par2*ones(1,N); end
  if Npar3==1 then par3=par3*ones(1,N); end
end

if ~exists('g0','local') then g0=%CL_g0; end

select type_output
  case 'dt'
    dm = par1;
    F = par2;
    isp = par3;
    dt = g0.*isp.*dm./F;
    output=dt;
  case 'dm'
    dt = par1;
    F = par2;
    isp = par3;
    dm = F.*dt./(g0.*isp);
    output=dm;
  case 'F'
    dt = par1;
    dm = par2;
    isp = par3;
    F = g0.*isp.*dm./dt;
    output=F;
  case 'isp'
    dt = par1;
    dm = par2;
    F = par3;
    isp = F.*dt./(g0.*dm);
    output=isp
  else
    CL__error('unknown input parameter');
end

endfunction
