//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [kep_t2] = CL_ex_secularJ2(t1, kep_t1, t2, er, mu, j2)
// Secular J2 orbit propagation
//
// Calling Sequence
// [kep_t2] = CL_ex_secularJ2(t1,kep_t1,t2 [,er,mu,j2])
//
// Description
// <itemizedlist><listitem>
// Propagation of Keplerian orbital elements kep_t1 to the time t2 including the secular J2 effects only.
// <para>The secular effects of J2 consist in drifts on periapsis argument, right ascension of ascending node and mean anomaly. </para>
// <para> See <link linkend="CL_op_driftJ2">CL_op_driftJ2</link> for more details.</para>
// <para> There can either be several sets of orbital elements and one final time or one set of orbital elements 
// and multiple final times. </para>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="kep_par.gif"/></imageobject></inlinemediaobject></para></listitem>
// </itemizedlist>
//
// Parameters
// t1: Initial time [days] (1xP)
// kep_t1: Keplerian orbital elements [sma;ecc;inc;pom;raan;anm] (6xP). P can be > 1 if there is only one final time (i.e. if N==1).
// t2: Final time [days] (1xN). N can be > 1 if there is only one set of orbital elements (i.e. if P==1). 
// kep_t2: keplerian orbital elements at t2 [sma;ecc;inc;pom;raan;anm] (6 x max(P,N))
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// j2: (optional) (Unnormalized) zonal coefficient (second zonal harmonic) (default is %CL_j2)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_ex_kepler
// CL_ex_lyddane
// CL_ex_eckHech
// CL_op_driftJ2
//
// Examples
// // propagation of one satellite to several times:
// dga = 7070 * 1000;
// ecc = 0.001;
// inc = CL_deg2rad(98);
// pom = CL_deg2rad(90);
// gom = 0;
// anm = 0;
// t1 = 21915;
// t2 = t1:0.1:t1+1;
// kep_t1=[dga;ecc;inc;pom;gom;anm];
// [kep_t2] = CL_ex_secularJ2(t1, kep_t1, t2);
//
// // propagation of 2 element sets to one final time:
// t2 = t1+1;
// t1=[t1,t1+0.5];
// kep_t1=[kep_t1,[dga+100*1000;ecc+0.1;inc;pom;gom;anm]];
// [kep_t2] = CL_ex_secularJ2(t1, kep_t1, t2);

// Gestion arguments optionnels

// Declarations:
global %CL_eqRad %CL_j2 %CL_mu;

// Code:

if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('j2','local') then j2=%CL_j2; end

// Coherence des entrees :
// t1
coherence = (size(t1,1)==1)
if ~coherence then CL__error('t1 must be a vector (1 x n)'); end
// kep_t1
coherence = (size(kep_t1,1)==6)
if ~coherence then CL__error('kep_t1 must be a vector (6 x n)'); end
// t2
coherence = (size(t2,1)==1)
if ~coherence then CL__error('t2 must be a vector (1 x m)'); end
// t1 et kep_t1
coherence = (size(t1,2) == size(kep_t1,2));
if ~coherence then CL__error('if t1 is a vector (1 x n) then kep_t1 must be a vector (6 x n)'); end
// t2 et kep_t1
coherence = (size(t2,2)~=1) & (size(kep_t1,2) ~=1);
if coherence then CL__error('You cant have t2 a vector (1 x m) and kep_t1 a vector (6 x n) at the same time'); end


dim = max(length(t2), size(kep_t1,'c'));
kep_t2 = zeros(6,dim);

// derive J2 (attention : ordre des parametres = a, e, i)
[dpom, dgom, danm] = CL_op_driftJ2(kep_t1(1,:),kep_t1(2,:),kep_t1(3,:),er, mu, j2);

// nouveaux parametres kepleriens
kep_t2(1,:) = kep_t1(1,:) ;
kep_t2(2,:) = kep_t1(2,:) ;
kep_t2(3,:) = kep_t1(3,:) ;
kep_t2(4,:) = kep_t1(4,:) + dpom .* (t2-t1) * 86400;
kep_t2(5,:) = kep_t1(5,:) + dgom .* (t2-t1) * 86400;
kep_t2(6,:) = CL_rMod(kep_t1(6,:) + danm .* (t2-t1) * 86400,2*%pi);

endfunction
