#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from zope.container.interfaces import IContainer, IContained
from zope.container.constraints import contains, containers
from zope.interface import Interface, implements
from zope.schema.interfaces import IList, IBool
from zope.schema import TextLine, Text, Bool, List, Date, Datetime

from schooltool.intervention import InterventionGettext as _


class IInterventionRootMarker(Interface):
    """A marker interface for the root intervention objects."""


class IInterventionStudentMarker(Interface):
    """A marker interface for the student intervention objects."""


class IInterventionMarker(Interface):
    """A marker interface for the actual intervention objects. 
       Presently, these include messages and goals."""


class IInterventionTabProxy(Interface):
    """An interface for getting the displayable value of a message or goal."""


class IStudentSchoolYearsProxy(Interface):
    """A marker interface for the student schoolyear proxy."""


class IStudentSchoolYearProxy(Interface):
    """A marker interface for the student schoolyear proxy."""


class IPersonListField(IList):
    """A field that represents a list of people."""


class PersonListField(List):
    """A field that represents a list of people."""

    implements(IPersonListField)


class IGoalMetField(IBool):
    """A field that represents the goal met boolean."""


class GoalMetField(Bool):
    """A field that represents the goal met boolean."""

    implements(IGoalMetField)


class IInterventionRoot(IContainer):
    """Container of IInterventionSchoolYear objects."""

    contains('schooltool.intervention.interfaces.IInterventionSchoolYear')


class IInterventionSchoolYear(IContainer):
    """Container of the IInteventionStudent objects."""

    contains('schooltool.intervention.interfaces.IInterventionStudent')
    containers(IInterventionRoot)


class IInterventionStudent(IContainer):
    """Container of the student's intervention containers."""

    contains('zope.app.IContainer')
    containers(IInterventionSchoolYear)


class IInterventionMessages(IContainer):
    """Container of intervention messages."""

    contains('schooltool.intervention.interfaces.IInterventionMessage')
    containers(IInterventionStudent)


class IInterventionMessage(IContained):
    """Intervention message about a given student."""

    sender = List(
        title=u"List of contacts (relationship) for the sender(s)"
        )

    recipients = PersonListField(
        title=_("Recipients"),
        )

    subject = TextLine(
        title=_("Subject"),
        required=False,
        )

    body = Text(
        title=_("Message body")
        )

    status_change = Bool(
        title=u"Status Change Message",
        required=False
        )

    created = Datetime(
        title=u"Creation Date/Time",
        )

    containers(IInterventionMessages)


class IInterventionGoals(IContainer):
    """Container of the studnet's interventions goals."""

    contains('schooltool.intervention.interfaces.IInterventionGoal')
    containers(IInterventionStudent)


class IInterventionGoal(IContained):
    """intervention goal for a given student."""

    presenting_concerns = Text(
        title=_("Presenting concerns"),
        required=False,
        )

    goal = Text(
        title=_("Goal"),
        )

    strengths = Text(
        title=_("Strengths"),
        required=False,
        )

    indicators = Text(
        title=_("Indicators"),
        required=False,
        )

    intervention = Text(
        title=_("Intervention"),
        required=False,
        )

    timeline = Date(
        title=_("Timeline"),
        )

    persons_responsible = PersonListField(
        title=_("Persons responsible"),
        description=_(u"The selected people will receive messages regarding"
                        " this goal through their interventions inbox and"
                        " email (if selected below)."),
        )

    at_one_time_responsible = List(
        title=_("Persons who have been at one time responsible"),
        )

    goal_met = GoalMetField(
        title=u"Goal met",
        required=False,
        default=False,
        )

    follow_up_notes = Text(
        title=_("Follow up notes"),
        required=False
        )

    notified = Bool(
        title=u"Notification sent to persons responsible"
        )

    creator = List(
        title=u"List of contacts (relationship) for the creator(s)"
        )

    created = Datetime(
        title=u"Creation Date/Time",
        )

    containers(IInterventionGoals)


class IInterventionPersonsResponsible(Interface):
    """An interface for getting the persons_responsible of a message or goal."""


class IInterventionSchoolYearId(Interface):
    """An interface for getting the schoolYearId of a message or goal."""


class IInterventionType(Interface):
    """An interface for getting the intervention type as a string."""


class IInterventionGetter(Interface):
    """An interface for getting the displayable value of a message or goal."""


class ISectionMessagesProxy(Interface):
    """An interface for adapting the SectionMessagesProxy."""


class IStudentMessagesGoalsProxy(Interface):
    """An interface for adapting the StudentMessagesGoalsProxy."""


class IIndependentActivities(Interface):
    """Bridge interface to remove intervention dependency on gradebook."""

