#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from zope.authentication.interfaces import IUnauthenticatedPrincipal
from zope.component import getUtility
from zope.intid.interfaces import IIntIds
from zope.security.proxy import removeSecurityProxy

from schooltool.app.membership import URIGroup
from schooltool.contact.interfaces import IContact
from schooltool.course.interfaces import ISection
from schooltool.person.interfaces import IPerson
from schooltool.relationship.relationship import getRelatedObjects
from schooltool.securitypolicy import crowds

from schooltool.intervention import InterventionGettext as _
import interfaces


class InterventionInstructorsCrowd(crowds.Crowd):
    """Crowd of instructors of the student indicated by the given 
       intervention object."""

    title = _(u'Instructors')
    description = _(u'Instructors of a student in any of his sections.')

    def _getSections(self, ob):
        return [section for section in getRelatedObjects(ob, URIGroup)
                if ISection.providedBy(section)]

    def contains(self, principal):
        if IUnauthenticatedPrincipal.providedBy(principal):
            return False
        teacher = IPerson(principal)
        student = removeSecurityProxy(IPerson(self.context))
        for section in self._getSections(student):
            if teacher in section.instructors:
                return True
        return False


class InterventionAdvisorsCrowd(crowds.Crowd):
    """Crowd of advisors of the student indicated by the given 
       intervention object."""

    title = _(u'Advisors')
    description = _(u'Advisors of a student.')

    def contains(self, principal):
        if IUnauthenticatedPrincipal.providedBy(principal):
            return False
        teacher = IPerson(principal)
        student = removeSecurityProxy(IPerson(self.context))
        return teacher in student.advisors


class BaseResponsibleCrowd(crowds.Crowd):
    """Crowd of any user who is on the list of persons responsible for the
       message or goal."""

    def contains(self, principal):
        if IUnauthenticatedPrincipal.providedBy(principal):
            return False
        if not interfaces.IInterventionMarker.providedBy(self.context):
            return False
        responsible = interfaces.IInterventionPersonsResponsible(self.context)
        contact_id = getUtility(IIntIds).getId(IContact(IPerson(principal)))
        return contact_id in responsible


class StaffResponsibleCrowd(BaseResponsibleCrowd):
    """Crowd of school staff who are on the list of persons responsible for the
       message or goal."""

    title = _(u'Staff responsible')
    description = _(u'Staff members responsible for the message or goal.')

    def contains(self, principal):
        if not super(StaffResponsibleCrowd, self).contains(principal):
            return False
        student = removeSecurityProxy(IPerson(self.context))
        return IPerson(principal).username != student.username


class StudentResponsibleCrowd(BaseResponsibleCrowd):
    """Crowd containing the student who is in the list of persons responsible
       for the message or goal."""

    title = _(u'Students responsible')
    description = _(u'Students responsible for the message or goal.')

    def contains(self, principal):
        if not super(StudentResponsibleCrowd, self).contains(principal):
            return False
        student = removeSecurityProxy(IPerson(self.context))
        return IPerson(principal).username == student.username

