#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
"""
Unit tests for the intervention catalog.
"""
import unittest
import doctest
from datetime import datetime

from zope.app.testing.placelesssetup import setUp, tearDown
from zope.component import provideAdapter
from zope.interface import implements

from schooltool.intervention import interfaces


def doctest_catalogSetUp():
    """
    The catalog setup will create indeces message and goal objects.  First
    we need to set up some stubs.

        >>> class StudentStub:
        ...     def __init__(self, first_name, last_name):
        ...         self.first_name = first_name
        ...         self.last_name = last_name

        >>> class MessageStub:
        ...     implements(interfaces.IInterventionMessage)
        ...     student = StudentStub('John', 'Deere')
        ...     recipients = ['teacher2', 'advisor1']
        ...     created = datetime(2009, 9, 9)

        >>> class GoalStub:
        ...     implements(interfaces.IInterventionGoal)
        ...     student = StudentStub('Jane', 'Doe')
        ...     persons_responsible = ['teacher1', 'advisor2']
        ...     created = datetime(2010, 1, 1)

        >>> class SchoolYearStub:
        ...     __name__ = '2009-2010'

    We'll call catalogSetUp passing an empty dict.

        >>> from schooltool.intervention.catalog import catalogSetUp
        >>> catalog = {}
        >>> catalogSetUp(catalog)

    Tests for the 'first_name' index:

        >>> catalog['first_name'].index_doc(11, MessageStub())
        >>> catalog['first_name'].index_doc(22, GoalStub())
        >>> sorted(catalog['first_name'].documents_to_values.items())
        [(11, 'John'), (22, 'Jane')]

    Tests for the 'last_name' index:

        >>> catalog['last_name'].index_doc(11, MessageStub())
        >>> catalog['last_name'].index_doc(22, GoalStub())
        >>> sorted(catalog['last_name'].documents_to_values.items())
        [(11, 'Deere'), (22, 'Doe')]

    Tests for the 'persons_responsible' index:

        >>> from schooltool.intervention.catalog import (
        ...     getMessagePersonsResponsible, getGoalPersonsResponsible)
        >>> provideAdapter(getMessagePersonsResponsible,
        ...     adapts=[interfaces.IInterventionMessage],
        ...     provides=interfaces.IInterventionPersonsResponsible)
        >>> provideAdapter(getGoalPersonsResponsible,
        ...     adapts=[interfaces.IInterventionGoal],
        ...     provides=interfaces.IInterventionPersonsResponsible)
        >>> catalog['persons_responsible'].index_doc(11, MessageStub())
        >>> catalog['persons_responsible'].index_doc(22, GoalStub())
        >>> sorted(catalog['persons_responsible'].documents_to_values.items())
        [(11, ['teacher2', 'advisor1']), (22, ['teacher1', 'advisor2'])]

    Tests for the 'created' index:

        >>> catalog['created'].index_doc(11, MessageStub())
        >>> catalog['created'].index_doc(22, GoalStub())
        >>> sorted(catalog['created'].documents_to_values.items())
        [(11, datetime.datetime(2009, 9, 9, 0, 0)),
         (22, datetime.datetime(2010, 1, 1, 0, 0))]

    Tests for the 'schoolYearId' index:

        >>> provideAdapter(lambda x: SchoolYearStub(),
        ...     adapts=[interfaces.IInterventionMessage],
        ...     provides=interfaces.IInterventionSchoolYear)
        >>> provideAdapter(lambda x: SchoolYearStub(),
        ...     adapts=[interfaces.IInterventionGoal],
        ...     provides=interfaces.IInterventionSchoolYear)
        >>> catalog['schoolYearId'].index_doc(11, MessageStub())
        >>> catalog['schoolYearId'].index_doc(22, GoalStub())
        >>> sorted(catalog['schoolYearId'].documents_to_values.items())
        [(11, '2009-2010'), (22, '2009-2010')]

    Tests for the 'intervention_type' index:

        >>> from schooltool.intervention.catalog import (
        ...     getMessageType, getGoalType)
        >>> provideAdapter(getMessageType,
        ...     adapts=[interfaces.IInterventionMessage],
        ...     provides=interfaces.IInterventionType)
        >>> provideAdapter(getGoalType,
        ...     adapts=[interfaces.IInterventionGoal],
        ...     provides=interfaces.IInterventionType)
        >>> catalog['intervention_type'].index_doc(11, MessageStub())
        >>> catalog['intervention_type'].index_doc(22, GoalStub())
        >>> sorted(catalog['intervention_type'].documents_to_values.items())
        [(11, 'message'), (22, 'goal')]
    """


def test_suite():
    optionflags = doctest.NORMALIZE_WHITESPACE | doctest.ELLIPSIS
    suite = doctest.DocTestSuite(optionflags=optionflags,
                                 setUp=setUp, tearDown=tearDown)
    return unittest.TestSuite([suite])


if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')

