/*
 *                            COPYRIGHT
 *
 *  cschem - modular/flexible schematics editor - sch-rnd (executable)
 *  Copyright (C) 2022 Tibor 'Igor2' Palinkas
 *
 *  (Supported by NLnet NGI0 PET Fund in 2022)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */

#include <libcschem/config.h>

#include <genht/hash.h>

#include <librnd/core/conf.h>
#include <librnd/core/conf_multi_temp.h>
#include <librnd/core/compat_fs.h>
#include <librnd/core/hid.h>
#include <librnd/core/event.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/tool.h>

#include <libcschem/concrete.h>
#include <libcschem/project.h>
#include <libcschem/event.h>

#include "conf_core.h"
#include "sheet.h"
#include "project.h"

#include "multi.h"

static gdl_list_t sheets;
htsp_t sch_rnd_projects;

static void sch_rnd_multi_conf_save(csch_sheet_t *s, int clear_glob)
{
	if (s->saved_rnd_conf == NULL) {
		s->saved_rnd_conf = rnd_conf_state_alloc();
		assert(s->saved_rnd_conf != NULL);
	}

	if (s->saved_sch_conf == NULL) {
		s->saved_sch_conf = calloc(sizeof(conf_core), 1);
		assert(s->saved_sch_conf != NULL);
	}

	rnd_conf_state_save(s->saved_rnd_conf);
	memcpy(s->saved_sch_conf, &conf_core, sizeof(conf_core));
	if (clear_glob)
		memset(&conf_core, 0, sizeof(conf_core));
}


static void sch_rnd_multi_conf_load(csch_sheet_t *s)
{
	assert(s->saved_rnd_conf != NULL);
	assert(s->saved_sch_conf != NULL);

	rnd_conf_state_load(s->saved_rnd_conf);
	memcpy(&conf_core, s->saved_sch_conf, sizeof(conf_core));
	memset(s->saved_sch_conf, 0, sizeof(conf_core));
}


extern rnd_hidlib_t *csch_hidlib;
extern csch_project_t *prj;

static void switced_to(csch_sheet_t *s)
{
	if ((rnd_gui != NULL) && (rnd_gui->set_hidlib != NULL))
		rnd_gui->set_hidlib(rnd_gui, &s->hidlib);
	if ((rnd_render != NULL) && (rnd_render->set_hidlib != NULL) && (rnd_render != rnd_gui))
		rnd_render->set_hidlib(rnd_render, &s->hidlib);
	prj = s->parent;
	rnd_event(&s->hidlib, RND_EVENT_BOARD_CHANGED, "i", 0);
	rnd_event(&s->hidlib, RND_EVENT_BOARD_FN_CHANGED, NULL);
}

static void sch_rnd_multi_switch_to_(csch_sheet_t *s)
{
	csch_hidlib = &s->hidlib;
	sch_rnd_multi_conf_load(s);
	switced_to(s);
}


void sch_rnd_multi_switch_to(csch_sheet_t *s)
{
	csch_sheet_t *curr = (csch_sheet_t *)csch_hidlib;

	TODO("multi: librnd4.0.0 will make this obsolete and removable");
	/* switch to nothing (useful when loading a new sheet) */
	if (s == NULL) {
		sch_rnd_multi_conf_save(curr, 1);
		csch_hidlib = NULL;
		prj = NULL;
		return;
	}

	/* first switch from nothing to s */
	if (curr == NULL) {
		csch_hidlib = &s->hidlib;
		switced_to(s);
		return;
	}

	/* switching to the current is no-op */
	if (s == curr)
		return;

	sch_rnd_multi_conf_save(curr, 1);
	sch_rnd_multi_switch_to_(s);
}

void sch_rnd_multi_abort_switch_to(csch_sheet_t *s)
{
	sch_rnd_multi_switch_to_(s);
}

void sch_rnd_multi_switch_to_initial(csch_sheet_t *s)
{
	sch_rnd_multi_switch_to_(s);
}


void sch_rnd_multi_switch_to_delta(csch_sheet_t *curr, int step)
{
	if (curr == NULL) {
		if (step == 0)
			return; /* would switch to the hidlib that's already active in the GUI */
		if ((rnd_gui != NULL) && (rnd_gui->get_hidlib != NULL))
		curr = (csch_sheet_t *)rnd_gui->get_hidlib(rnd_gui);
	}
	else if (step == 0) {
		sch_rnd_multi_switch_to(curr);
		return;
	}

	if (curr == NULL)
		return; /* no known starting point */

	if (step > 0) {
		for(;step > 0; step--) {
			curr = curr->link.next;
			if (curr == NULL)
				curr = gdl_first(&sheets);
		}
	}
	else if (step < 0) {
		for(;step < 0; step++) {
			curr = curr->link.prev;
			if (curr == NULL)
				curr = gdl_last(&sheets);
		}
	}

	if (curr != NULL)
		sch_rnd_multi_switch_to(curr);
}

static csch_project_t *sch_rnd_multi_load_project_(const char *project_fn)
{
	csch_project_t *prj = htsp_get(&sch_rnd_projects, project_fn);
	if (prj == NULL) {
		char *pfn = rnd_strdup(project_fn);
		prj = csch_load_project_by_sheet_name(pfn, 0);
		htsp_set(&sch_rnd_projects, pfn, prj);
	}
	return prj;
}


csch_project_t *sch_rnd_multi_load_project(const char *load_fn, const char *project_fn_in)
{
	char *project_fn, *project_dir;
	char *freeme1, *freeme2;
	csch_project_t *prj;

	if (project_fn_in != NULL)
		return sch_rnd_multi_load_project_(project_fn_in);

	/* there may be no existing project file - assume project is the directory */
	project_dir = freeme1 = rnd_dirname_real(load_fn);
	project_fn = freeme2 = rnd_concat(project_dir, "/project.lht", NULL);

	prj = sch_rnd_multi_load_project_(project_fn);

	free(freeme1);
	free(freeme2);
	return prj;
}


static void sch_rnd_multi_load_prj_conf(csch_sheet_t *sheet)
{
	csch_project_t *prj = sheet->parent;
	if (prj != NULL)
		rnd_conf_load_as(RND_CFR_PROJECT, prj->filename, 0);
}


static csch_sheet_t *sch_rnd_multi_load_(const char *fn, const char *fmt, const char *project_fn);


/* Append all unique file names to fns (keeps order) */
static void map_all_sheets(const rnd_conflist_t *list, vts0_t *fns)
{
	rnd_conf_listitem_t *item;
	const char *shfn;
	int idx, n;

	rnd_conf_loop_list_str(list, item, shfn, idx) {
		int found = 0;

		for(n = 0; n < fns->used; n++) {
			if (strcmp(fns->array[n], shfn) == 0) {
				rnd_message(RND_MSG_ERROR, "Redundant file name in project: %s\n", shfn);
				found = 1;
				break;
			}
		}

		if (!found)
			vts0_append(fns, rnd_strdup(shfn));
	}
}

/* Load a project file from prj_fn and all sheets; return first sheet */
static csch_sheet_t *sch_rnd_multi_load_project_sheets(const char *prj_fn)
{
	csch_project_t *prj = sch_rnd_multi_load_project_(prj_fn);
	csch_sheet_t *res = NULL;
	vts0_t fns = {0};
	long n, aux_start;

	if (prj == NULL)
		return NULL;

	rnd_conf_load_project(prj_fn, NULL);
	rnd_conf_merge_all("prj/root_sheets");
	rnd_conf_merge_all("prj/aux_sheets");


	map_all_sheets(&conf_core.prj.root_sheets, &fns);
	aux_start = fns.used;
	map_all_sheets(&conf_core.prj.aux_sheets, &fns);

	if (fns.used > 0) {
		for(n = 0; n < fns.used; n++) {
			csch_sheet_t *sheet = sch_rnd_multi_load_(fns.array[n], NULL, prj_fn);
			if (sheet != NULL) {
				if (res == NULL)
					res = sheet;
				if (n >= aux_start)
					sheet->prj_non_root = 1;
			}
			free(fns.array[n]);
		}
	}
	else
		rnd_message(RND_MSG_ERROR, "Can not load project file '%s': does not name any sheet\n", prj_fn);

	vts0_uninit(&fns);

	return res;
}

static csch_sheet_t *sch_rnd_multi_load_(const char *fn, const char *fmt, const char *project_fn)
{
	csch_sheet_t *sheet, *fsh;
	csch_project_t *prj;
	const char *real_fn = fn;
	char *freeme = NULL;


	if ((project_fn == NULL) && ((fmt == NULL) || (*fmt == '\0'))) {
		const char *end = fn + strlen(fn) - 11;
		if (strcmp(end, "project.lht") == 0)
			return sch_rnd_multi_load_project_sheets(fn);
	}

	prj = sch_rnd_multi_load_project(fn, project_fn);

	TODO("multi: this assumes there's no current sheet, so loaders have to clear conf before call");

	rnd_conf_reset(RND_CFR_DESIGN, fn);

	fsh = gdl_first(&sheets);
	if (fsh != NULL)
		rnd_conf_state_init_from(fsh->saved_rnd_conf);

	rnd_conf_merge_all(NULL); /* get font settings right for the text update in the loader */

	/* If sheet name is not absolute and there is a project file being loaded,
	   sheet name needs to be translated relative to the project file's dir */
	if ((project_fn != NULL) && !rnd_is_path_abs(fn)) {
		const char *s, *sep = NULL;

		/* check if project file is not in ./; sep is the last path separator */
		for(s = project_fn; *s != '\0'; s++) {
			if ((*s == '/') || (*s == RND_DIR_SEPARATOR_C))
				sep = s;
		}

		/* project file is not in ./, change real_fn */
		if (sep != NULL) {
			gds_t tmp = {0};
			gds_append_len(&tmp, project_fn, sep - project_fn + 1); /* +1 to keep the sep in project file's path */
			gds_append_str(&tmp, real_fn);
			real_fn = freeme = tmp.array;
		}
	}

	if (csch_project_load_sheet(prj, real_fn, fmt, &sheet) != 0) {
		free(freeme);
		return NULL;
	}

	sch_rnd_multi_load_prj_conf(sheet);
	rnd_conf_merge_all(NULL); /* to get the project file applied */
	sch_rnd_prj_postproc(prj);


	gdl_append(&sheets, sheet, link);

	rnd_tool_select_by_name(&sheet->hidlib, "arrow");

	rnd_event(&sheet->hidlib, CSCH_EVENT_SHEET_POSTLOAD, NULL);
	rnd_event(&sheet->hidlib, RND_EVENT_LOAD_POST, NULL);

	TODO("multi: this assumes there's no current sheet, so loaders have to clear conf before call");
	sch_rnd_multi_conf_save(sheet, 1);

	free(freeme);
	return sheet;
}

csch_sheet_t *sch_rnd_multi_load(const char *fn, const char *fmt)
{
	return sch_rnd_multi_load_(fn, fmt, NULL);
}

csch_sheet_t *sch_rnd_multi_new_empty(void)
{
	csch_sheet_t *sheet, *fsh;
	csch_project_t *prj = csch_project_alloc();

	htsp_set(&sch_rnd_projects, rnd_strdup("<none>"), prj);

	fsh = gdl_first(&sheets);
	if (fsh != NULL)
		rnd_conf_state_init_from(fsh->saved_rnd_conf);

	sheet = sch_rnd_sheet_new(prj);
	if (sheet != NULL) {
		sch_rnd_multi_load_prj_conf(sheet);
		sch_rnd_prj_postproc(prj);
		gdl_append(&sheets, sheet, link);
	}
	else {
		TODO("multi: memleak");
/*		sch_rnd_prj_free(prj);*/
	}

	rnd_conf_merge_all(NULL); /* to get the project file applied */

	TODO("multi: this assumes there's no current sheet, so loaders have to clear conf before call");
	sch_rnd_multi_conf_save(sheet, 1);

	return sheet;
}

void sch_rnd_multi_unload(csch_sheet_t *sheet)
{
	csch_project_t *prj = sheet->parent;

	gdl_remove(&sheets, sheet, link);
	if (prj != NULL) {
		long n;
		for(n = 0; n < prj->sheets.used; n++) {
			if (prj->sheets.array[n] == sheet) {
				vtp0_remove(&prj->sheets, n, 1);
				n--;
			}
		}
	}
	csch_sheet_uninit(sheet);
}

csch_sheet_t *sch_rnd_multi_neighbour_sheet(csch_sheet_t *sheet)
{
	if (sheet->link.next != NULL)
		return sheet->link.next;
	return sheet->link.prev;
}

void sch_rnd_multi_init(void)
{
	htsp_init(&sch_rnd_projects, strhash, strkeyeq);
}

void sch_rnd_multi_uninit(void)
{
	htsp_uninit(&sch_rnd_projects);
}
