/*
 *                            COPYRIGHT
 *
 *  cschem - modular/flexible schematics editor - core library
 *  Copyright (C) 2022 Tibor 'Igor2' Palinkas
 *
 *  (Supported by NLnet NGI0 PET Fund in 2022)
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.*
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */

#include "config.h"

#include <genvector/gds_char.h>
#include <librnd/core/compat_misc.h>
#include "concrete.h"
#include "compile.h"

#include "util_export.h"

char *cschem_export_filename(rnd_hidlib_t *hl, const char *explicit, const char *ext)
{
	csch_sheet_t *sheet = (csch_sheet_t *)hl;
	gds_t tmp;

	if (explicit != NULL)
		return rnd_strdup(explicit);


	gds_init(&tmp);

	if (hl->filename != NULL) {
		int n;
		gds_append_str(&tmp, sheet->loadname);

		/* truncate "extension" */
		for(n = tmp.used-1; n > 0; n--) {
			if (tmp.array[n] == '.') {
				tmp.used = n;
				break;
			}
		}
	}
	else
		gds_append_str(&tmp, "unknown");

	gds_append_str(&tmp, ext);

	return tmp.array; /* tmp is not uninited because ownership is passed back to the caller */
}

static char nope[] = "no abstract";

int cschem_export_compile_pre(csch_project_t *prj, const char *view_name, void **cookie)
{
	int view_id = CSCH_VIEW_DEFAULT;

	*cookie = prj->abst;

	if ((view_name == NULL) || (*view_name == '\0')) {
		if (prj->abst != NULL) {
			*cookie = nope; /* do not attempt to free prj->abst on _post */
			return 0; /* already compiled and no different view requested */
		}
	}
	else { /* view requested */
		view_id = csch_view_get_id(prj, view_name);
		if (view_id < 0) {
			rnd_message(RND_MSG_ERROR, "Invalid view name: '%s'\n", view_name);
			goto error;
		}
	}

	/* old one saved in cookie */
	prj->abst = calloc(sizeof(csch_abstract_t), 1);
	csch_abstract_init(prj->abst);
	if (csch_compile_project(prj, view_id, prj->abst, 0) != 0) {
		csch_abstract_uninit(prj->abst);
		prj->abst = *cookie;
		rnd_message(RND_MSG_ERROR, "Failed to compile view: '%s'\n", view_name);
		goto error;
	}

	return 0;

	error:;
	*cookie = nope; /* do not attempt to free prj->abst on _post */
	return -1;
}

void cschem_export_compile_post(csch_project_t *prj, void **cookie)
{
	if (*cookie != nope) {
		/* remove abstract model compiled for the export, restore original */
		csch_abstract_uninit(prj->abst);
		prj->abst = *cookie;
	}

	*cookie = NULL;
}
