/*
 *                            COPYRIGHT
 *
 *  cschem - modular/flexible schematics editor - libcschem (core library)
 *  Copyright (C) 2020,2022 Tibor 'Igor2' Palinkas
 *
 *  (Supported by NLnet NGI0 PET Fund in 2022)
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.*
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */

#include "config.h"

#include "plug_io.h"

#include "export.h"

void csch_sheet_export_name(csch_sheet_t *sheet, char epath[CSCH_PATH_MAX], const char *exp_fmt, int sheet_idx)
{
	if (sheet->loadname != NULL) {
		char *ext;
		if (strlen(sheet->loadname) >= CSCH_PATH_MAX) {
			sprintf(epath, "long_named_sheet_%d*", sheet_idx);
			return;
		}
		strcpy(epath, sheet->loadname);
		ext = strrchr(epath, '.');
		if ((strcmp(ext, ".rs") == 0) || (strcmp(ext, ".lht") == 0) || (strcmp(ext, ".csch") == 0))
			strcpy(ext, "*");
	}
	else
		sprintf(epath, "anon_sheet_%d*", sheet_idx);
}

void csch_project_export_name(csch_project_t *prj, csch_sheet_t *sheet, char epath[CSCH_PATH_MAX], const char *exp_fmt, const char *view_name)
{

	/* automatic view name */
	if (view_name == NULL) {
		void **v = vtp0_get(&prj->views, prj->curr, 0);

		if ((v != NULL) && (*v != NULL)) {
			csch_view_t *view = *v;
			view_name = view->fgw_ctx.name;
		}
		else
			view_name = "no_view";
	}

	if (prj->loadname != NULL) {
		char *end;
		if (strlen(prj->loadname) + strlen(view_name) >= CSCH_PATH_MAX)
			goto fallback;
		strcpy(epath, prj->loadname);
		end = strrchr(epath, '/');
		if (strcmp(end, "/project.lht") == 0)
			sprintf(end, "/netlist.%s*", view_name);
	}
	else {
		fallback:;
		if (sheet != NULL) {
			if (sheet->loadname != NULL) {
				long len = strlen(sheet->loadname);
				char *s, *end;

				memcpy(epath, sheet->loadname, len);
				for(s = end = epath + len; s > sheet->loadname+1; s--) {
					if (*s == '.') {
						end = s;
						break;
					}
				}
				*end = '.';
				end++;
				len = strlen(view_name);
				memcpy(end, view_name, len);
				end += len;
				end[0] = '*';
				end[1] = '\0';
			}
			else
				goto fallback2;
		}
		else {
			fallback2:;
			sprintf(epath, "netlist.%s*", view_name);
		}
	}
}

void csch_project_export_sheets(csch_project_t *prj, const char *exp_fmt)
{
	char epath[CSCH_PATH_MAX];
	int n;
	for(n = 0; n < vtp0_len(&prj->sheets); n++) {
		csch_sheet_t *sheet = prj->sheets.array[n];
		csch_sheet_export_name(sheet, epath, exp_fmt, n);
		csch_export_sheet(sheet, epath, exp_fmt);
	}
}
