/****************************************************************************
**  CUBE        http://www.score-p.org/                                    **
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/
/**
 * \file metric.h
   \brief Declares  types and functions to deal with metrics.
 */
#ifndef CUBEW_METRIC_H
#define CUBEW_METRIC_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdint.h>
#include <sys/types.h>
#include "cubew_types.h"
#include "cubew_cnode.h"
#include "cubew_report_layouts_types.h"

#if defined( BACKEND_CUBE_COMPRESSED ) || defined( FRONTEND_CUBE_COMPRESSED )
#include "zlib.h"
#endif /* BACKEND_CUBE_COMPRESSED */


/** defines a marker at the beginning of the file */
#define CUBE_INDEXFILE_MARKER "CUBEX.INDEX"
#define CUBE_INDEXFILE_MARKER_SIZE 11

/** defines a marker at the beginning of the file */
#define CUBE_DATAFILE_MARKER "CUBEX.DATA"
#define CUBE_DATAFILE_MARKER_SIZE 10

#define CUBE_DATAFILE_COMPRESSED_MARKER        "ZCUBEX.DATA"
#define CUBE_DATAFILE_COMPRESSED_MARKER_SIZE    11

typedef struct marray cube_marray;   /**< Synonim of an array for using it as an array of metrics. */

/**
 * A Structure of a metric.
 */

enum CubeMetricType { CUBE_METRIC_INCLUSIVE = 0, CUBE_METRIC_EXCLUSIVE = 1, CUBE_METRIC_SIMPLE = 2, CUBE_METRIC_POSTDERIVED = 3, CUBE_METRIC_PREDERIVED_INCLUSIVE = 4, CUBE_METRIC_PREDERIVED_EXCLUSIVE = 5 };

enum  IndexFormat { CUBE_INDEX_FORMAT_NONE, CUBE_INDEX_FORMAT_SPARSE, CUBE_INDEX_FORMAT_BITVECTOR, CUBE_INDEX_FORMAT_DENSE };

enum  DataType
{
    CUBE_DATA_TYPE_UNKNOWN,
    CUBE_DATA_TYPE_DOUBLE,
    CUBE_DATA_TYPE_UINT8,
    CUBE_DATA_TYPE_INT8,
    CUBE_DATA_TYPE_UINT16,
    CUBE_DATA_TYPE_INT16,
    CUBE_DATA_TYPE_UINT32,
    CUBE_DATA_TYPE_INT32,
    CUBE_DATA_TYPE_UINT64,
    CUBE_DATA_TYPE_INT64,
    CUBE_DATA_TYPE_TAU_ATOMIC,
    CUBE_DATA_TYPE_COMPLEX,
    CUBE_DATA_TYPE_RATE,
    CUBE_DATA_TYPE_MIN_DOUBLE,
    CUBE_DATA_TYPE_MAX_DOUBLE,
    CUBE_DATA_TYPE_SCALE_FUNC
};




typedef union
{
    struct
    {
        uint32_t endian;
        uint16_t version;
        /**         uint32_t nprcs;
           //         uint32_t nthrds;
           //         uint32_t ncnds;
           //         uint8_t  data_type; */
        uint8_t metric_format;
    } __attribute__ ( ( __packed__ ) ) named;
    char as_array[ 1 ];
} metric_header;


typedef struct cube_metric
{
    char*               disp_name;
    char*               uniq_name;
    char*               dtype;
    char*               uom;
    char*               val;
    char*               url;
    char*               descr;
    int                 id;
    cube_marray*        child;
    struct cube_metric* parent;

/*
    advanced part
 */
    uint32_t              ncn;
    uint32_t              nthrd;
    enum DataType         data_type; /*  proper datatype  */
    enum IndexFormat      metric_format;
    enum CubeMetricType   metric_type;
    FILE*                 data_file;
    char*                 known_cnodes;          /* bitmask of cnodes */
    char*                 written_cnodes;
    signed long long      start_pos_of_datafile; /**< used to calculate size of datafile */
    enum bool_t           compressed;
    enum bool_t           im_writing;            /**< marks, that last writing call was done for this metric. Needed to write TARed layout of CUBE */
    enum bool_t           im_finished;           /**< marks, that the files with the data were closed */
    report_layout_writer* layout;                /** Saves the structure controling the organisation of the report file layout */
    uint64_t*             sub_index;
    uint64_t              subi_size;
    uint64_t              icompressed;
    uint64_t              n_compressed;
    uint64_t              incr_compressed;
#if defined( BACKEND_CUBE_COMPRESSED ) || defined( FRONTEND_CUBE_COMPRESSED )
    Bytef*                compression_buffer;
#endif /* BACKEND_CUBE_COMPRESSED */
    off_t                 last_seek_position;
    cube_dyn_array*       local_cnode_enumeration;
    cube_dyn_array*       optimal_cnodes_sequence;
    carray*               optimal_cnodes_sequence_with_cndptrs;
    char*                 expression;                   /** only derivated metrics do carry here the expression for the calculation. Another metrics ignore this field. */
    char*                 init_expression;              /** only derivated metrics do carry here the initiallization expression for the calculation. Another metrics ignore this field. */
} cube_metric;







cube_metric*
cube_metric_create( cube_metric* metric );
void
cube_metric_init( cube_metric*          metric,
                  const char*           disp_name,
                  const char*           uniq_name,
                  const char*           dtype,
                  const char*           uom,
                  const char*           val,
                  const char*           url,
                  const char*           descr,
                  cube_metric*          parent,
                  report_layout_writer* layout,
                  enum CubeMetricType   metric_type,
                  enum bool_t           compressed );
void
cube_metric_construct_child( cube_metric* metric );
void
cube_metric_free( cube_metric* metric );
void
cube_metric_finish( cube_metric* metric,
                    int          wrte_index );


void
cube_metric_set_expression( cube_metric* metric,
                            char*        expression );

char*
cube_metric_get_expression( cube_metric* metric );

void
cube_metric_set_init_expression( cube_metric* metric,
                                 char*        expression );

char*
cube_metric_get_init_expression( cube_metric* metric );



void
cube_metric_setup_for_writing( cube_metric*    metric,
                               cube_dyn_array* cnds,
                               cube_dyn_array* nthds,
                               cube_dyn_array* roots_cnds );
void
cube_metric_set_known_cnodes( cube_metric* metric,
                              char*        known_cnodes /*, unsigned size*/ );

#if defined( BACKEND_CUBE_COMPRESSED ) || defined( FRONTEND_CUBE_COMPRESSED )
void
cube_metric_setup_subindex( cube_metric* metric ); /* creates a subindex, having known_cnodes or ncn data*/

#endif


uint32_t
cube_metric_get_position_of_row( cube_metric* metric,
                                 unsigned     cid );
char*
cube_metric_get_own_type( cube_metric* metric );

cube_metric*
cube_metric_get_child( cube_metric* metric,
                       int          i );
cube_metric*
cube_metric_get_parent( cube_metric* metric );
char*
cube_metric_get_disp_name( cube_metric* metric );
char*
cube_metric_get_uniq_name( cube_metric* metric );
char*
cube_metric_get_dtype( cube_metric* metric );
char*
cube_metric_get_uom( cube_metric* metric );
char*
cube_metric_get_val( cube_metric* metric );
char*
cube_metric_get_descr( cube_metric* metric );
char*
cube_metric_get_url( cube_metric* metric );
int
cube_metric_num_children( cube_metric* metric );
void
cube_metric_add_child( cube_metric* parent,
                       cube_metric* met );

/**   void  cube_metric_assign_ids(cube_metric* metric, int* id); */
int
cube_metric_get_level( cube_metric* metric );


void
cube_metric_writeXML( cube_metric* metric,
                      FILE*        fp );
void
cube_metric_writeXML_data( cube_metric* metric,
                           FILE*        fp );

void
cube_metric_set_id( cube_metric* metric,
                    int          new_id );
void
cube_metric_set_uniq_name( cube_metric* metric,
                           char*        uniq_name );
int
cube_metric_get_id( cube_metric* metric );
int
cube_metric_equal( cube_metric* a,
                   cube_metric* b );


void*
cube_metric_transform_row_of_doubles( cube_metric* metric,
                                      double*      data_row );
void*
cube_metric_transform_row_of_uint64( cube_metric* metric,
                                     uint64_t*    data_row );

void
cube_metric_write_row( cube_metric* metric,
                       cube_cnode*  cnd,
                       void*        data_row );
void
cube_metric_write_row_of_doubles( cube_metric* metric,
                                  cube_cnode*  cnd,
                                  double*      data_row );
void
cube_metric_write_row_of_uint64( cube_metric* metric,
                                 cube_cnode*  cnd,
                                 uint64_t*    data_row );



/* List of the canonical writing calls. They perform type casting if needed and check metric if it supports that kind of value. Internally they call cube_write_sev_row(...)*/

void
cube_metric_write_row_of_cube_type_uint8( cube_metric*     met,
                                          cube_cnode*      cnd,
                                          cube_type_uint8* sevs );
void
cube_metric_write_row_of_cube_type_int8( cube_metric*    met,
                                         cube_cnode*     cnd,
                                         cube_type_int8* sevs );
void
cube_metric_write_row_of_cube_type_uint16( cube_metric*      met,
                                           cube_cnode*       cnd,
                                           cube_type_uint16* sevs );
void
cube_metric_write_row_of_cube_type_int16( cube_metric*     met,
                                          cube_cnode*      cnd,
                                          cube_type_int16* sevs );
void
cube_metric_write_row_of_cube_type_uint32( cube_metric*      met,
                                           cube_cnode*       cnd,
                                           cube_type_uint32* sevs );
void
cube_metric_write_row_of_cube_type_int32( cube_metric*     met,
                                          cube_cnode*      cnd,
                                          cube_type_int32* sevs );
void
cube_metric_write_row_of_cube_type_uint64( cube_metric*      met,
                                           cube_cnode*       cnd,
                                           cube_type_uint64* sevs );
void
cube_metric_write_row_of_cube_type_int64( cube_metric*     met,
                                          cube_cnode*      cnd,
                                          cube_type_int64* sevs );
void
cube_metric_write_row_of_cube_type_double( cube_metric*      met,
                                           cube_cnode*       cnd,
                                           cube_type_double* sevs );
void
cube_metric_write_row_of_cube_type_complex( cube_metric*       met,
                                            cube_cnode*        cnd,
                                            cube_type_complex* sevs );
void
cube_metric_write_row_of_cube_type_tau_atomic( cube_metric*          met,
                                               cube_cnode*           cnd,
                                               cube_type_tau_atomic* sevs );
void
cube_metric_write_row_of_cube_type_scale_func( cube_metric*          met,
                                               cube_cnode*           cnd,
                                               cube_type_scale_func* sevs );
void
cube_metric_write_row_of_cube_type_rate( cube_metric*    met,
                                         cube_cnode*     cnd,
                                         cube_type_rate* sevs );




void
cube_metric_create_enumeration( cube_metric*    metric,
                                cube_dyn_array* roots_cnodes,
                                cube_dyn_array* cnodes );
void
__cube_metric_traverse_cnodes_simple( unsigned*    map,
                                      unsigned*    sequence,
                                      cube_cnode** cnodes_sequence,
                                      unsigned*    index,
                                      unsigned     size,
                                      cube_cnode*  cnode );
void
__cube_metric_traverse_cnodes_wide_search( unsigned*    map,
                                           unsigned*    sequence,
                                           cube_cnode** cnodes_sequence,
                                           unsigned*    index,
                                           unsigned     size,
                                           cube_cnode*  cnode );
void
__cube_metric_traverse_cnodes_deep_search( unsigned*    map,
                                           unsigned*    sequence,
                                           cube_cnode** cnodes_sequence,
                                           unsigned*    index,
                                           unsigned     size,
                                           cube_cnode*  cnode );

carray*
cube_metric_return_enumeration( cube_metric* metric );


char*
__cube_metric_bitstring_transformation( cube_metric* metric,
                                        char*        known_cnodes );

uint64_t
cube_metric_size_of_data_file( cube_metric* met );
uint64_t
cube_metric_size_of_index_file( cube_metric* met );

void*
cube_metric_get_empty_row( cube_metric* met );

#ifdef __cplusplus
}
#endif

#endif
