/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include "readcb.h"

#include <algorithm>

#include <elg_readcb.h>

#include <pearl/Callsite.h>
#include <pearl/Cartesian.h>
#include <pearl/Communicator.h>
#include <pearl/Error.h>
#include <pearl/Event_rep.h>
#include <pearl/GlobalDefs.h>
#include <pearl/LocalTrace.h>
#include <pearl/Location.h>
#include <pearl/MpiCollEnd_rep.h>
#include <pearl/String.h>
#include <pearl/SystemNode.h>
#include <pearl/RmaWindow.h>

#include "DefsFactory.h"
#include "EpikArchive.h"
#include "EventFactory.h"
#include "MappingTable.h"
#include "Process.h"

using namespace std;
using namespace pearl;


//--- Macros ----------------------------------------------------------------

#define CONV_ID(id)     (((id)  == ELG_NO_ID)  ? PEARL_NO_ID  : (id))
#define CONV_NUM(num)   (((num) == ELG_NO_NUM) ? PEARL_NO_NUM : (num))


//--- Constants -------------------------------------------------------------

// Used for unknown entities
const string str_unknown("");


//--- Local helper functions ------------------------------------------------

namespace
{

Region::Role convertRegionRole(elg_ui1 type, bool isArtificial)
{
    switch (type) {
        case ELG_FUNCTION:
            return Region::ROLE_FUNCTION;

        case ELG_LOOP:
            return Region::ROLE_LOOP;

        case ELG_USER_REGION:
            return Region::ROLE_CODE;

        case ELG_FUNCTION_COLL_BARRIER:
            return Region::ROLE_BARRIER;

        case ELG_FUNCTION_COLL_ONE2ALL:
            return Region::ROLE_COLL_ONE2ALL;

        case ELG_FUNCTION_COLL_ALL2ONE:
            return Region::ROLE_COLL_ALL2ONE;

        case ELG_FUNCTION_COLL_ALL2ALL:
            return Region::ROLE_COLL_ALL2ALL;

        case ELG_FUNCTION_COLL_OTHER:
            return Region::ROLE_COLL_OTHER;

        case ELG_OMP_PARALLEL:
            return Region::ROLE_PARALLEL;

        case ELG_OMP_LOOP:
            return Region::ROLE_LOOP;

        case ELG_OMP_SECTIONS:
            return Region::ROLE_SECTIONS;

        case ELG_OMP_SECTION:
            return Region::ROLE_SECTION;

        case ELG_OMP_WORKSHARE:
            return Region::ROLE_WORKSHARE;

        case ELG_OMP_SINGLE:
            return Region::ROLE_SINGLE;

        case ELG_OMP_MASTER:
            return Region::ROLE_MASTER;

        case ELG_OMP_CRITICAL:
            return Region::ROLE_CRITICAL;

        case ELG_OMP_ATOMIC:
            return Region::ROLE_ATOMIC;

        case ELG_OMP_BARRIER:
            return Region::ROLE_BARRIER;

        case ELG_OMP_IBARRIER:
            return Region::ROLE_IMPLICIT_BARRIER;

        case ELG_OMP_FLUSH:
            return Region::ROLE_FLUSH;

        case ELG_OMP_CRITICAL_SBLOCK:
            return Region::ROLE_CRITICAL_SBLOCK;

        case ELG_OMP_SINGLE_SBLOCK:
            return Region::ROLE_SINGLE_SBLOCK;

        case ELG_OMP_ORDERED:
            return Region::ROLE_ORDERED;

        case ELG_OMP_ORDERED_SBLOCK:
            return Region::ROLE_ORDERED_SBLOCK;

        default:
            if (isArtificial)
                return Region::ROLE_ARTIFICIAL;
            break;
    }

    return Region::ROLE_UNKNOWN;
}


Region::Paradigm convertParadigm(const string& descr)
{
    if (descr == "USR")
        return Region::PARADIGM_USER;

    if (descr == "OMP")
        return Region::PARADIGM_OPENMP;

    if (descr == "MPI")
        return Region::PARADIGM_MPI;

    if (descr == "EPIK")
        return Region::PARADIGM_MEASUREMENT_SYSTEM;

    return Region::PARADIGM_UNKNOWN;
}


MpiCollEnd_rep::coll_type convertCollectiveType(const Region& region)
{
    // Sanity check
    if (!is_mpi_collective(region))
        throw FatalError("Inconsistent trace data -- "
                         "MPI collective event found in non-collective region!");

    // Barrier
    if (is_mpi_barrier(region)) {
        return MpiCollEnd_rep::BARRIER;
    }

    // Prepare region name
    const string name = region.getCanonicalName().getString();

    // 1-to-N collectives
    if (is_mpi_12n(region)) {
        if (name == "MPI_Bcast") {
            return MpiCollEnd_rep::BCAST;
        } else if (name == "MPI_Scatter") {
            return MpiCollEnd_rep::SCATTER;
        } else if (name == "MPI_Scatterv") {
            return MpiCollEnd_rep::SCATTERV;
        }
    }

    // N-to-1 collectives
    if (is_mpi_n21(region)) {
        if (name == "MPI_Reduce") {
            return MpiCollEnd_rep::REDUCE;
        } else if (name == "MPI_Gather") {
            return MpiCollEnd_rep::GATHER;
        } else if (name == "MPI_Gatherv") {
            return MpiCollEnd_rep::GATHERV;
        }
    }

    // N-to-N collectives
    if (is_mpi_n2n(region)) {
        if (name == "MPI_Allgather") {
            return MpiCollEnd_rep::ALLGATHER;
        } else if (name == "MPI_Allgatherv") {
            return MpiCollEnd_rep::ALLGATHERV;
        } else if (name == "MPI_Allreduce") {
            return MpiCollEnd_rep::ALLREDUCE;
        } else if (name == "MPI_Alltoall") {
            return MpiCollEnd_rep::ALLTOALL;
        } else if (name == "MPI_Alltoallv") {
            return MpiCollEnd_rep::ALLTOALLV;
        } else if (name == "MPI_Alltoallw") {
            return MpiCollEnd_rep::ALLTOALLW;
        } else if (name == "MPI_Reduce_scatter") {
            return MpiCollEnd_rep::REDUCE_SCATTER;
        } else if (name == "MPI_Reduce_scatter_block") {
            return MpiCollEnd_rep::REDUCE_SCATTER_BLOCK;
        }
    }

    // Prefix reductions
    if (is_mpi_scan(region)) {
        if (name == "MPI_Scan") {
            return MpiCollEnd_rep::SCAN;
        } else if (name == "MPI_Exscan") {
            return MpiCollEnd_rep::EXSCAN;
        }
    }

    // We should never reach this point...
    assert(false);
    return MpiCollEnd_rep::BARRIER;
}


}   // unnamed namespace


//---------------------------------------------------------------------------
//
//  struct DefsCbData
//
//---------------------------------------------------------------------------

//--- Constructors & destructor ---------------------------------------------

DefsCbData::DefsCbData(EpikArchive& archive,
                       GlobalDefs&  defs)
  : m_archive(archive),
    m_defs(defs),
    m_strcnt(0),
    m_strid(PEARL_NO_ID),
    m_worldsize(1)
{
}


//--- Mapping functions -----------------------------------------------------

string DefsCbData::get_string(ident_t id) const
{
  if (id == PEARL_NO_ID)
    return str_unknown;

  if (id >= m_strmap.size())
    throw RuntimeError("Oops! Invalid string ID.");

  return m_strmap[id];
}


ident_t DefsCbData::get_file(ident_t id) const
{
  if (id == PEARL_NO_ID)
    return PEARL_NO_ID;

  if (id >= m_filemap.size())
    throw RuntimeError("Oops! Invalid file name ID.");

  return m_filemap[id];
}


//---------------------------------------------------------------------------
//
//  struct TraceCbData
//
//---------------------------------------------------------------------------

//--- Constructors & destructor ---------------------------------------------

TraceCbData::TraceCbData(const EpikArchive& archive,
                         const GlobalDefs&  defs,
                         LocalTrace&        trace,
                         MappingTable&      table,
                         uint32_t           version,
                         Location::IdType   locId,
                         uint32_t           tid)
  : m_archive(archive),
    m_defs(defs),
    m_trace(trace),
    m_table(table),
    m_version(version),
    m_location(locId),
    m_tid(tid),
    m_requestId(PEARL_NO_REQUEST)
{
}


//---------------------------------------------------------------------------
//
//  EPILOG callback functions
//
//---------------------------------------------------------------------------


//*** Generic callbacks *****************************************************

void elg_readcb_ALL(elg_ui1 type,     /* record type */
                    elg_ui1 length,   /* record length */
                    void*   userdata)
{
  /* The type and length fields are ignored during reading. */
}


//*** Definition records ****************************************************

#define CALLBACK_SETUP \
  DefsCbData* data = static_cast<DefsCbData*>(userdata); \
  GlobalDefs& defs = data->m_defs; \
  try {

#define CALLBACK_SETUP_NO_DEFS \
  DefsCbData* data = static_cast<DefsCbData*>(userdata); \
  try {

#define CALLBACK_CLEANUP \
  } /* Closes the try block */ \
  catch (exception& ex) { \
    data->m_message = ex.what(); \
  }


void elg_readcb_STRING(elg_ui4     strid,   /* string identifier */
                       elg_ui1     cntc,    /* number of continuation records */
                       const char* str,     /* string */
                       void*       userdata)
{
  CALLBACK_SETUP

  // Sanity check
  if (strid != data->m_strmap.size())
    throw RuntimeError("Oops! Invalid string ID.");

  // Store (partial) string in callback data structure
  if (cntc == 0) {
    DefsFactory::instance()->createString(defs, strid, str);
    data->m_strmap.push_back(string(str));
  } else {
    data->m_strid  = strid;
    data->m_strcnt = cntc;
    data->m_strmap.push_back(string(str, 250));
  }

  CALLBACK_CLEANUP
}


void elg_readcb_STRING_CNT(const char* str,   /* continued string */
                           void*       userdata)
{
  CALLBACK_SETUP

  // Sanity check
  if (data->m_strcnt == 0)
    throw RuntimeError("Oops! Unexpected ELG_STRING_CNT record.");

  // Append (partial) string to existing entry in callback data structure
  data->m_strcnt--;
  string& tmp = data->m_strmap[data->m_strid];
  if (data->m_strcnt == 0) {
    tmp.append(str);
    DefsFactory::instance()->createString(defs, data->m_strid, tmp);
  } else {
    tmp.append(str, 255);
  }

  CALLBACK_CLEANUP
}


void elg_readcb_MACHINE(elg_ui4 mid,     /* machine identifier */
                        elg_ui4 nodec,   /* number of nodes */
                        elg_ui4 mnid,    /* machine name identifier */
                        void*   userdata)
{
  CALLBACK_SETUP

  uint32_t nodeId = defs.numSystemNodes();
  data->m_machinemap[mid] = nodeId;
  DefsFactory::instance()->createSystemNode(defs,
                                            nodeId,
                                            CONV_ID(mnid),
                                            String::MACHINE_ID,
                                            SystemNode::NO_ID);

  CALLBACK_CLEANUP
}


void elg_readcb_NODE(elg_ui4 nid,    /* node identifier */
                     elg_ui4 mid,    /* machine identifier */
                     elg_ui4 cpuc,   /* number of CPUs */
                     elg_ui4 nnid,   /* node name identifer */
                     elg_d8  cr,     /* number of clock cycles/sec */
                     void*   userdata)
{
  CALLBACK_SETUP

  uint32_t nodeId = defs.numSystemNodes();
  data->m_nodemap[nid] = nodeId;
  DefsFactory::instance()->createSystemNode(defs,
                                            nodeId,
                                            CONV_ID(nnid),
                                            String::NODE_ID,
                                            data->m_machinemap[mid]);

  CALLBACK_CLEANUP
}


void elg_readcb_PROCESS(elg_ui4 pid,    /* process identifier */
                        elg_ui4 pnid,   /* process name identifier */
                        void*   userdata)
{
  CALLBACK_SETUP

  data->m_processmap[CONV_ID(pid)] = CONV_ID(pnid);

  CALLBACK_CLEANUP
}


void elg_readcb_THREAD(elg_ui4 tid,    /* thread identifier */
                       elg_ui4 pid,    /* process identifier */
                       elg_ui4 tnid,   /* thread name identifier */
                       void*   userdata)
{
  CALLBACK_SETUP

  data->m_threadmap[
    map<pair<ident_t,ident_t>,ident_t>::key_type(CONV_ID(pid), CONV_ID(tid))
  ] = CONV_ID(tnid);

  CALLBACK_CLEANUP
}


void elg_readcb_LOCATION(elg_ui4 lid,   /* location identifier */
                         elg_ui4 mid,   /* machine identifier */
                         elg_ui4 nid,   /* node identifier */
                         elg_ui4 pid,   /* process identifier */
                         elg_ui4 tid,   /* thread identifier */
                         void*   userdata)
{
  CALLBACK_SETUP

  uint32_t pnid = String::NO_ID;
  map<ident_t,ident_t>::const_iterator pit = data->m_processmap.find(CONV_ID(pid));
  if (pit != data->m_processmap.end())
    pnid = pit->second;
  try {
    defs.getLocationGroup(CONV_ID(pid));
  }
  catch (RuntimeError& ex) {
    DefsFactory::instance()->createLocationGroup(defs,
                                                 CONV_ID(pid),
                                                 pnid,
                                                 LocationGroup::TYPE_PROCESS,
                                                 data->m_nodemap[CONV_ID(nid)]);
    const LocationGroup& group = defs.getLocationGroup(CONV_ID(pid));
    Process& process = const_cast<Process&>(static_cast<const Process&>(group));
    process.setRank(CONV_ID(pid));
  }

  // Find thread name string definition
  uint32_t tnid = String::NO_ID;
  map<pair<ident_t,ident_t>,ident_t>::const_iterator tit =
    data->m_threadmap.find(
      map<pair<ident_t,ident_t>,ident_t>::key_type(CONV_ID(pid), CONV_ID(tid)));
  if (tit  != data->m_threadmap.end())
    tnid = tit->second;

  // Calculate location ID and store mapping
  Location::IdType locId = (static_cast<Location::IdType>(tid) << 32) + pid;
  data->m_archive.addIdMapping(lid, locId);

  // Create location instance
  DefsFactory::instance()->createLocation(defs,
                                          locId,
                                          tnid,
                                          Location::TYPE_CPU_THREAD,
                                          0,
                                          CONV_ID(pid));

  CALLBACK_CLEANUP
}


void elg_readcb_FILE(elg_ui4 fid,    /* file identifier */
                     elg_ui4 fnid,   /* file name identifier */
                     void*   userdata)
{
  CALLBACK_SETUP_NO_DEFS

  // Sanity check
  if (fid != data->m_filemap.size())
    throw RuntimeError("Oops! Invalid file name ID.");

  // Store file name in callback data structure
  data->m_filemap.push_back(CONV_ID(fnid));

  CALLBACK_CLEANUP
}


void elg_readcb_REGION(elg_ui4 rid,     /* region identifier */
                       elg_ui4 rnid,    /* region name identifier */
                       elg_ui4 fid,     /* source file identifier */
                       elg_ui4 begln,   /* begin line number */
                       elg_ui4 endln,   /* end line number */
                       elg_ui4 rdid,    /* region description identifier */
                       elg_ui1 rtype,   /* region type */
                       void*   userdata)
{
  CALLBACK_SETUP

  uint32_t      fnameId = data->get_file(fid);
  const String& name    = defs.getString(CONV_ID(rnid));
  const String& desc    = defs.getString(CONV_ID(rdid));
  const String& file    = defs.getString(fnameId);

  // Handle special "PAUSING" and "TRACING" regions
  DefsFactory::InternalRegionType type         = DefsFactory::USER_REGION;
  bool                            isArtificial = false;
  if ("EPIK" == desc.getString() && "EPIK" == file.getString()) {
    isArtificial = true;
    if ("PAUSING" == name.getString()) {
      type = DefsFactory::PAUSING_REGION;
    }
    else if ("TRACING" == name.getString()) {
      type = DefsFactory::FLUSHING_REGION;
    }
  }

  DefsFactory::instance()->createRegion(defs,
                                        CONV_ID(rid),
                                        CONV_ID(rnid),
                                        CONV_ID(rnid),
                                        CONV_ID(rdid),
                                        convertRegionRole(rtype, isArtificial),
                                        convertParadigm(data->get_string(CONV_ID(rdid))),
                                        fnameId,
                                        CONV_NUM(begln),
                                        CONV_NUM(endln),
                                        type);

  CALLBACK_CLEANUP
}


void elg_readcb_CALL_SITE(elg_ui4 csid,   /* call site identifier */
                          elg_ui4 fid,    /* source file identifier */
                          elg_ui4 lno,    /* line number */
                          elg_ui4 erid,   /* region identifer to be entered */
                          elg_ui4 lrid,   /* region identifer to be left */
                          void*   userdata)
{
  CALLBACK_SETUP

  DefsFactory::instance()->createCallsite(defs,
                                          CONV_ID(csid),
                                          data->get_file(fid),
                                          CONV_NUM(lno),
                                          CONV_ID(erid));

  CALLBACK_CLEANUP
}


void elg_readcb_CALL_PATH(elg_ui4 cpid,   /* call-path identifier */
                          elg_ui4 rid,    /* node region identifier */
                          elg_ui4 ppid,   /* parent call-path identifier */
                          elg_ui8 order,  /* node order specifier */
                          void*   userdata)
{
  CALLBACK_SETUP

  DefsFactory::instance()->createCallpath(defs,
                                          CONV_ID(cpid),
                                          CONV_ID(rid),
                                          PEARL_NO_ID,
                                          CONV_ID(ppid));

  CALLBACK_CLEANUP
}


void elg_readcb_METRIC(elg_ui4 metid,      /* metric identifier */
                       elg_ui4 metnid,     /* metric name identifier */
                       elg_ui4 metdid,     /* metric description identifier */
                       elg_ui1 metdtype,   /* metric data type */
                       elg_ui1 metmode,    /* metric mode */
                       elg_ui1 metiv,      /* time interval referenced */
                       void*   userdata)
{
  CALLBACK_SETUP

  DefsFactory::instance()->createMetric(defs,
                                        CONV_ID(metid),
                                        data->get_string(CONV_ID(metnid)),
                                        data->get_string(CONV_ID(metdid)),
                                        metdtype,
                                        metmode,
                                        metiv);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_GROUP(elg_ui4 gid,         /* group identifier */
                          elg_ui1 mode,        /* mode flags */
                          elg_ui4 grpc,        /* number of ranks in group */
                          elg_ui4 grpv[],      /* local |-> global rank mapping */
                          void*   userdata)
{
  CALLBACK_SETUP

  // Set up process identifier list
  DefsFactory::process_group process_ids;
  process_ids.reserve(grpc);
  for (elg_ui4 i = 0; i < grpc; ++i)
    process_ids.push_back(grpv[i]);

  // Create MPI group record
  DefsFactory::instance()->createMpiGroup(defs,
                                          CONV_ID(gid),
                                          String::NO_ID,
                                          process_ids,
                                          mode & ELG_GROUP_SELF,
                                          mode & ELG_GROUP_WORLD);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_COMM_DIST(elg_ui4 cid,    /* communicator id unique to root */
                              elg_ui4 root,   /* global rank of root process */
                              elg_ui4 lcid,   /* local communicator id on process */
                              elg_ui4 lrank,  /* local rank of process */
                              elg_ui4 size,   /* size of communicator */
                              void*   userdata)
{
  DefsCbData* data = static_cast<DefsCbData*>(userdata);

  data->m_message = "Unexpected ELG_MPI_COMM_DIST record!";
}


void elg_readcb_MPI_COMM_REF(elg_ui4 cid,      /* communicator identifier */
                             elg_ui4 gid,      /* group identifier */
                             void*   userdata)
{
  CALLBACK_SETUP

  // Create MPI communicator record
  DefsFactory::instance()->createMpiComm(defs,
                                         CONV_ID(cid),
                                         String::NO_ID,
                                         CONV_ID(gid),
                                         Communicator::NO_ID);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_COMM(elg_ui4  cid,    /* communicator identifier */
                         elg_ui1  mode,   /* mode flags */
                         elg_ui4  grpc,   /* size of bitstring in bytes */
                         elg_ui1* grpv,   /* bitstring defining the group */
                         void*    userdata)
{
  CALLBACK_SETUP

  // Set up process identifier list
  DefsFactory::process_group process_ids;
  process_ids.reserve(grpc * 8);
  for (uint32_t byte = 0; byte < grpc; ++byte) {
    uint8_t  value = grpv[byte];

    for (int bit = 0; bit < 8; ++bit) {
      if (value & (1 << bit)) {
        uint32_t pid = byte * 8 + bit;

        // Store global process identifier
        process_ids.push_back(pid);
      }
    }
  }

  // MPI_COMM_WORLD is always defined by communicator with global ID 0
  if (0 == cid)
    data->m_worldsize = process_ids.size();

  // Determine WORLD flag
  bool is_world = (data->m_worldsize == process_ids.size());

  // Create MPI group record
  DefsFactory::instance()->createMpiGroup(defs,
                                          CONV_ID(cid),
                                          String::NO_ID,
                                          process_ids,
                                          mode & ELG_GROUP_SELF,
                                          is_world);

  // Create MPI communicator record
  DefsFactory::instance()->createMpiComm(defs,
                                         CONV_ID(cid),
                                         String::NO_ID,
                                         CONV_ID(cid),
                                         Communicator::NO_ID);

  CALLBACK_CLEANUP
}


void elg_readcb_CART_TOPOLOGY(elg_ui4  topid,     /* topology id */
                              elg_ui4  tnid,      /* topology name id */
                              elg_ui4  cid,       /* communicator id */
                              elg_ui1  ndims,     /* number of dimensions */
                              elg_ui4* dimv,      /* number of processes in each dim */  
                              elg_ui1* periodv,   /* periodicity in each dim */
                              elg_ui4* dimids,    /* dimension name ids */
                              void*    userdata)
{
  CALLBACK_SETUP

  // Set up dimension vector
  Cartesian::cart_dims dimensions;
  dimensions.reserve(ndims);
  for (elg_ui1 i = 0; i < ndims; ++i)
    dimensions.push_back(dimv[i]);

  // Set up periodicity vector
  Cartesian::cart_period periodic;
  periodic.reserve(ndims);
  for (elg_ui1 i = 0; i < ndims; ++i)
    periodic.push_back(periodv[i]);

  // Create new cartesian grid
  if (cid != ELG_NO_ID) {
    DefsFactory::instance()->createMpiCartesian(defs,
                                                CONV_ID(topid),
                                                dimensions,
                                                periodic,
                                                CONV_ID(cid));
  } else {
    DefsFactory::instance()->createCartesian(defs,
                                             CONV_ID(topid),
                                             dimensions,
                                             periodic);
  }

  CALLBACK_CLEANUP
}


void elg_readcb_CART_COORDS(elg_ui4  topid,    /* topology id */
                            elg_ui4  lid,      /* location id */
                            elg_ui1  ndims,    /* number of dimensions */
                            elg_ui4* coordv,   /* coordinates in each dimension */
                            void*    userdata)
{
  CALLBACK_SETUP

  // Retrieve cartesian topology
  Cartesian* cart = defs.get_cartesian(CONV_ID(topid));

  // Set up coordinate vector
  Cartesian::cart_coords coordinate;
  coordinate.reserve(ndims);
  for (elg_ui1 i = 0; i < ndims; ++i)
    coordinate.push_back(coordv[i]);

  // Set coordinate
  cart->set_coords(defs.getLocation(data->m_archive.mapLocationId(lid)),
                   coordinate);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_WIN(elg_ui4 wid,   /* window identifier */
                        elg_ui4 cid,   /* communicator identifier */
                        void*   userdata)
{
  CALLBACK_SETUP

  DefsFactory::instance()->createMpiWindow(defs,
                                           CONV_ID(wid),
                                           CONV_ID(cid));

  CALLBACK_CLEANUP
}


void elg_readcb_MAP_OFFSET(elg_ui4 rank,   /* rank number the offset relates to */
                           elg_ui4 offset, /* file offset in mapping file */
                           void* userdata )
{
  CALLBACK_SETUP

  data->m_archive.setOffset(rank, offset);

  CALLBACK_CLEANUP
}

#undef CALLBACK_SETUP
#undef CALLBACK_CLEANUP


//*** Mapping file records **************************************************

#define CALLBACK_SETUP \
  MappingTable& table = *static_cast<MappingTable*>(userdata); \
  try {

#define CALLBACK_CLEANUP \
  } /* Closes the try block */ \
  catch (exception& ex) { \
    table.set_message(ex.what()); \
  }


void elg_readcb_MAP_SECTION(elg_ui4 rank,   /* rank number the section relates to */
                            void* userdata)
{
  CALLBACK_SETUP

  table.set_rank(rank);
  table.set_finished();

  CALLBACK_CLEANUP
}


void elg_readcb_OFFSET(elg_d8 ltime,    /* local time */
                       elg_d8 offset,   /* offset to global time */
                       void*  userdata)
{
  CALLBACK_SETUP

  table.set_time(ltime, offset);

  CALLBACK_CLEANUP
}


void elg_readcb_IDMAP(elg_ui1 type,     /* object type to be mapped */
                      elg_ui1 mode,     /* mapping mode (dense/sparse) */
                      elg_ui4 count,    /* number of entries */
                      elg_ui4 mapv[],   /* vector of mappings */
                      void*   userdata)
{
  CALLBACK_SETUP

  table.set_table(type, mode, count, mapv);

  CALLBACK_CLEANUP
}

#undef CALLBACK_SETUP
#undef CALLBACK_CLEANUP


//--- Obsolete definition records -------------------------------------------

void elg_readcb_LAST_DEF(void* userdata)
{
}


void elg_readcb_NUM_EVENTS(elg_ui4 eventc,   /* number of events */
                           void*   userdata)
{
}


void elg_readcb_EVENT_TYPES(elg_ui4 ntypes,     /* number of event types */
                            elg_ui1 typev[],    /* vector of event types */
                            void*   userdata)
{
}


void elg_readcb_EVENT_COUNTS(elg_ui4 ntypes,     /* number of event types */
                             elg_ui4 countv[],   /* vector of event counts */
                             void*   userdata)
{
}


//*** Event records *********************************************************

// Set up shortcuts for commonly used fields in userdata and
// ignore events from other locations
#define CALLBACK_SETUP \
  TraceCbData*      data  = static_cast<TraceCbData*>(userdata); \
  const GlobalDefs& defs  = data->m_defs; \
  LocalTrace&       trace = data->m_trace; \
  MappingTable&     table = data->m_table; \
  \
  try { \
    Location::IdType mapped_lid; \
    mapped_lid = data->m_archive.mapLocationId(table.map_location(lid)); \
    if (data->m_location != mapped_lid) { \
      data->m_requestId = PEARL_NO_REQUEST; \
      return; \
    }

#define CALLBACK_CLEANUP \
    data->m_requestId = PEARL_NO_REQUEST; \
  } /* Closes the try block */ \
  catch (exception& ex) { \
    data->m_message = ex.what(); \
  }


#define ATTR_CALLBACK_SETUP \
  TraceCbData*      data  = static_cast<TraceCbData*>(userdata);


//--- Attributes ------------------------------------------------------------

void elg_readcb_ATTR_UI1(elg_ui1 type, elg_ui1 val, void* userdata)
{
}

void elg_readcb_ATTR_UI4(elg_ui1 type, elg_ui4 val, void* userdata)
{
  ATTR_CALLBACK_SETUP

  switch (type) {
  case ELG_ATTR_REQUEST:
      data->m_requestId = val;
      break;

  default:
      break;
  }
}


//--- Programming-model-independent events ----------------------------------

void elg_readcb_ENTER(elg_ui4  lid,    /* location identifier */
                      elg_d8   time,   /* time stamp */
                      elg_ui4  rid,    /* region identifier of the region being entered */
                      elg_ui1  metc,   /* number of metrics */
                      elg_ui8* metv,   /* metric values */
                      void*    userdata)
{
  CALLBACK_SETUP

  // Callstack update
  uint32_t regionId = table.map_region(CONV_ID(rid));
  data->m_callstack.push(regionId);

  // Create event
  Event_rep* event =
    EventFactory::instance()->createEnter(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  // Special handling for MPI collectives
  const Region& region = defs.getRegion(regionId);
  if (is_mpi_collective(region)) {
    event =
      EventFactory::instance()->createMpiCollBegin(defs,
                                                   table.map_time(time));

    if (event)
      trace.add_event(event);
  }

  // Special handling for MPI RMA collectives
  else if (is_mpi_rma_collective(region)) {
    event =
      EventFactory::instance()->createMpiRmaCollBegin(defs,
                                                      table.map_time(time));

    if (event)
      trace.add_event(event);
  }

  CALLBACK_CLEANUP
}


void elg_readcb_EXIT(elg_ui4  lid,    /* location identifier */
                     elg_d8   time,   /* time stamp */
                     elg_ui1  metc,   /* number of metrics */
                     elg_ui8* metv,   /* metric values */
                     void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_ENTER_CS(elg_ui4  lid,    /* location identifier */
                         elg_d8   time,   /* time stamp */
                         elg_ui4  csid,   /* call site identifier of the call site being entered */
                         elg_ui1  metc,   /* number of metrics */
                         elg_ui8* metv,   /* metric values */
                         void*    userdata)
{
  CALLBACK_SETUP

  // Callstack update
  uint32_t callsiteId = table.map_callsite(CONV_ID(csid));
  uint32_t regionId   = defs.getCallsite(callsiteId).getCallee().getId();
  data->m_callstack.push(regionId);

  // Create event
  Event_rep* event =
    EventFactory::instance()->createEnterCS(defs,
                                            table.map_time(time),
                                            callsiteId,
                                            metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


//--- MPI-1 event records ---------------------------------------------------

void elg_readcb_MPI_SEND(elg_ui4 lid,    /* location identifier */
                         elg_d8  time,   /* time stamp */
                         elg_ui4 dlid,   /* destination location identifier */
                         elg_ui4 cid,    /* communicator identifier */
                         elg_ui4 tag,    /* message tag */
                         elg_ui4 sent,   /* message length in bytes */
                         void*   userdata)
{
  CALLBACK_SETUP

  // Map communicator ID
  ident_t comm_id = table.map_communicator(CONV_ID(cid));

  // Pre-1.8 traces: convert global |-> local destination rank
  if (data->m_version < 1008)
    dlid = defs.get_comm(comm_id)->getGroup().getLocalRank(dlid);


  Event_rep* event;
  if (data->m_requestId == PEARL_NO_REQUEST) {
    event =
      EventFactory::instance()->createMpiSend(defs,
                                              table.map_time(time),
                                              comm_id,
                                              dlid,
                                              tag,
                                              sent);
  } else {
    event =
      EventFactory::instance()->createMpiSendRequest(defs,
                                                     table.map_time(time),
                                                     comm_id,
                                                     dlid,
                                                     tag,
                                                     sent,
                                                     data->m_requestId);
  }

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_RECV(elg_ui4 lid,    /* location identifier */
                         elg_d8  time,   /* time stamp */
                         elg_ui4 slid,   /* source location identifier */
                         elg_ui4 cid,    /* communicator identifier */
                         elg_ui4 tag,    /* message tag */
                         void*   userdata)
{
  CALLBACK_SETUP

  // Map communicator ID
  ident_t comm_id = table.map_communicator(CONV_ID(cid));

  // Pre-1.8 traces: convert global |-> local source rank
  if (data->m_version < 1008)
    slid = defs.get_comm(comm_id)->getGroup().getLocalRank(slid);

  Event_rep* event;
  if (data->m_requestId == PEARL_NO_REQUEST) {
    event =
      EventFactory::instance()->createMpiRecv(defs,
                                              table.map_time(time),
                                              comm_id,
                                              slid,
                                              tag,
                                              0);
  } else {
    event =
      EventFactory::instance()->createMpiRecvComplete(defs,
                                                      table.map_time(time),
                                                      comm_id,
                                                      slid,
                                                      tag,
                                                      0,
                                                      data->m_requestId);
  }

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_COLLEXIT(elg_ui4  lid,      /* location identifier */
                             elg_d8   time,     /* time stamp */
                             elg_ui1  metc,     /* number of metrics */
                             elg_ui8* metv,     /* metric values */
                             elg_ui4  rlid,     /* root location identifier */
                             elg_ui4  cid,      /* communicator identifier */
                             elg_ui4  sent,     /* bytes sent */
                             elg_ui4  recvd,    /* bytes received */
                             void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Map communicator ID
  ident_t comm_id = table.map_communicator(CONV_ID(cid));

  // Pre-1.8 traces: convert global |-> local root rank
  if (rlid != ELG_NO_ID && data->m_version < 1008)
    rlid = defs.get_comm(comm_id)->getGroup().getLocalRank(rlid);

  const Region& region = defs.getRegion(regionId);
  Event_rep* event =
    EventFactory::instance()->createMpiCollEnd(defs,
                                               table.map_time(time),
                                               convertCollectiveType(region),
                                               comm_id,
                                               (rlid == ELG_NO_ID)
                                                 ? PEARL_NO_ID
                                                 : rlid,
                                               sent,
                                               recvd);

  if (event)
    trace.add_event(event);

  event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}

//--- MPI-1 non-blocking event records --------------------------------------

void elg_readcb_MPI_SEND_COMPLETE(elg_ui4 lid,       /* location identifier              */
                                  elg_d8  time,      /* time stamp */
                                  elg_ui4 reqid,     /* request identifier of completed send */
                                  void* userdata)
{
  CALLBACK_SETUP

  assert(reqid != PEARL_NO_ID);

  Event_rep* event =
    EventFactory::instance()->createMpiSendComplete(defs, 
                                                    table.map_time(time), 
                                                    reqid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_RECV_REQUEST(elg_ui4 lid,       /* location identifier              */
                                 elg_d8  time,      /* time stamp */
                                 elg_ui4 reqid,     /* receive request identifier       */
                                 void* userdata)
{
  CALLBACK_SETUP

  assert(reqid != PEARL_NO_ID);

  Event_rep* event = 
    EventFactory::instance()->createMpiRecvRequest(defs, 
                                                   table.map_time(time),
                                                   reqid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_REQUEST_TESTED(elg_ui4 lid,       /* location identifier              */
                                   elg_d8  time,      /* time stamp */
                                   elg_ui4 reqid,     /* receive request identifier       */
                                   void*   userdata)
{
  CALLBACK_SETUP

  assert(reqid != PEARL_NO_ID);

  Event_rep* event =
    EventFactory::instance()->createMpiRequestTested(defs, 
                                                     table.map_time(time),
                                                     reqid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_CANCELLED(elg_ui4 lid,       /* location identifier              */
                              elg_d8  time,      /* time stamp */
                              elg_ui4 reqid,     /* receive request identifier       */
                              void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createMpiCancelled(defs, 
                                                 table.map_time(time),
                                                 reqid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


//--- MPI-2 one-sided event records -----------------------------------------

void elg_readcb_MPI_PUT_1TS(elg_ui4 lid,      /* location identifier */
                            elg_d8  time,     /* time stamp */
                            elg_ui4 tlid,     /* target location identifier */
                            elg_ui4 wid,      /* window identifier */
                            elg_ui4 rmaid,    /* RMA operation identifier */
                            elg_ui4 nbytes,   /* message length in bytes */
                            void*   userdata)
{
  CALLBACK_SETUP

  // Map window ID
  ident_t win_id = table.map_window(CONV_ID(wid));

  // Pre-1.8 traces: convert global |-> local target rank
  if (data->m_version < 1008)
    tlid = defs.get_window(win_id)->get_comm()->getGroup().getLocalRank(CONV_ID(tlid));

  Event_rep* event =
    EventFactory::instance()->createMpiRmaPutStart(defs,
                                                   table.map_time(time),
                                                   rmaid,
                                                   tlid,
                                                   nbytes,
                                                   win_id);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_PUT_1TE(elg_ui4 lid,     /* location identifier */
                            elg_d8  time,    /* time stamp */
                            elg_ui4 dlid,    /* destination location identifier */
                            elg_ui4 wid,     /* window identifier */
                            elg_ui4 rmaid,   /* RMA operation identifier */
                            void*   userdata)
{
  /* NOTE: As we are no longer rewriting the trace, these events will
   *       never exist in the remote trace. We work directly on the
   *       _REMOTE events. Eventually, these event will deprecate. */
}


void elg_readcb_MPI_PUT_1TE_REMOTE(elg_ui4 lid,     /* location identifier */
                                   elg_d8  time,    /* time stamp */
                                   elg_ui4 dlid,    /* destination location identifier  */
                                   elg_ui4 wid,     /* window identifier */
                                   elg_ui4 rmaid,   /* RMA operation identifier */
                                   void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createMpiRmaPutEnd(defs,
                                                 table.map_time(time),
                                                 rmaid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_GET_1TS(elg_ui4 lid,      /* location identifier */
                            elg_d8  time,     /* time stamp */
                            elg_ui4 dlid,     /* destination location identifier */
                            elg_ui4 wid,      /* window identifier */
                            elg_ui4 rmaid,    /* RMA operation identifier */
                            elg_ui4 nbytes,   /* message length in bytes */
                            void*   userdata)
{
  /* NOTE: As we are no longer rewriting the trace, these events will
   *       never exist in the remote trace. We work directly on the
   *       _REMOTE events. Eventually, these event will deprecate. */
}


void elg_readcb_MPI_GET_1TS_REMOTE(elg_ui4 lid,      /* location identifier */
                                   elg_d8  time,     /* time stamp */
                                   elg_ui4 dlid,     /* destination location identifier */
                                   elg_ui4 wid,      /* window identifier */
                                   elg_ui4 rmaid,    /* RMA operation identifier */
                                   elg_ui4 nbytes,   /* message length in bytes */
                                   void*   userdata)
{
  CALLBACK_SETUP

  // Map window ID
  ident_t win_id = table.map_window(CONV_ID(wid));

  // Pre-1.8 traces: convert global |-> local origin rank
  if (data->m_version < 1008)
    dlid = defs.get_window(win_id)->get_comm()->getGroup().getLocalRank(CONV_ID(dlid));

  Event_rep* event =
    EventFactory::instance()->createMpiRmaGetStart(defs,
                                                   table.map_time(time),
                                                   rmaid,
                                                   dlid,
                                                   nbytes,
                                                   win_id);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_GET_1TE(elg_ui4 lid,     /* location identifier */
                            elg_d8  time,    /* time stamp */
                            elg_ui4 slid,    /* source location identifier */
                            elg_ui4 wid,     /* window identifier */
                            elg_ui4 rmaid,   /* RMA operation identifier */
                            void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createMpiRmaGetEnd(defs,
                                                 table.map_time(time),
                                                 rmaid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_GET_1TO(elg_ui4 lid,     /* location identifier */
                            elg_d8  time,    /* time stamp */
                            elg_ui4 rmaid,   /* RMA operation identifier */
                            void*   userdata)
{
  /* NOTE: These events are currently completely ignored, as they are
   *       not used for analysis, but only for visualisation. */
}


void elg_readcb_MPI_WINEXIT(elg_ui4  lid,     /* location identifier */
                            elg_d8   time,    /* time stamp */
                            elg_ui1  metc,    /* number of metrics */
                            elg_ui8* metv,    /* metric values */
                            elg_ui4  wid,     /* window identifier */
                            elg_ui4  cid,     /* communicator identifier */
                            elg_ui1  synex,   /* synchronization exit flag */
                            void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createMpiRmaGats(defs,
                                               table.map_time(time),
                                               table.map_window(CONV_ID(wid)),
                                               (data->m_version < 1008)
                                                 ? table.map_communicator(CONV_ID(cid))
                                                 : table.map_group(CONV_ID(cid)),
                                               synex);

  if (event)
    trace.add_event(event);

  event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_WINCOLLEXIT(elg_ui4  lid,    /* location identifier */
                                elg_d8   time,   /* time stamp */
                                elg_ui1  metc,   /* number of metrics */
                                elg_ui8* metv,   /* metric values */
                                elg_ui4  wid,    /* window identifier */
                                void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createMpiRmaCollEnd(defs,
                                                  table.map_time(time),
                                                  table.map_window(CONV_ID(wid)));

  if (event)
    trace.add_event(event);

  // Create event
  event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_WIN_LOCK(elg_ui4 lid,     /* location identifier */
                             elg_d8  time,    /* time stamp */
                             elg_ui4 llid,    /* lock location identifier */
                             elg_ui4 wid,     /* window identifier */
                             elg_ui1 ltype,   /* lock type */
                             void*   userdata)
{
  CALLBACK_SETUP

  // Map window ID
  ident_t win_id = table.map_window(CONV_ID(wid));

  // Pre-1.8 traces: convert global |-> local remote rank
  if (data->m_version < 1008)
    llid = defs.get_window(win_id)->get_comm()->getGroup().getLocalRank(CONV_ID(llid));

  Event_rep* event =
    EventFactory::instance()->createMpiRmaLock(defs,
                                               table.map_time(time),
                                               llid,
                                               win_id,
                                               ltype);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_MPI_WIN_UNLOCK(elg_ui4 lid,    /* location identifier */
                               elg_d8  time,   /* time stamp */
                               elg_ui4 llid,   /* lock location identifier */
                               elg_ui4 wid,    /* window identifier */
                               void*   userdata)
{
  CALLBACK_SETUP

  // Map window ID
  ident_t win_id = table.map_window(CONV_ID(wid));

  // Pre-1.8 traces: convert global |-> local remote rank
  if (data->m_version < 1008)
    llid = defs.get_window(win_id)->get_comm()->getGroup().getLocalRank(CONV_ID(llid));

  Event_rep* event =
    EventFactory::instance()->createMpiRmaUnlock(defs,
                                                 table.map_time(time),
                                                 llid,
                                                 win_id);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


//--- Generic one-sided event records ---------------------------------------

void elg_readcb_PUT_1TS(elg_ui4 lid,      /* location identifier */
                        elg_d8  time,     /* time stamp */
                        elg_ui4 dlid,     /* destination location identifier */
                        elg_ui4 rmaid,    /* RMA operation identifier */
                        elg_ui4 nbytes,   /* message length in bytes */
                        void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createRmaPutStart(defs,
                                                table.map_time(time),
                                                rmaid,
                                                dlid,
                                                nbytes);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_PUT_1TE(elg_ui4 lid,     /* location identifier */
                        elg_d8  time,    /* time stamp */
                        elg_ui4 slid,    /* source location identifier */
                        elg_ui4 rmaid,   /* RMA operation identifier */
                        void*   userdata)
{
  /* NOTE: As we are no longer rewriting the trace, these events will
   *       never exist in the remote trace. We work directly on the
   *       _REMOTE events. Eventually, these event will deprecate. */
}


void elg_readcb_PUT_1TE_REMOTE(elg_ui4 lid,     /* location identifier */
                               elg_d8  time,    /* time stamp */
                               elg_ui4 dlid,    /* destination location identifier */
                               elg_ui4 rmaid,   /* RMA operation identifier */
                               void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createRmaPutEnd(defs,
                                              table.map_time(time),
                                              rmaid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_GET_1TS(elg_ui4 lid,      /* location identifier */
                        elg_d8  time,     /* time stamp */
                        elg_ui4 tlid,     /* target location identifier */
                        elg_ui4 rmaid,    /* RMA operation identifier */
                        elg_ui4 nbytes,   /* message length in bytes */
                        void*   userdata)
{
  /* NOTE: As we are no longer rewriting the trace, these events will
   *       never exist in the remote trace. We work directly on the
   *       _REMOTE events. Eventually, these event will deprecate.*/
}


void elg_readcb_GET_1TS_REMOTE(elg_ui4 lid,      /* location identifier */
                               elg_d8  time,     /* time stamp */
                               elg_ui4 tlid,     /* target location identifier */
                               elg_ui4 rmaid,    /* RMA operation identifier */
                               elg_ui4 nbytes,   /* message length in bytes */
                               void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createRmaGetStart(defs,
                                                table.map_time(time),
                                                rmaid,
                                                tlid,
                                                nbytes);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_GET_1TE(elg_ui4 lid,     /* location identifier */
                        elg_d8  time,    /* time stamp */
                        elg_ui4 slid,    /* source location identifier */
                        elg_ui4 rmaid,   /* RMA operation identifier */
                        void*   userdata)
{
  /* NOTE: In pearl most of the data provided in the event can be
   *       obtained with a reference to the Start event, using the
   *       rma_id. */
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createRmaGetEnd(defs,
                                              table.map_time(time),
                                              rmaid);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_COLLEXIT(elg_ui4  lid,     /* location identifier */
                         elg_d8   time,    /* time stamp */
                         elg_ui1  metc,    /* number of metrics */
                         elg_ui8* metv,    /* metric values */
                         elg_ui4  rlid,    /* root location identifier */
                         elg_ui4  cid,     /* communicator identifier */
                         elg_ui4  sent,    /* bytes sent */
                         elg_ui4  recvd,   /* bytes received */
                         void*    userdata)
{
  /*
   * One-sided communication is currently ignored. However, EXIT events
   * need to be handled correctly.
   */

  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_ALOCK(elg_ui4 lid,    /* location identifier */
                      elg_d8  time,   /* time stamp */
                      elg_ui4 lkid,   /* identifier of the lock being aquired */
                      void*   userdata)
{
  /* One-sided communication is currently ignored. */
}


void elg_readcb_RLOCK(elg_ui4 lid,    /* location identifier */
                      elg_d8  time,   /* time stamp */
                      elg_ui4 lkid,   /* identifier of the lock being aquired */
                      void*   userdata)
{
  /* One-sided communication is currently ignored. */
}


//--- OpenMP event records --------------------------------------------------

void elg_readcb_OMP_FORK(elg_ui4 lid,    /* location identifier */
                         elg_d8  time,   /* time stamp */
                         void*   userdata)
{
  CALLBACK_SETUP

  uint32_t teamSize = defs.getLocation(data->m_location).getParent()->numLocations();
  Event_rep* event =
    EventFactory::instance()->createOmpFork(defs,
                                            table.map_time(time),
                                            teamSize);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_OMP_JOIN(elg_ui4 lid,    /* location identifier */
                         elg_d8  time,   /* time stamp */
                         void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createOmpJoin(defs,
                                            table.map_time(time));

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_OMP_ALOCK(elg_ui4 lid,    /* location identifier */
                          elg_d8  time,   /* time stamp */
                          elg_ui4 lkid,   /* identifier of the lock being acquired */
                          void*   userdata)
{
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createOmpAcquireLock(defs,
                                                   table.map_time(time),
                                                   lkid,
                                                   PEARL_NO_ID);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_OMP_RLOCK(elg_ui4 lid,    /* location identifier */
                          elg_d8  time,   /* time stamp */
                          elg_ui4 lkid,   /* identifier of the lock being acquired */
                          void*   userdata)
{  
  CALLBACK_SETUP

  Event_rep* event =
    EventFactory::instance()->createOmpReleaseLock(defs,
                                                   table.map_time(time),
                                                   lkid,
                                                   PEARL_NO_ID);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_OMP_COLLEXIT(elg_ui4  lid,    /* location identifier */
                             elg_d8   time,   /* time stamp */
                             elg_ui1  metc,   /* number of metrics */
                             elg_ui8* metv,   /* metric values */
                             void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


//--- Other event records ---------------------------------------------------

void elg_readcb_LOG_OFF(elg_ui4  lid,    /* location identifier */
                        elg_d8   time,   /* time stamp */
                        elg_ui1  metc,   /* number of metrics */
                        elg_ui8* metv,   /* metric values */
                        void*    userdata)
{
  CALLBACK_SETUP

  // Callstack update
  uint32_t regionId = defs.getPausingRegion().getId();
  data->m_callstack.push(regionId);

  Event_rep* event =
    EventFactory::instance()->createEnter(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_LOG_ON(elg_ui4  lid,    /* location identifier */
                       elg_d8   time,   /* time stamp */
                       elg_ui1  metc,   /* number of metrics */
                       elg_ui8* metv,   /* metric values */
                       void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_ENTER_TRACING(elg_ui4  lid,    /* location identifier */
                              elg_d8   time,   /* time stamp */
                              elg_ui1  metc,   /* number of metrics */
                              elg_ui8* metv,   /* metric values */
                              void*    userdata)
{
  CALLBACK_SETUP

  // Callstack update
  uint32_t regionId = defs.getFlushingRegion().getId();
  data->m_callstack.push(regionId);

  Event_rep* event =
    EventFactory::instance()->createEnter(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}


void elg_readcb_EXIT_TRACING(elg_ui4  lid,    /* location identifier */
                             elg_d8   time,   /* time stamp */
                             elg_ui1  metc,   /* number of metrics */
                             elg_ui8* metv,   /* metric values */
                             void*    userdata)
{
  CALLBACK_SETUP

  // Consistency check and callstack update
  if (data->m_callstack.empty())
    throw FatalError("Unbalanced ENTER/LEAVE events (Too many LEAVEs).");
  uint32_t regionId = data->m_callstack.top();
  data->m_callstack.pop();

  // Create event
  Event_rep* event =
    EventFactory::instance()->createLeave(defs,
                                          table.map_time(time),
                                          regionId,
                                          metv);

  if (event)
    trace.add_event(event);

  CALLBACK_CLEANUP
}

#undef CALLBACK_SETUP
#undef CALLBACK_CLEANUP
