/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#define __STDC_LIMIT_MACROS

#include <config.h>
#include "Otf2Archive.h"

#include <cassert>
#include <cstddef>

#include <pearl/Callsite.h>
#include <pearl/Error.h>
#include <pearl/GlobalDefs.h>
#include <pearl/LocalTrace.h>
#include <pearl/LocationGroup.h>
#include <pearl/ProcessGroup.h>
#include <pearl/MpiCollEnd_rep.h>
#include <pearl/Region.h>
#include <pearl/String.h>

#include "DefsFactory.h"
#include "EventFactory.h"
#include "Process.h"

using namespace std;
using namespace pearl;


/*-------------------------------------------------------------------------*/
/**
 *  @file    Otf2Archive.cpp
 *  @ingroup PEARL_base
 *  @brief   Implementation of the class Otf2Archive.
 *
 *  This file provides the implementation of the class Otf2Archive and
 *  related functions.
 **/
/*-------------------------------------------------------------------------*/

//--- Local helper functions & data structures ------------------------------

namespace
{
struct CallbackData
{
    GlobalDefs* mDefinitions;
    LocalTrace* mTrace;
    string      mMessage;
    uint64_t    mTimerResolution;
    uint64_t    mGlobalOffset;
};


extern "C" {
    // *INDENT-OFF*
    // Conversion routines
    Region::Role convertRegionRole(OTF2_RegionRole role);
    Region::Paradigm convertParadigm(OTF2_Paradigm paradigm);
    LocationGroup::Type convertLocationGroupType(OTF2_LocationGroupType type);
    Location::Type convertLocationType(OTF2_LocationType type);
    timestamp_t convertTime(uint64_t timestamp, const CallbackData* const data);
    MpiCollEnd_rep::coll_type convertCollectiveType(OTF2_CollectiveOp type);

    // SION callbacks
    OTF2_ErrorCode sionCbGetRank(void*         userdata,
                                 OTF2_FileType fileType,
                                 uint64_t      locationId,
                                 int32_t*      rank);
    static OTF2_FileSionCallbacks sion_callbacks = { NULL, NULL, sionCbGetRank };

    // Definition callbacks
    OTF2_CallbackCode defCbCallpath      (void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               parentId,
                                          uint32_t               regionId);
    OTF2_CallbackCode defCbCallsite      (void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               filenameId,
                                          uint32_t               line,
                                          uint32_t               enteredId,
                                          uint32_t               leftId);
    OTF2_CallbackCode defCbClockProps    (void*                  userdata,
                                          uint64_t               resolution,
                                          uint64_t               offset,
                                          uint64_t               length);
    OTF2_CallbackCode defCbGroup         (void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               nameId,
                                          OTF2_GroupType         type,
                                          OTF2_Paradigm          paradigm,
                                          OTF2_GroupFlag         groupFlags,
                                          uint32_t               size,
                                          const uint64_t*        members);
    OTF2_CallbackCode defCbLocation      (void*                  userdata,
                                          uint64_t               id,
                                          uint32_t               nameId,
                                          OTF2_LocationType      type,
                                          uint64_t               numEvents,
                                          uint32_t               parentId);
    OTF2_CallbackCode defCbLocationGroup (void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               nameId,
                                          OTF2_LocationGroupType type,
                                          uint32_t               parentId);
    OTF2_CallbackCode defCbComm          (void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               nameId,
                                          uint32_t               groupId,
                                          uint32_t               parentId);
    OTF2_CallbackCode defCbRegion        (void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               nameId,
                                          uint32_t               canonicalNameId,
                                          uint32_t               descriptionId,
                                          OTF2_RegionRole        role,
                                          OTF2_Paradigm          paradigm,
                                          OTF2_RegionFlag        flags,
                                          uint32_t               filenameId,
                                          uint32_t               startLine,
                                          uint32_t               endLine);
    OTF2_CallbackCode defCbString        (void*                  userdata,
                                          uint32_t               id,
                                          const char*            str);
    OTF2_CallbackCode defCbSystemTreeNode(void*                  userdata,
                                          uint32_t               id,
                                          uint32_t               nameId,
                                          uint32_t               classNameId,
                                          uint32_t               parentId);


    // Event callbacks
    OTF2_CallbackCode evtCbFlush         (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             stoptime);
    OTF2_CallbackCode evtCbEnter         (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             regionId);
    OTF2_CallbackCode evtCbLeave         (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             regionId);
    OTF2_CallbackCode evtCbMeasureOnOff  (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          OTF2_MeasurementMode mode);
    OTF2_CallbackCode evtCbMpiCollBegin  (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList);
    OTF2_CallbackCode evtCbMpiCollEnd    (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          OTF2_CollectiveOp    type,
                                          uint32_t             communicatorId,
                                          uint32_t             root,
                                          uint64_t             bytesSent,
                                          uint64_t             bytesReceived);
    OTF2_CallbackCode evtCbMpiIrecv      (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             sender,
                                          uint32_t             communicatorId,
                                          uint32_t             tag,
                                          uint64_t             bytesReceived,
                                          uint64_t             requestId);
    OTF2_CallbackCode evtCbMpiIrecvReq   (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             requestId);
    OTF2_CallbackCode evtCbMpiIsend      (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             receiver,
                                          uint32_t             communicatorId,
                                          uint32_t             tag,
                                          uint64_t             bytesSent,
                                          uint64_t             requestId);
    OTF2_CallbackCode evtCbMpiIsendComp  (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             requestId);
    OTF2_CallbackCode evtCbMpiRecv       (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             sender,
                                          uint32_t             communicatorId,
                                          uint32_t             tag,
                                          uint64_t             bytesReceived);
    OTF2_CallbackCode evtCbMpiCancelled  (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             requestId);
    OTF2_CallbackCode evtCbMpiReqTested  (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             requestId);
    OTF2_CallbackCode evtCbMpiSend       (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             receiver,
                                          uint32_t             communicatorId,
                                          uint32_t             tag,
                                          uint64_t             bytesSent);
    OTF2_CallbackCode evtCbOmpAcquireLock(uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             lockId,
                                          uint32_t             order);
    OTF2_CallbackCode evtCbOmpFork       (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             numThreads);
    OTF2_CallbackCode evtCbOmpJoin       (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList);
    OTF2_CallbackCode evtCbOmpReleaseLock(uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint32_t             lockId,
                                          uint32_t             order);
    OTF2_CallbackCode evtCbOmpTaskCompl  (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             taskId);
    OTF2_CallbackCode evtCbOmpTaskCreate (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             taskId);
    OTF2_CallbackCode evtCbOmpTaskSwitch (uint64_t             locationId,
                                          uint64_t             timestamp,
                                          uint64_t             index,
                                          void*                userdata,
                                          OTF2_AttributeList*  attributeList,
                                          uint64_t             taskId);
    // *INDENT-ON*
}

}   // unnamed namespace


//--- Constructors & destructor ---------------------------------------------

Otf2Archive::Otf2Archive(const string& anchorName,
                         const string& archiveDir)
    : TraceArchive(anchorName, archiveDir),
      mArchiveReader(NULL),
      mTimerResolution(0),
      mGlobalOffset(0)
{
}


Otf2Archive::~Otf2Archive()
{
    OTF2_Reader_Close(mArchiveReader);
}


//--- Private methods -------------------------------------------------------

void
Otf2Archive::open()
{
    mArchiveReader = OTF2_Reader_Open(getAnchorName().c_str());
    if (!mArchiveReader)
        throw RuntimeError("Error opening OTF2 experiment archive \"" +
                           getAnchorName() + "\"");

    // Register SION callbacks
    OTF2_Reader_SetFileSionCallbacks(mArchiveReader, &sion_callbacks, NULL);
}


void
Otf2Archive::readDefinitions(GlobalDefs* defs)
{
    // Create definition reader
    OTF2_GlobalDefReader* defReader = OTF2_Reader_GetGlobalDefReader(mArchiveReader);
    if (!defReader)
        throw RuntimeError("Error creating global definition reader!");

    // Create & fill callback data structure
    OTF2_GlobalDefReaderCallbacks* callbacks = OTF2_GlobalDefReaderCallbacks_New();
    if (!callbacks)
        throw RuntimeError("Error allocating OTF2 callback data structure!");
    OTF2_GlobalDefReaderCallbacks_SetCallpathCallback(callbacks, defCbCallpath);
    OTF2_GlobalDefReaderCallbacks_SetCallsiteCallback(callbacks, defCbCallsite);
    OTF2_GlobalDefReaderCallbacks_SetClockPropertiesCallback(callbacks, defCbClockProps);
    OTF2_GlobalDefReaderCallbacks_SetGroupCallback(callbacks, defCbGroup);
    OTF2_GlobalDefReaderCallbacks_SetLocationCallback(callbacks, defCbLocation);
    OTF2_GlobalDefReaderCallbacks_SetLocationGroupCallback(callbacks, defCbLocationGroup);
    OTF2_GlobalDefReaderCallbacks_SetCommCallback(callbacks, defCbComm);
    OTF2_GlobalDefReaderCallbacks_SetRegionCallback(callbacks, defCbRegion);
    OTF2_GlobalDefReaderCallbacks_SetStringCallback(callbacks, defCbString);
    OTF2_GlobalDefReaderCallbacks_SetSystemTreeNodeCallback(callbacks, defCbSystemTreeNode);

    // Install callbacks
    CallbackData cbData;
    cbData.mDefinitions = defs;
    OTF2_GlobalDefReader_SetCallbacks(defReader, callbacks, &cbData);
    OTF2_GlobalDefReaderCallbacks_Delete(callbacks);

    // Read definition data
    uint64_t recordsRead;
    OTF2_ErrorCode result = OTF2_GlobalDefReader_ReadDefinitions(defReader, UINT64_MAX, &recordsRead);
    if (OTF2_SUCCESS != result || !cbData.mMessage.empty())
        throw RuntimeError(cbData.mMessage);

    // Store timer data
    mTimerResolution = cbData.mTimerResolution;
    mGlobalOffset    = cbData.mGlobalOffset;

    // Close definition reader
    result = OTF2_Reader_CloseGlobalDefReader(mArchiveReader, defReader);
    if (OTF2_SUCCESS != result)
        throw RuntimeError("Error closing global definition reader!");

    // For OTF2 traces, global time offset subtraction is implicitly
    // applied during trace reading
    defs->setGlobalOffset(0.0);

    // Re-register SION callbacks to provide global definitions as userdata
    OTF2_Reader_SetFileSionCallbacks(mArchiveReader, &sion_callbacks, defs);
}


/// @todo Suppress error messages thrown by OTF2 if optional local definition
///       file could not be found
void
Otf2Archive::readTrace(const GlobalDefs& defs,
                       const Location&   location,
                       LocalTrace*       trace)
{
    // Create event trace reader
    mMutex.lock();
    OTF2_EvtReader* evtReader = OTF2_Reader_GetEvtReader(mArchiveReader, location.getId());
    OTF2_DefReader* defReader = OTF2_Reader_GetDefReader(mArchiveReader, location.getId());
    mMutex.unlock();
    if (!evtReader)
        throw RuntimeError("Error creating local trace reader!");

    // NOTE: The local definition file is optional, i.e., the creation of the
    // local definition reader may fail. We therefore do not raise an error
    // in this case.

    // Create and fill callback data structure
    OTF2_EvtReaderCallbacks* callbacks = OTF2_EvtReaderCallbacks_New();
    if (!callbacks)
        throw RuntimeError("Error allocating OTF2 callback data structure!");
    OTF2_EvtReaderCallbacks_SetBufferFlushCallback(callbacks, evtCbFlush);
    OTF2_EvtReaderCallbacks_SetEnterCallback(callbacks, evtCbEnter);
    OTF2_EvtReaderCallbacks_SetLeaveCallback(callbacks, evtCbLeave);
    OTF2_EvtReaderCallbacks_SetMeasurementOnOffCallback(callbacks, evtCbMeasureOnOff);
    OTF2_EvtReaderCallbacks_SetMpiCollectiveBeginCallback(callbacks, evtCbMpiCollBegin);
    OTF2_EvtReaderCallbacks_SetMpiCollectiveEndCallback(callbacks, evtCbMpiCollEnd);
    OTF2_EvtReaderCallbacks_SetMpiIrecvCallback(callbacks, evtCbMpiIrecv);
    OTF2_EvtReaderCallbacks_SetMpiIrecvRequestCallback(callbacks, evtCbMpiIrecvReq);
    OTF2_EvtReaderCallbacks_SetMpiIsendCallback(callbacks, evtCbMpiIsend);
    OTF2_EvtReaderCallbacks_SetMpiIsendCompleteCallback(callbacks, evtCbMpiIsendComp);
    OTF2_EvtReaderCallbacks_SetMpiRecvCallback(callbacks, evtCbMpiRecv);
    OTF2_EvtReaderCallbacks_SetMpiRequestCancelledCallback(callbacks, evtCbMpiCancelled);
    OTF2_EvtReaderCallbacks_SetMpiRequestTestCallback(callbacks, evtCbMpiReqTested);
    OTF2_EvtReaderCallbacks_SetMpiSendCallback(callbacks, evtCbMpiSend);
    OTF2_EvtReaderCallbacks_SetOmpAcquireLockCallback(callbacks, evtCbOmpAcquireLock);
    OTF2_EvtReaderCallbacks_SetOmpForkCallback(callbacks, evtCbOmpFork);
    OTF2_EvtReaderCallbacks_SetOmpJoinCallback(callbacks, evtCbOmpJoin);
    OTF2_EvtReaderCallbacks_SetOmpReleaseLockCallback(callbacks, evtCbOmpReleaseLock);
    OTF2_EvtReaderCallbacks_SetOmpTaskCompleteCallback(callbacks, evtCbOmpTaskCompl);
    OTF2_EvtReaderCallbacks_SetOmpTaskCreateCallback(callbacks, evtCbOmpTaskCreate);
    OTF2_EvtReaderCallbacks_SetOmpTaskSwitchCallback(callbacks, evtCbOmpTaskSwitch);

    // Install callbacks
    CallbackData cbData;
    cbData.mDefinitions     = const_cast<GlobalDefs*>(&defs);
    cbData.mTrace           = trace;
    cbData.mTimerResolution = mTimerResolution;
    cbData.mGlobalOffset    = mGlobalOffset;
    OTF2_EvtReader_SetCallbacks(evtReader, callbacks, &cbData);
    OTF2_EvtReaderCallbacks_Delete(callbacks);

    // Read local definition data
    uint64_t       recordsRead;
    OTF2_ErrorCode result;
    if (defReader) {    
        mMutex.lock();
        result = OTF2_DefReader_ReadDefinitions(defReader, UINT64_MAX,
                                                &recordsRead);
        OTF2_Reader_CloseDefReader(mArchiveReader, defReader);
        mMutex.unlock();
        if (OTF2_SUCCESS != result)
            throw RuntimeError("Error reading local definition data!");
    }

    // Read event trace data
    result = OTF2_EvtReader_ReadEvents(evtReader, UINT64_MAX, &recordsRead);
    if (OTF2_SUCCESS != result || !cbData.mMessage.empty())
        throw RuntimeError(cbData.mMessage);
    mMutex.lock();
    OTF2_Reader_CloseEvtReader(mArchiveReader, evtReader);
    mMutex.unlock();
}


//--- Conversion routines ---------------------------------------------------

namespace
{

Region::Role
convertRegionRole(OTF2_RegionRole role)
{
    switch (role) {
        case OTF2_REGION_ROLE_UNKNOWN:
            return Region::ROLE_UNKNOWN;

        case OTF2_REGION_ROLE_FUNCTION:
            return Region::ROLE_FUNCTION;

        case OTF2_REGION_ROLE_WRAPPER:
            return Region::ROLE_WRAPPER;

        case OTF2_REGION_ROLE_LOOP:
            return Region::ROLE_LOOP;

        case OTF2_REGION_ROLE_CODE:
            return Region::ROLE_CODE;

        case OTF2_REGION_ROLE_PARALLEL:
            return Region::ROLE_PARALLEL;

        case OTF2_REGION_ROLE_SECTIONS:
            return Region::ROLE_SECTIONS;

        case OTF2_REGION_ROLE_SECTION:
            return Region::ROLE_SECTION;

        case OTF2_REGION_ROLE_WORKSHARE:
            return Region::ROLE_WORKSHARE;

        case OTF2_REGION_ROLE_SINGLE:
            return Region::ROLE_SINGLE;

        case OTF2_REGION_ROLE_SINGLE_SBLOCK:
            return Region::ROLE_SINGLE_SBLOCK;

        case OTF2_REGION_ROLE_MASTER:
            return Region::ROLE_MASTER;

        case OTF2_REGION_ROLE_CRITICAL:
            return Region::ROLE_CRITICAL;

        case OTF2_REGION_ROLE_CRITICAL_SBLOCK:
            return Region::ROLE_CRITICAL_SBLOCK;

        case OTF2_REGION_ROLE_ATOMIC:
            return Region::ROLE_ATOMIC;

        case OTF2_REGION_ROLE_BARRIER:
            return Region::ROLE_BARRIER;

        case OTF2_REGION_ROLE_IMPLICIT_BARRIER:
            return Region::ROLE_IMPLICIT_BARRIER;

        case OTF2_REGION_ROLE_FLUSH:
            return Region::ROLE_FLUSH;

        case OTF2_REGION_ROLE_ORDERED:
            return Region::ROLE_ORDERED;

        case OTF2_REGION_ROLE_ORDERED_SBLOCK:
            return Region::ROLE_ORDERED_SBLOCK;

        case OTF2_REGION_ROLE_TASK:
            return Region::ROLE_TASK;

        case OTF2_REGION_ROLE_TASK_CREATE:
            return Region::ROLE_TASK_CREATE;

        case OTF2_REGION_ROLE_TASK_WAIT:
            return Region::ROLE_TASK_WAIT;

        case OTF2_REGION_ROLE_COLL_ONE2ALL:
            return Region::ROLE_COLL_ONE2ALL;

        case OTF2_REGION_ROLE_COLL_ALL2ONE:
            return Region::ROLE_COLL_ALL2ONE;

        case OTF2_REGION_ROLE_COLL_ALL2ALL:
            return Region::ROLE_COLL_ALL2ALL;

        case OTF2_REGION_ROLE_COLL_OTHER:
            return Region::ROLE_COLL_OTHER;

        case OTF2_REGION_ROLE_FILE_IO:
            return Region::ROLE_FILE_IO;

        case OTF2_REGION_ROLE_POINT2POINT:
            return Region::ROLE_POINT2POINT;

        case OTF2_REGION_ROLE_RMA:
            return Region::ROLE_RMA;

        case OTF2_REGION_ROLE_DATA_TRANSFER:
            return Region::ROLE_DATA_TRANSFER;

        case OTF2_REGION_ROLE_ARTIFICIAL:
            return Region::ROLE_ARTIFICIAL;

        default:
            break;
    }

    return Region::ROLE_UNKNOWN;
}


Region::Paradigm
convertParadigm(OTF2_Paradigm paradigm)
{
    switch (paradigm) {
        case OTF2_PARADIGM_UNKNOWN:
            return Region::PARADIGM_UNKNOWN;

        case OTF2_PARADIGM_USER:
            return Region::PARADIGM_USER;

        case OTF2_PARADIGM_COMPILER:
            return Region::PARADIGM_COMPILER;

        case OTF2_PARADIGM_OPENMP:
            return Region::PARADIGM_OPENMP;

        case OTF2_PARADIGM_MPI:
            return Region::PARADIGM_MPI;

        case OTF2_PARADIGM_CUDA:
            return Region::PARADIGM_CUDA;

        case OTF2_PARADIGM_MEASUREMENT_SYSTEM:
            return Region::PARADIGM_MEASUREMENT_SYSTEM;

        default:
            break;
    }

    return Region::PARADIGM_UNKNOWN;
}


LocationGroup::Type
convertLocationGroupType(OTF2_LocationGroupType type)
{
    switch (type) {
        case OTF2_LOCATION_GROUP_TYPE_PROCESS:
            return LocationGroup::TYPE_PROCESS;

        default:
            break;
    }

    return LocationGroup::TYPE_UNKNOWN;
}


Location::Type
convertLocationType(OTF2_LocationType type)
{
    switch (type) {
        case OTF2_LOCATION_TYPE_CPU_THREAD:
            return Location::TYPE_CPU_THREAD;

        case OTF2_LOCATION_TYPE_GPU:
            return Location::TYPE_GPU;

        case OTF2_LOCATION_TYPE_METRIC:
            return Location::TYPE_METRIC;

        default:
            break;
    }

    return Location::TYPE_UNKNOWN;
}


timestamp_t
convertTime(uint64_t                  timestamp,
            const CallbackData* const data)
{
    return static_cast<double>(timestamp - data->mGlobalOffset)
           / data->mTimerResolution;
}


MpiCollEnd_rep::coll_type
convertCollectiveType(OTF2_CollectiveOp type)
{
    switch (type) {
        case OTF2_COLLECTIVE_OP_BARRIER:
            return MpiCollEnd_rep::BARRIER;

        case OTF2_COLLECTIVE_OP_BCAST:
            return MpiCollEnd_rep::BCAST;

        case OTF2_COLLECTIVE_OP_GATHER:
            return MpiCollEnd_rep::GATHER;

        case OTF2_COLLECTIVE_OP_GATHERV:
            return MpiCollEnd_rep::GATHERV;

        case OTF2_COLLECTIVE_OP_SCATTER:
            return MpiCollEnd_rep::SCATTER;

        case OTF2_COLLECTIVE_OP_SCATTERV:
            return MpiCollEnd_rep::SCATTERV;

        case OTF2_COLLECTIVE_OP_ALLGATHER:
            return MpiCollEnd_rep::ALLGATHER;

        case OTF2_COLLECTIVE_OP_ALLGATHERV:
            return MpiCollEnd_rep::ALLGATHERV;

        case OTF2_COLLECTIVE_OP_ALLTOALL:
            return MpiCollEnd_rep::ALLTOALL;

        case OTF2_COLLECTIVE_OP_ALLTOALLV:
            return MpiCollEnd_rep::ALLTOALLV;

        case OTF2_COLLECTIVE_OP_ALLTOALLW:
            return MpiCollEnd_rep::ALLTOALLW;

        case OTF2_COLLECTIVE_OP_ALLREDUCE:
            return MpiCollEnd_rep::ALLREDUCE;

        case OTF2_COLLECTIVE_OP_REDUCE:
            return MpiCollEnd_rep::REDUCE;

        case OTF2_COLLECTIVE_OP_REDUCE_SCATTER:
            return MpiCollEnd_rep::REDUCE_SCATTER;

        case OTF2_COLLECTIVE_OP_SCAN:
            return MpiCollEnd_rep::SCAN;

        case OTF2_COLLECTIVE_OP_EXSCAN:
            return MpiCollEnd_rep::EXSCAN;

        case OTF2_COLLECTIVE_OP_REDUCE_SCATTER_BLOCK:
            return MpiCollEnd_rep::REDUCE_SCATTER_BLOCK;
    }

    // We should never reach this point...
    assert(false);
    return MpiCollEnd_rep::NUMBER_OF_COLL_TYPES;
}


//--- SION callbacks --------------------------------------------------------

OTF2_ErrorCode
sionCbGetRank(void*         userdata,
              OTF2_FileType fileType,
              uint64_t      locationId,
              int32_t*      rank)
{
    GlobalDefs* defs = static_cast<GlobalDefs*>(userdata);

    try {
        const Location& location = defs->getLocation(locationId);
        *rank = location.getRank();
    }
    catch (std::exception& ex) {
        return OTF2_ERROR_INDEX_OUT_OF_BOUNDS;
    }

    return OTF2_SUCCESS;
}


//--- Definition callbacks --------------------------------------------------

#define CALLBACK_SETUP \
        CallbackData* data  = static_cast<CallbackData*>(userdata); \
        GlobalDefs*   defs  = data->mDefinitions; \
        LocalTrace*   trace = data->mTrace; \
        try {

#define CALLBACK_CLEANUP \
        } /* Closes the try block */ \
        catch (std::exception& ex) { \
            data->mMessage = ex.what(); \
            return OTF2_CALLBACK_INTERRUPT; \
        }


OTF2_CallbackCode
defCbCallpath(void*    userdata,
              uint32_t id,
              uint32_t parentId,
              uint32_t regionId)
{
    CALLBACK_SETUP
    DefsFactory::instance()->createCallpath(*defs,
                                            id,
                                            regionId,
                                            Callsite::NO_ID,
                                            parentId);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


/// @todo Handle @a leftId
OTF2_CallbackCode
defCbCallsite(void*    userdata,
              uint32_t id,
              uint32_t filenameId,
              uint32_t line,
              uint32_t enteredId,
              uint32_t leftId)
{
    CALLBACK_SETUP
    DefsFactory::instance()->createCallsite(*defs,
                                            id,
                                            filenameId,
                                            line,
                                            enteredId);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
defCbClockProps(void*    userdata,
                uint64_t resolution,
                uint64_t offset,
                uint64_t length)
{
    CALLBACK_SETUP
    data->mTimerResolution = resolution;
    data->mGlobalOffset    = offset;
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


/// @todo Handle location, region and metric groups
OTF2_CallbackCode
defCbGroup(void*           userdata,
           uint32_t        id,
           uint32_t        nameId,
           OTF2_GroupType  type,
           OTF2_Paradigm   paradigm,
           OTF2_GroupFlag  groupFlags,
           uint32_t        size,
           const uint64_t* members)
{
    CALLBACK_SETUP

    // Only handle MPI groups for now
    if (OTF2_PARADIGM_MPI != paradigm)
        return OTF2_CALLBACK_SUCCESS;

    // Case 1: MPI group
    if (OTF2_GROUP_TYPE_COMM_GROUP == type
        || OTF2_GROUP_TYPE_COMM_SELF == type) {
        bool isSelf = (OTF2_GROUP_TYPE_COMM_SELF == type);

        // Set up ranks array
        DefsFactory::process_group ranks;
        ranks.reserve(size);
        for (uint32_t index = 0; index < size; ++index)
            ranks.push_back(members[index]);

        // Create group
        DefsFactory::instance()->createMpiGroup(*defs,
                                                id,
                                                nameId,
                                                ranks,
                                                isSelf,
                                                false);
    }

    // Case 2: MPI locations
    if (OTF2_GROUP_TYPE_COMM_LOCATIONS == type) {
        // Set rank of location groups
        for (uint32_t rank = 0; rank < size; ++rank) {
            uint64_t locationId = members[rank];

            const Location& location = defs->getLocation(locationId);
            Process*        process  = static_cast<Process*>(location.getParent());
            process->setRank(rank);
        }
    }

    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


/// @todo Handle non-CPU type locations
OTF2_CallbackCode
defCbLocation(void*             userdata,
              uint64_t          id,
              uint32_t          nameId,
              OTF2_LocationType type,
              uint64_t          numEvents,
              uint32_t          parentId)
{
    CALLBACK_SETUP

    // Sanity check
    if (OTF2_LOCATION_TYPE_CPU_THREAD != type)
        throw RuntimeError("Locations of non-CPU type not yet supported!");

    DefsFactory::instance()->createLocation(*defs,
                                            id,
                                            nameId,
                                            convertLocationType(type),
                                            numEvents,
                                            parentId);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
defCbLocationGroup(void*                  userdata,
                   uint32_t               id,
                   uint32_t               nameId,
                   OTF2_LocationGroupType type,
                   uint32_t               parentId)
{
    CALLBACK_SETUP
    DefsFactory::instance()->createLocationGroup(*defs,
                                                 id,
                                                 nameId,
                                                 convertLocationGroupType(type),
                                                 parentId);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
defCbComm(void*    userdata,
          uint32_t id,
          uint32_t nameId,
          uint32_t groupId,
          uint32_t parentId)
{
    CALLBACK_SETUP

    // Create communicator if associated group exists (only MPI groups are
    // handled right now => only create MPI communicators)
    try {
        const ProcessGroup& group = defs->getProcessGroup(groupId);
        DefsFactory::instance()->createMpiComm(*defs,
                                               id,
                                               nameId,
                                               group.getId(),
                                               parentId);
    }
    catch (const RuntimeError& ex) {
        // Unfortunately, we can't distinguish between an invalid ID
        // and a group that was ignored during reading...
        return OTF2_CALLBACK_SUCCESS;
    }
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


/// @todo Handle region flags
OTF2_CallbackCode
defCbRegion(void*           userdata,
            uint32_t        id,
            uint32_t        nameId,
            uint32_t        canonicalNameId,
            uint32_t        descriptionId,
            OTF2_RegionRole role,
            OTF2_Paradigm   paradigm,
            OTF2_RegionFlag flags,
            uint32_t        filenameId,
            uint32_t        startLine,
            uint32_t        endLine)
{
    CALLBACK_SETUP

    const String& name = defs->getString(nameId);
    const String& file = defs->getString(filenameId);

    // Handle special "MEASUREMENT OFF" region
    DefsFactory::InternalRegionType type = DefsFactory::USER_REGION;
    if (nameId == canonicalNameId
        && OTF2_REGION_ROLE_ARTIFICIAL == role
        && String::UNDEFINED == file
        && 0 == startLine
        && 0 == endLine) {
        if ("MEASUREMENT OFF" == name.getString()
            && OTF2_PARADIGM_USER == paradigm) {
            type = DefsFactory::PAUSING_REGION;
        } else if ("TRACE BUFFER FLUSH" == name.getString()
                   && OTF2_PARADIGM_MEASUREMENT_SYSTEM == paradigm) {
            type = DefsFactory::FLUSHING_REGION;
        }
    }

    DefsFactory::instance()->createRegion(*defs,
                                          id,
                                          canonicalNameId,
                                          nameId,
                                          descriptionId,
                                          convertRegionRole(role),
                                          convertParadigm(paradigm),
                                          filenameId,
                                          startLine,
                                          endLine,
                                          type);

    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
defCbString(void*       userdata,
            uint32_t    id,
            const char* str)
{
    CALLBACK_SETUP
    DefsFactory::instance()->createString(*defs,
                                          id,
                                          str);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
defCbSystemTreeNode(void*    userdata,
                    uint32_t id,
                    uint32_t nameId,
                    uint32_t classNameId,
                    uint32_t parentId)
{
    CALLBACK_SETUP
    DefsFactory::instance()->createSystemNode(*defs,
                                              id,
                                              nameId,
                                              classNameId,
                                              parentId);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


//--- Event callbacks -------------------------------------------------------

OTF2_CallbackCode
evtCbFlush(uint64_t            locationId,
           uint64_t            timestamp,
           uint64_t            index,
           void*               userdata,
           OTF2_AttributeList* attributeList,
           uint64_t            stoptime)
{
    CALLBACK_SETUP
    Region::IdType regionId = defs->getFlushingRegion().getId();
    Event_rep* event =
        EventFactory::instance()->createEnter(*defs,
                                              convertTime(timestamp, data),
                                              regionId,
                                              NULL);
    if (event)
        trace->add_event(event);
    event =
        EventFactory::instance()->createLeave(*defs,
                                              convertTime(stoptime, data),
                                              regionId,
                                              NULL);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


/// @todo Handle HWC metrics
OTF2_CallbackCode
evtCbEnter(uint64_t            locationId,
           uint64_t            timestamp,
           uint64_t            index,
           void*               userdata,
           OTF2_AttributeList* attributeList,
           uint32_t            regionId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createEnter(*defs,
                                              convertTime(timestamp, data),
                                              regionId,
                                              NULL);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


/// @todo Handle HWC metrics
OTF2_CallbackCode
evtCbLeave(uint64_t            locationId,
           uint64_t            timestamp,
           uint64_t            index,
           void*               userdata,
           OTF2_AttributeList* attributeList,
           uint32_t            regionId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createLeave(*defs,
                                              convertTime(timestamp, data),
                                              regionId,
                                              NULL);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMeasureOnOff(uint64_t             locationId,
                  uint64_t             timestamp,
                  uint64_t             index,
                  void*                userdata,
                  OTF2_AttributeList*  attributeList,
                  OTF2_MeasurementMode mode)
{
    CALLBACK_SETUP

    Region::IdType regionId = defs->getPausingRegion().getId();
    Event_rep* event;
    if (OTF2_MEASUREMENT_OFF == mode) {
        event =
            EventFactory::instance()->createEnter(*defs,
                                                  convertTime(timestamp, data),
                                                  regionId,
                                                  NULL);
    } else {
        event =
            EventFactory::instance()->createLeave(*defs,
                                                  convertTime(timestamp, data),
                                                  regionId,
                                                  NULL);
    }
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiCollBegin(uint64_t            locationId,
                  uint64_t            timestamp,
                  uint64_t            index,
                  void*               userdata,
                  OTF2_AttributeList* attributeList)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiCollBegin(*defs,
                                                     convertTime(timestamp, data));
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiCollEnd(uint64_t            locationId,
                uint64_t            timestamp,
                uint64_t            index,
                void*               userdata,
                OTF2_AttributeList* attributeList,
                OTF2_CollectiveOp   type,
                uint32_t            communicatorId,
                uint32_t            root,
                uint64_t            bytesSent,
                uint64_t            bytesReceived)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiCollEnd(*defs,
                                                   convertTime(timestamp, data),
                                                   convertCollectiveType(type),
                                                   communicatorId,
                                                   root,
                                                   bytesSent,
                                                   bytesReceived);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiIrecv(uint64_t               locationId,
              uint64_t               timestamp,
              uint64_t               index,
              void*                  userdata,
              OTF2_AttributeList*    attributeList,
              uint32_t               sender,
              uint32_t               communicatorId,
              uint32_t               tag,
              uint64_t               bytesReceived,
              uint64_t               requestId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiRecvComplete(*defs,
                                                        convertTime(timestamp, data),
                                                        communicatorId,
                                                        sender,
                                                        tag,
                                                        bytesReceived,
                                                        requestId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiIrecvReq(uint64_t               locationId,
                 uint64_t               timestamp,
                 uint64_t               index,
                 void*                  userdata,
                 OTF2_AttributeList*    attributeList,
                 uint64_t               requestId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiRecvRequest(*defs,
                                                       convertTime(timestamp, data),
                                                       requestId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiIsend(uint64_t               locationId,
              uint64_t               timestamp,
              uint64_t               index,
              void*                  userdata,
              OTF2_AttributeList*    attributeList,
              uint32_t               receiver,
              uint32_t               communicatorId,
              uint32_t               tag,
              uint64_t               bytesSent,
              uint64_t               requestId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiSendRequest(*defs,
                                                       convertTime(timestamp, data),
                                                       communicatorId,
                                                       receiver,
                                                       tag,
                                                       bytesSent,
                                                       requestId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiIsendComp(uint64_t               locationId,
                  uint64_t               timestamp,
                  uint64_t               index,
                  void*                  userdata,
                  OTF2_AttributeList*    attributeList,
                  uint64_t               requestId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiSendComplete(*defs,
                                                        convertTime(timestamp, data),
                                                        requestId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiRecv(uint64_t               locationId,
             uint64_t               timestamp,
             uint64_t               index,
             void*                  userdata,
             OTF2_AttributeList*    attributeList,
             uint32_t               sender,
             uint32_t               communicatorId,
             uint32_t               tag,
             uint64_t               bytesReceived)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiRecv(*defs,
                                                convertTime(timestamp, data),
                                                communicatorId,
                                                sender,
                                                tag,
                                                bytesReceived);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiCancelled(uint64_t               locationId,
                  uint64_t               timestamp,
                  uint64_t               index,
                  void*                  userdata,
                  OTF2_AttributeList*    attributeList,
                  uint64_t               requestId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiCancelled(*defs,
                                                     convertTime(timestamp, data),
                                                     requestId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiReqTested(uint64_t               locationId,
                  uint64_t               timestamp,
                  uint64_t               index,
                  void*                  userdata,
                  OTF2_AttributeList*    attributeList,
                  uint64_t               requestId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiRequestTested(*defs,
                                                         convertTime(timestamp, data),
                                                         requestId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbMpiSend(uint64_t               locationId,
             uint64_t               timestamp,
             uint64_t               index,
             void*                  userdata,
             OTF2_AttributeList*    attributeList,
             uint32_t               receiver,
             uint32_t               communicatorId,
             uint32_t               tag,
             uint64_t               bytesSent)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createMpiSend(*defs,
                                                convertTime(timestamp, data),
                                                communicatorId,
                                                receiver,
                                                tag,
                                                bytesSent);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpAcquireLock(uint64_t               locationId,
                    uint64_t               timestamp,
                    uint64_t               index,
                    void*                  userdata,
                    OTF2_AttributeList*    attributeList,
                    uint32_t               lockId,
                    uint32_t               order)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpAcquireLock(*defs,
                                                       convertTime(timestamp, data),
                                                       lockId,
                                                       order);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpFork(uint64_t               locationId,
             uint64_t               timestamp,
             uint64_t               index,
             void*                  userdata,
             OTF2_AttributeList*    attributeList,
             uint32_t               numThreads)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpFork(*defs,
                                                convertTime(timestamp, data),
                                                numThreads);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpJoin(uint64_t               locationId,
             uint64_t               timestamp,
             uint64_t               index,
             void*                  userdata,
             OTF2_AttributeList*    attributeList)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpJoin(*defs,
                                                convertTime(timestamp, data));
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpReleaseLock(uint64_t               locationId,
                    uint64_t               timestamp,
                    uint64_t               index,
                    void*                  userdata,
                    OTF2_AttributeList*    attributeList,
                    uint32_t               lockId,
                    uint32_t               order)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpReleaseLock(*defs,
                                                       convertTime(timestamp, data),
                                                       lockId,
                                                       order);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpTaskCompl(uint64_t               locationId,
                  uint64_t               timestamp,
                  uint64_t               index,
                  void*                  userdata,
                  OTF2_AttributeList*    attributeList,
                  uint64_t               taskId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpTaskComplete(*defs,
                                                        convertTime(timestamp, data),
                                                        taskId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpTaskCreate(uint64_t               locationId,
                   uint64_t               timestamp,
                   uint64_t               index,
                   void*                  userdata,
                   OTF2_AttributeList*    attributeList,
                   uint64_t               taskId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpTaskCreate(*defs,
                                                      convertTime(timestamp, data),
                                                      taskId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


OTF2_CallbackCode
evtCbOmpTaskSwitch(uint64_t               locationId,
                   uint64_t               timestamp,
                   uint64_t               index,
                   void*                  userdata,
                   OTF2_AttributeList*    attributeList,
                   uint64_t               taskId)
{
    CALLBACK_SETUP
    Event_rep* event =
        EventFactory::instance()->createOmpTaskSwitch(*defs,
                                                      convertTime(timestamp, data),
                                                      taskId);
    if (event)
        trace->add_event(event);
    CALLBACK_CLEANUP

    return OTF2_CALLBACK_SUCCESS;
}


}   // unnamed namespace
