/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include "EventFactory.h"

#include <cstddef>
#ifdef PEARL_ENABLE_METRICS
  #include <cstring>
#endif

#include <pearl/Enter_rep.h>
#include <pearl/EnterCS_rep.h>
#include <pearl/Error.h>
#include <pearl/GlobalDefs.h>
#include <pearl/Leave_rep.h>
#include <pearl/RmaGetEnd_rep.h>
#include <pearl/RmaGetStart_rep.h>
#include <pearl/RmaPutEnd_rep.h>
#include <pearl/RmaPutStart_rep.h>

using namespace std;
using namespace pearl;


//--- Static member variables -----------------------------------------------

const EventFactory* EventFactory::m_instance = NULL;


//---------------------------------------------------------------------------
//
//  class EventFactory
//
//---------------------------------------------------------------------------

//--- Constructors & destructor ---------------------------------------------

EventFactory::~EventFactory()
{
}


//--- Generic factory methods -----------------------------------------------

Event_rep* EventFactory::createEnter(const GlobalDefs& defs,
                                     timestamp_t       time,
                                     uint32_t          regionId,
                                     uint64_t*         metrics) const
{
  return new Enter_rep(time,
                       defs.getRegion(regionId),
                       copyMetrics(defs, metrics));
}


Event_rep* EventFactory::createEnterCS(const GlobalDefs& defs,
                                       timestamp_t       time,
                                       uint32_t          callsiteId,
                                       uint64_t*         metrics) const
{
  return new EnterCS_rep(time,
                         defs.getCallsite(callsiteId),
                         copyMetrics(defs, metrics));
}


Event_rep* EventFactory::createLeave(const GlobalDefs& defs,
                                     timestamp_t       time,
                                     uint32_t          regionId,
                                     uint64_t*         metrics) const
{
  return new Leave_rep(time,
                       defs.getRegion(regionId),
                       copyMetrics(defs, metrics));
}


//--- MPI-1 specific factory methods ----------------------------------------

Event_rep* EventFactory::createMpiSend(const GlobalDefs& defs,
                                       timestamp_t       time,
                                       ident_t           comm,
                                       uint32_t          dest,
                                       uint32_t          tag,
                                       uint64_t          bytesSent) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRecv(const GlobalDefs& defs,
                                       timestamp_t       time,
                                       ident_t           comm,
                                       uint32_t          source,
                                       uint32_t          tag,
                                       uint64_t          bytesReceived) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiCollBegin(const GlobalDefs& defs,
                                            timestamp_t       time) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiCollEnd(const GlobalDefs&         defs,
                                          timestamp_t               time,
                                          MpiCollEnd_rep::coll_type type,
                                          ident_t                   comm,
                                          uint32_t                  root,
                                          uint64_t                  bytesSent,
                                          uint64_t                  bytesReceived) const
{
  return NULL;
}


//--- MPI-1 non-blocking specific methods ------------------------------------

Event_rep* EventFactory::createMpiCancelled(const GlobalDefs& defs,
                                            timestamp_t       time,
                                            uint64_t          requestId) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiSendRequest(const GlobalDefs& defs,
                                              timestamp_t       time,
                                              ident_t           comm,
                                              uint32_t          dest,
                                              uint32_t          tag,
                                              uint64_t          bytesSent,
                                              uint64_t          requestId) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiSendComplete(const GlobalDefs& defs,
                                               timestamp_t       time,
                                               uint64_t          requestId) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRecvComplete(const GlobalDefs& defs,
                                               timestamp_t       time,
                                               ident_t           comm,
                                               uint32_t          source,
                                               uint32_t          tag,
                                               uint64_t          bytesReceived,
                                               uint64_t          requestId) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRecvRequest(const GlobalDefs& defs,
                                              timestamp_t       time,
                                              uint64_t          requestId) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRequestTested(const GlobalDefs& defs,
                                                timestamp_t       time,
                                                uint64_t          requestId) const
{
  return NULL;
}


//--- OpenMP specific factory methods ----------------------------------------

Event_rep* EventFactory::createOmpFork(const GlobalDefs& defs,
                                       timestamp_t       time,
                                       uint32_t          teamSize) const
{
  return NULL;
}


Event_rep* EventFactory::createOmpJoin(const GlobalDefs& defs,
                                       timestamp_t       time) const
{
  return NULL;
}


Event_rep* EventFactory::createOmpAcquireLock(const GlobalDefs& defs,
                                              timestamp_t       time,
                                              uint32_t          lockId,
                                              uint32_t          order) const
{
  return NULL;
}


Event_rep* EventFactory::createOmpReleaseLock(const GlobalDefs& defs,
                                              timestamp_t       time,
                                              uint32_t          lockId,
                                              uint32_t          order) const
{
  return NULL;
}


Event_rep* EventFactory::createOmpTaskCreate(const GlobalDefs& defs,
                                             timestamp_t       time,
                                             uint64_t          taskId) const
{
  return NULL;
}


Event_rep* EventFactory::createOmpTaskComplete(const GlobalDefs& defs,
                                               timestamp_t       time,
                                               uint64_t          taskId) const
{
  return NULL;
}


Event_rep* EventFactory::createOmpTaskSwitch(const GlobalDefs& defs,
                                             timestamp_t       time,
                                             uint64_t          taskId) const
{
  return NULL;
}


//--- RMA factory methods ------------------------------------------

Event_rep* EventFactory::createRmaPutStart(const GlobalDefs& defs,
                                           timestamp_t       time,
                                           uint32_t          rma_id,
                                           uint32_t          target,
                                           uint64_t          bytesSent) const
{
  return new RmaPutStart_rep(time, rma_id, target, bytesSent);
}


Event_rep* EventFactory::createRmaPutEnd  (const GlobalDefs& defs,
                                           timestamp_t       time,
                                           uint32_t          rma_id) const
{
  return new RmaPutEnd_rep(time, rma_id);
}


Event_rep* EventFactory::createRmaGetStart(const GlobalDefs& defs,
                                           timestamp_t       time,
                                           uint32_t          rma_id,
                                           uint32_t          origin,
                                           uint64_t          bytesReceived) const
{
  return new RmaGetStart_rep(time, rma_id, origin, bytesReceived);
}


Event_rep* EventFactory::createRmaGetEnd  (const GlobalDefs& defs,
                                           timestamp_t       time,
                                           uint32_t          rma_id) const
{
  return new RmaGetEnd_rep(time, rma_id);
}


//--- MPI-2 RMA specific factory methods ----------------------------------------

Event_rep* EventFactory::createMpiRmaPutStart(const GlobalDefs& defs,
                                              timestamp_t       time,
                                              uint32_t          rma_id,
                                              uint32_t          target,
                                              uint64_t          bytesSent,
                                              ident_t           window) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaPutEnd(const GlobalDefs& defs,
                                            timestamp_t       time,
                                            uint32_t          rma_id) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaGetStart(const GlobalDefs& defs,
                                              timestamp_t       time,
                                              uint32_t          rma_id,
                                              uint32_t          origin,
                                              uint64_t          bytesReceived,
                                              ident_t           window) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaGetEnd(const GlobalDefs& defs,
                                            timestamp_t       time,
                                            uint32_t          rma_id) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaLock(const GlobalDefs& defs,
                                          timestamp_t       time,
                                          uint32_t          location,
                                          ident_t           window,
                                          bool              sync) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaUnlock(const GlobalDefs& defs,
                                            timestamp_t       time,
                                            uint32_t          location,
                                            ident_t           window) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaGats(const GlobalDefs& defs,
                                          timestamp_t       time,
                                          ident_t           window,
                                          ident_t           group,
                                          bool              sync) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaCollBegin(const GlobalDefs& defs,
                                               timestamp_t       time) const
{
  return NULL;
}


Event_rep* EventFactory::createMpiRmaCollEnd(const GlobalDefs& defs,
                                             timestamp_t       time,
                                             ident_t           window) const
{
  return NULL;
}


//--- Buffer-based factory methods ------------------------------------------

Event_rep* EventFactory::createEvent(event_t           type,
                                     const GlobalDefs& defs,
                                     Buffer&           buffer) const
{
  Event_rep* result = NULL;

  switch (type) {
    case ENTER:
      result = new Enter_rep(defs, buffer);
      break;

    case ENTER_CS:
      result = new EnterCS_rep(defs, buffer);
      break;

    case LEAVE:
      result = new Leave_rep(defs, buffer);
      break;

    case RMA_PUT_START:
      result = new RmaPutStart_rep(defs, buffer);
      break;

    case RMA_PUT_END:
      result = new RmaPutEnd_rep(defs, buffer);
      break;

    case RMA_GET_START:
      result = new RmaGetStart_rep(defs, buffer);
      break;

    case RMA_GET_END:
      result = new RmaGetEnd_rep(defs, buffer);
      break;

    default:
      throw RuntimeError("EventFactory::createEvent() -- Unknown event type.");
  }

  return result;
}


//--- Singleton interface ---------------------------------------------------

void EventFactory::registerFactory(const EventFactory* factory)
{
  assert(factory);

  if (m_instance)
    throw FatalError("EventFactory::registerFactory(const EventFactory*) -- "
                     "Factory already registered.");

  m_instance = factory;
}


//--- Data copy methods -----------------------------------------------------

uint64_t* EventFactory::copyMetrics(const GlobalDefs& defs,
                                    uint64_t*         metrics) const
{
  uint64_t* result = NULL;

#ifdef PEARL_ENABLE_METRICS
  // Copy hardware counter metric values
  uint32_t count = defs.num_metrics();
  if (count > 0) {
    result = new uint64_t[count];
    memcpy(result, metrics, count * sizeof(uint64_t));
  }
#endif

  return result;
}
