/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef PEARL_EPIKARCHIVE_H
#define PEARL_EPIKARCHIVE_H


#include <map>
#include <vector>

#include <pearl/Location.h>
#include <pearl/TraceArchive.h>


/*-------------------------------------------------------------------------*/
/**
 *  @file    EpikArchive.h
 *  @ingroup PEARL_base
 *  @brief   Declaration of the class EpikArchive.
 *
 *  This header file provides the declaration of the class EpikArchive and
 *  related functions.
 **/
/*-------------------------------------------------------------------------*/

namespace pearl
{

/*-------------------------------------------------------------------------*/
/**
 *  @class   EpikArchive
 *  @ingroup PEARL_base
 *  @brief   Representation of an EPIK trace experiment archive.
 *
 *  @todo Add documentation.
 **/
/*-------------------------------------------------------------------------*/

class EpikArchive
    : public TraceArchive
{
    public:
        //--- Public methods -------------------------------

        /// @name Constructors & destructor
        /// @{

        /// @brief Constructor.
        ///
        /// Creates a new EPIK trace experiment representation and initializes
        /// it with the given archive name.
        ///
        /// @param  anchorName  Name of the trace experiment archive anchor
        ///                     file/directory
        /// @param  archiveDir  Name of the trace experiment archive directory
        ///
        EpikArchive(const std::string& anchorName,
                    const std::string& archiveDir);

        /// @}
        /// @name Mapping location IDs
        /// @{

        /// @brief Add identifier mapping.
        ///
        /// Adds a new EPIK to internal location identifier mapping.
        ///
        /// @param  epkLocationId  EPIK location identifier
        /// @param  intLocationId  Internal location identifier
        ///
        void
        addIdMapping(uint32_t         epkLocationId,
                     Location::IdType intLocationId);

        /// @brief Map location identifier
        ///
        /// Maps an EPIK location identifier to the corresponding internal
        /// location identifier.
        ///
        /// @param  epkLocationId  EPIK location identifier
        ///
        /// @return Internal location identifier
        ///
        Location::IdType
        mapLocationId(uint32_t epkLocationId) const;

        /// @}
        /// @name Mapping file offset handling
        /// @{

        /// @brief Set mapping file offset.
        ///
        /// Stores the file offset for the mapping data of the given @a rank
        /// in the 'epik.map' file.
        ///
        /// @param  rank    MPI rank
        /// @param  offset  File offset
        ///
        void
        setOffset(uint32_t rank,
                  uint32_t offset);

        /// @brief Get mapping file offset.
        ///
        /// Returns the file offset for the mapping data of the given @a rank
        /// in the 'epik.map' file.
        ///
        /// @param  rank  MPI rank
        ///
        /// @return File offset
        ///
        uint32_t
        getOffset(uint32_t rank) const;

        /// @}


    private:
        //--- Type definitions -----------------------------

        /// Container type for location ID mapping
        typedef std::map<uint32_t, Location::IdType> LocationMap;

        /// Container type for mapping file offsets
        typedef std::vector<uint32_t> OffsetContainer;


        //--- Data members ---------------------------------

        /// %Location ID mapping: EPIK location ID |-@> internal ID
        LocationMap mLocationMap;

        /// Mapping file offset values: rank |-@> file offset
        OffsetContainer mOffsets;


        //--- Private methods & friends --------------------

        virtual void
        open();
        virtual void
        readDefinitions(GlobalDefs* defs);
        virtual void
        readTrace(const GlobalDefs& defs,
                  const Location&   location,
                  LocalTrace*       trace);
};


}   // namespace pearl


#endif   // !PEARL_EPIKARCHIVE_H
