/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include "EpikArchive.h"

#include <cstdlib>
#include <memory>

#include <pearl/Error.h>
#include <pearl/LocalTrace.h>

#include "elg_readcb.h"
#include "epk_archive.h"

#include "MappingTable.h"
#include "readcb.h"

using namespace std;
using namespace pearl;


/*-------------------------------------------------------------------------*/
/**
 *  @file    EpikArchive.cpp
 *  @ingroup PEARL_base
 *  @brief   Implementation of the class EpikArchive.
 *
 *  This file provides the implementation of the class EpikArchive and
 *  related functions.
 **/
/*-------------------------------------------------------------------------*/

//--- Constructors & destructor ---------------------------------------------

EpikArchive::EpikArchive(const string& anchorName,
                         const string& archiveDir)
    : TraceArchive(anchorName, archiveDir)
{
    // Sanity checks
    const char* name = anchorName.c_str();
    if (!epk_archive_set_name(name))
        throw RuntimeError("Experiment archive \"" + anchorName +
                           "\" is not an EPIK archive.");

    if (!epk_archive_exists(name))
        throw RuntimeError("Experiment archive \"" + anchorName +
                           "\"does not exist.");
}


//--- Mapping location IDs --------------------------------------------------

void
EpikArchive::addIdMapping(uint32_t         epkLocationId,
                          Location::IdType intLocationId)
{
    LocationMap::iterator it = mLocationMap.find(epkLocationId);
    if (mLocationMap.end() != it)
        throw FatalError("EpikArchive::addIdMapping(uint32_t,Location::IdType)"
                     " -- mapping already defined!");

    mLocationMap.insert(it, LocationMap::value_type(epkLocationId,
                                                    intLocationId));
}


Location::IdType
EpikArchive::mapLocationId(uint32_t epkLocationId) const
{
    LocationMap::const_iterator it = mLocationMap.find(epkLocationId);
    if (mLocationMap.end() == it)
        throw FatalError("EpikArchive::mapLocationId(uint32_t) -- unknown ID!");

    return it->second;
}


//--- Mapping file offset handling ------------------------------------------

void
EpikArchive::setOffset(uint32_t rank,
                       uint32_t offset)
{
    if (rank >= mOffsets.size())
        mOffsets.resize(rank + 1);

    mOffsets[rank] = offset;
}


uint32_t
EpikArchive::getOffset(uint32_t rank) const
{
    if (rank >= mOffsets.size())
        throw RuntimeError("EpikArchive::getOffset(uint32_t) -- Rank out of bounds");

    return mOffsets[rank];
}


//--- Private methods -------------------------------------------------------

void
EpikArchive::open()
{
}


void
EpikArchive::readDefinitions(GlobalDefs* defs)
{
    // Determine file name
    char*  tmp_str  = epk_archive_filename(EPK_TYPE_ESD, epk_archive_get_name());
    string filename = tmp_str;
    free(tmp_str);

    // Open definitions file
    ElgRCB* file = elg_read_open(filename.c_str());
    if (!file)
        throw RuntimeError("Cannot open EPILOG trace definition file \"" +
                           filename + "\".");

    // Read data
    DefsCbData data(*this, *defs);
    while (elg_read_next_def(file, &data))
    {
        // Check for errors
        if (!data.m_message.empty())
            break;
    }
    elg_read_close(file);

    // Check for errors
    if (!data.m_message.empty()) {
        throw Error(data.m_message);
    }
}


void
EpikArchive::readTrace(const GlobalDefs& defs,
                       const Location&   location,
                       LocalTrace*       trace)
{
    string   name   = getAnchorName();
    uint32_t rank   = location.getRank();
    uint32_t offset = 0;
    if (!mOffsets.empty())
        offset = getOffset(rank);

    // Create mapping table
    auto_ptr<MappingTable> table(new MappingTable(name, rank, offset));

    // Determine file name
    char* tmp_str = epk_archive_rankname(EPK_TYPE_ELG, name.c_str(), rank);
    string filename = tmp_str;
    free(tmp_str);

    // Open local event trace file
    ElgRCB* file = elg_read_open(filename.c_str());
    if (!file)
        throw RuntimeError("Cannot open EPILOG event trace file \"" +
                           filename + "\".");

    // Read trace data
    TraceCbData data(*this, defs, *trace, *table, elg_read_version(file),
                     location.getId(), location.getThreadId());
    while (elg_read_next_event(file, &data))
    {
        // Check for errors
        if (!data.m_message.empty())
          break;
    }
    elg_read_close(file);

    // Check for errors
    if (!data.m_message.empty())
        throw Error(data.m_message);

    // Consistency check
    if (!data.m_callstack.empty())
        throw  FatalError("Unbalanced ENTER/LEAVE events (Too many ENTERs).");
}
