/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef PEARL_TRACEARCHIVE_H
#define PEARL_TRACEARCHIVE_H


#include <string>


/*-------------------------------------------------------------------------*/
/**
 *  @file    TraceArchive.h
 *  @ingroup PEARL_base
 *  @brief   Declaration of the class TraceArchive.
 *
 *  This header file provides the declaration of the class TraceArchive and
 *  related functions.
 **/
/*-------------------------------------------------------------------------*/


namespace pearl
{
//--- Forward declarations --------------------------------------------------

class GlobalDefs;
class LocalTrace;
class Location;


/*-------------------------------------------------------------------------*/
/**
 *  @class   TraceArchive
 *  @ingroup PEARL_base
 *  @brief   Representation of a trace experiment archive.
 *
 *  @todo Add documentation.
 **/
/*-------------------------------------------------------------------------*/

class TraceArchive
{
    public:
        //--- Public methods -------------------------------

        /// @name Constructors & destructor
        /// @{

        /// @brief Destructor.
        ///
        /// Implicitly closes the archive, releases all allocated resources
        /// and destroys the instance.
        ///
        virtual
        ~TraceArchive();

        /// @}
        /// @name Archive management
        /// @{

        /// @brief Opens an experiment archive.
        ///
        /// Opens the trace experiment archive specified by the given
        /// @a anchorName and returns an archive representation object if
        /// successful. Otherwise, a NULL pointer is returned.
        ///
        /// @param  anchorName  Name of the trace experiment archive anchor
        ///                     file/directory
        ///
        /// @return Pointer to archive representation if successful, NULL
        ///         otherwise
        ///
        static TraceArchive*
        open(const std::string& anchorName);

        /// @}
        /// @name Access archive data
        /// @{

        /// @brief Get archive anchor name.
        ///
        /// Returns the name of the trace experiment archive anchor file or
        /// directory.
        ///
        /// @return Archive anchor name
        ///
        std::string
        getAnchorName() const;

        /// @brief Get archive directory.
        ///
        /// Returns the name of the trace experiment archive directory. If
        /// the archive anchor is a directory (e.g., for EPIK trace archives),
        /// the returned value is identical to the anchor name. Otherwise
        /// (e.g., for OTF2 trace archives), it refers to the directory where
        /// the anchor file can be found.
        ///
        /// @return Archive directory name
        ///
        std::string
        getArchiveDirectory() const;

        /// @brief Get global definitions.
        ///
        /// Returns a pointer to a newly created global definitions object
        /// (with ownership transferred to the caller) providing access to the
        /// definition data of the trace experiment archive.
        ///
        ///
        /// @attention This method is not thread-safe. It should therefore only
        ///     be called in a single-threaded context or has to be protected
        ///     properly.
        ///
        /// @return Global definitions object
        ///
        GlobalDefs*
        getDefinitions();

        /// @brief Get trace data.
        ///
        /// Returns a pointer to a newly created event trace data object (with
        /// ownership transferred to the caller) providing access to the events
        /// for the given @a location in this experiment archive. References to
        /// definitions will be resolved using the global definitions object
        /// @a defs, created by a previous call to getDefinitions().
        ///
        /// @note In contrast to the getDefinitions() call, this method is
        ///     thread-safe.
        ///
        /// @param  defs      Global definitions object
        /// @param  location  %Location object
        ///
        /// @return Trace data object
        ///
        LocalTrace*
        getTrace(const GlobalDefs& defs,
                 const Location&   location);

        /// @}


    protected:
        //--- Protected methods ----------------------------

        /// @name Constructors & destructor
        /// @{

        /// @brief Constructor.
        ///
        /// Creates a new trace experiment representation and initializes it
        /// with the given @a anchorName and @a archiveDirectory.
        ///
        /// @param  anchorName  Name of the trace experiment archive anchor
        ///                     file/directory
        /// @param  archiveDir  Name of the trace experiment archive directory
        ///
        TraceArchive(const std::string& anchorName,
                     const std::string& archiveDir);

        /// @}


    private:
        //--- Data members ---------------------------------

        /// Name of anchor file/directory
        std::string mAnchorName;

        /// Name of archive directory
        std::string mArchiveDirectory;


        //--- Private methods & friends --------------------

        TraceArchive(const TraceArchive& rhs);
        TraceArchive&
        operator=(const TraceArchive& rhs);
        virtual void
        open() = 0;
        virtual void
        readDefinitions(GlobalDefs* defs) = 0;
        virtual void
        readTrace(const GlobalDefs& defs,
                  const Location&   location,
                  LocalTrace*       trace) = 0;
};


}   // namespace pearl


#endif   // !PEARL_TRACEARCHIVE_H
