/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef PEARL_SMALLOBJECT_H
#define PEARL_SMALLOBJECT_H


#include <cstddef>
#include <new>


/*-------------------------------------------------------------------------*/
/**
 *  @file    SmallObject.h
 *  @ingroup PEARL_base
 *  @brief   Declaration of the class SmallObject.
 *
 *  This header file provides the declaration of the class SmallObject.
 **/
/*-------------------------------------------------------------------------*/


namespace pearl
{
/*-------------------------------------------------------------------------*/
/**
 *  @class   SmallObject
 *  @ingroup PEARL_base
 *  @brief   Provides a base class for small objects using a custom memory
 *           management.
 *
 *  Each dynamically allocated memory block usually requires some extra
 *  memory for bookkeeping purposes. However, this can be quite space
 *  inefficient if a large number of small objects is allocated, as the
 *  per-object overhead is significant.
 *
 *  The SmallObject class serves as a base class that can be used for these
 *  kind of small objects, providing a customized memory management that
 *  allocates memory in larger chunks rather than individually for each
 *  object. This specialized memory management is restricted to objects of
 *  not more than 64 bytes, however, if objects grow larger (e.g., objects
 *  of a derived class) the default memory allocation routines are used
 *  transparently.
 **/
/*-------------------------------------------------------------------------*/

class SmallObject
{
    public:
        /// @name Memory allocation operators
        /// @{

        /// @brief Allocate storage space.
        ///
        /// Allocates @a objectSize bytes of storage space and returns a
        /// non-null pointer to the first byte of the block. An optimized
        /// allocator is used for small objects less than 64 bytes, the default
        /// memory allocator otherwise.
        ///
        /// @param  objectSize  Size of the requested memory block in bytes
        ///
        /// @exception bad_alloc if the memory allocation cannot be fulfilled
        ///
        static void*
        operator new(std::size_t objectSize)
                                       throw(std::bad_alloc);

        /// @brief Deallocate storage space.
        ///
        /// Deallocates the memory block pointed to by @a deadObject if it is
        /// non-null, releasing memory previously allocated by a call to
        /// operator new. If @a deadObject is NULL, this operator is a no-op.
        ///
        /// @param  deadObject  Pointer to memory block to be released
        /// @param  objectSize  Size of the memory block in bytes
        ///
        static void
        operator delete(void*       deadObject,
                        std::size_t objectSize);

        /// @}


    protected:
        /// @name Constructors & destructor
        /// @{

        /// @brief Destructor.
        ///
        /// Releases all allocated resources and destroys the instance.
        ///
        ~SmallObject();

        /// @}
};
}   // namespace pearl


#endif   // !PEARL_SMALLOBJECT_H
