;;;; implementation of CONSTANTP, needs both INFO and IR1-ATTRIBUTES

;;;; This software is part of the SBCL system. See the README file for
;;;; more information.
;;;;
;;;; This software is derived from the CMU CL system, which was
;;;; written at Carnegie Mellon University and released into the
;;;; public domain. The software is in the public domain and is
;;;; provided with absolutely no warranty. See the COPYING and CREDITS
;;;; files for more information.

(in-package "SB!C")

(!defvar *special-constant-variables* nil)

(defun %constantp (form environment envp)
  (let ((form (if envp
                  (%macroexpand form environment)
                  form)))
    (typecase form
      ;; This INFO test catches KEYWORDs as well as explicitly
      ;; DEFCONSTANT symbols.
      (symbol
       (or (eq (info :variable :kind form) :constant)
           (constant-special-variable-p form)))
      (list
       (let ((answer (constant-special-form-p form environment envp)))
         (if (eq answer :maybe)
             (values (constant-function-call-p form environment envp))
             answer)))
      (t t))))

(defun %constant-form-value (form environment envp)
  (let ((form (if envp
                  (%macroexpand form environment)
                  form)))
    (typecase form
      (symbol
       ;; KLUDGE: superficially, this might look good enough: we grab
       ;; the value from FORM's property list, and if it isn't there (or
       ;; is NIL, but hey) we use the host's value.  This works for
       ;; MOST-POSITIVE-FIXNUM and friends, but still fails for
       ;; float-related constants, where there is in fact no guarantee
       ;; that we can represent our target value at all in the host,
       ;; so we don't try.  We should rework all uses of floating
       ;; point so that we never try to use a host's value, and then
       ;; make some kind of assertion that we never attempt to take
       ;; a host value of a constant in the CL package.
       (or #+sb-xc-host (xc-constant-value form) (symbol-value form)))
      (list
       (multiple-value-bind (specialp value)
           (constant-special-form-value form environment envp)
         (if specialp value (constant-function-call-value
                             form environment envp))))
      (t
       form))))

(defun constant-special-variable-p (name)
  (and (member name *special-constant-variables*) t))

;;; FIXME: It would be nice to deal with inline functions
;;; too.
(defun constant-function-call-p (form environment envp)
  (let ((name (car form)))
    (if (and (legal-fun-name-p name)
             (eq :function (info :function :kind name))
             (let ((info (info :function :info name)))
               (and info (ir1-attributep (fun-info-attributes info)
                                         foldable)))
             (and (every (lambda (arg)
                           (%constantp arg environment envp))
                         (cdr form))))
        ;; Even though the function may be marked as foldable
        ;; the call may still signal an error -- eg: (CAR 1).
        (handler-case
            (values t (constant-function-call-value form environment envp))
          (error ()
            (values nil nil)))
        (values nil nil))))

(defun constant-function-call-value (form environment envp)
  (apply (fdefinition (car form))
         (mapcar (lambda (arg)
                   (%constant-form-value arg environment envp))
                 (cdr form))))

;;;; NOTE!!!
;;;;
;;;; If you add new special forms, check that they do not
;;;; alter the logic of existing ones: eg, currently
;;;; CONSTANT-FORM-VALUE directly evaluates the last expression
;;;; of a PROGN, as no assignment is allowed. If you extend
;;;; analysis to assignments then other forms must take this
;;;; into account.

(eval-when (:compile-toplevel :execute)
(defparameter *special-form-constantp-defs* (make-array 20 :fill-pointer 0)))

(defmacro !defconstantp (operator lambda-list &key test eval)
  (let ((args (make-symbol "ARGS")))
    (flet
        ;; FIXME: DESTRUCTURING-BIND should have the option to expand this way.
        ;; It would be useful for DEFINE-SOURCE-TRANSFORM as well.
        ((binding-maker (input on-error)
           (multiple-value-bind (llks req opt rest key aux env whole)
               (parse-lambda-list
                lambda-list
                :accept (lambda-list-keyword-mask '(&whole &optional &rest &body)))
             (declare (ignore llks key aux env))
             (aver (every (lambda (x) (and (symbolp x) x)) (append req opt rest)))
             (flet ((bind (var pred enforce-end)
                      `(,(car var)
                        ,(if enforce-end
                             `(if (and (,pred ,args) (not (cdr ,args)))
                                  (car ,args)
                                  ,on-error)
                             `(if (,pred ,args) (pop ,args) ,on-error)))))
               `((,args ,input)
                 ,@(when whole
                     ;; If both &WHOLE and &REST are present, the &WHOLE var
                     ;; must be a list, although we don't know that just yet.
                     ;; It will be verified when the &REST arg is bound.
                     `((,(car whole) ,(if rest `(truly-the list ,args) args))))
                 ,@(maplist (lambda (x)
                              (bind x (if (cdr x) 'listp 'consp)
                                    (and (not (cdr x)) (not opt) (not rest))))
                            req)
                 ,@(maplist (lambda (x) (bind x 'listp (and (not (cdr x)) (not rest))))
                            opt)
                 ,@(when rest
                     `((,(car rest)
                        (if (proper-list-p ,args)
                            (truly-the list ,args) ; to open-code EVERY #'P on &REST arg
                            ,on-error)))))))))
      `(eval-when (:compile-toplevel :execute)
         (vector-push-extend ',(list* operator test eval
                                      (binding-maker 'args '(go fail)))
                             *special-form-constantp-defs*)))))

;;; NOTE: special forms are tested in the order as written,
;;; so there is some benefit to listing important ones earliest.

(!defconstantp quote (value)
   :test t
   :eval value)

(!defconstantp if (test then &optional else)
   :test
   (and (constantp* test)
        (constantp* (if (constant-form-value* test)
                        then
                        else)))
   :eval (if (constant-form-value* test)
             (constant-form-value* then)
             (constant-form-value* else)))

;; FIXME: isn't it sufficient for non-final forms to be flushable and/or
;; maybe satisfy some other conditions? e.g. (PROGN (LIST 1) 'FOO) is constant.
(!defconstantp progn (&body forms)
   :test (every #'constantp* forms)
   :eval (constant-form-value* (car (last forms))))

(!defconstantp the (type form)
   ;; We can't call TYPEP because the form might be (THE (FUNCTION (t) t) #<fn>)
   ;; which is valid for declaration but not for discrimination.
   ;; Instead use %%TYPEP in non-strict mode. FIXME:
   ;; (1) CAREFUL-SPECIFIER-TYPE should never fail. See lp#1395910.
   ;; (2) CONTAINS-UNKNOWN-TYPE-P should grovel into ARRAY-TYPE-ELEMENT-TYPE
   ;; so that (C-U-T-P (SPECIFIER-TYPE '(OR (VECTOR BAD) FUNCTION))) => T
   ;; and then we can parse, check for unknowns, and get rid of HANDLER-CASE.
   :test (and (constantp* form)
              (handler-case
                  ;; in case the type-spec is malformed!
                  (let ((parsed (careful-specifier-type type)))
                    ;; xc can't rely on a "non-strict" mode of TYPEP.
                    (and parsed
                         #+sb-xc-host
                         (typep (constant-form-value* form)
                                (let ((*unparse-fun-type-simplify* t))
                                  (declare (special *unparse-fun-type-simplify*))
                                  (type-specifier parsed)))
                         #-sb-xc-host
                         (%%typep (constant-form-value* form) parsed nil)))
                (error () nil)))
   :eval (constant-form-value* form))

(!defconstantp unwind-protect (&whole subforms protected-form &body cleanup-forms)
   :test (every #'constantp* subforms)
   :eval (constant-form-value* protected-form))

(!defconstantp block (name &body forms)
   ;; We currently fail to detect cases like
   ;;
   ;; (BLOCK FOO
   ;;   ...CONSTANT-FORMS...
   ;;   (RETURN-FROM FOO CONSTANT-VALUE)
   ;;   ...ANYTHING...)
   ;;
   ;; Right now RETURN-FROM kills the constantness unequivocally.
   :test (every #'constantp* forms)
   :eval (constant-form-value* (car (last forms))))

(!defconstantp multiple-value-prog1 (&whole subforms first-form &body forms)
   :test (every #'constantp* subforms)
   :eval (constant-form-value* first-form))

(!defconstantp progv (symbols values &body forms)
   :test (and (constantp* symbols)
              (constantp* values)
              (let* ((symbol-values (constant-form-value* symbols))
                     (*special-constant-variables*
                      (append symbol-values *special-constant-variables*)))
                (progv
                    symbol-values
                    (constant-form-value* values)
                  (every #'constantp* forms))))
   :eval (progv
             (constant-form-value* symbols)
             (constant-form-value* values)
           (constant-form-value* (car (last forms)))))

;;;

(macrolet
    ((expand-cases (expr-selector default-clause)
       `(flet ((constantp* (x) (%constantp x environment envp))
               (constant-form-value* (x) (%constant-form-value x environment envp)))
          (declare (optimize speed) (ignorable #'constantp*)
                   (ftype (function (t) (values t &optional)) ; avoid "unknown values"
                          constantp* constant-form-value*))
          (let ((args (cdr (truly-the list form))))
            (case (car form)
              ,@(map 'list
                     (lambda (spec &aux (bindings (cdddr spec)))
                       `(,(first spec)
                         (let* ,bindings
                           (declare (ignorable ,@(mapcar #'car bindings)))
                           ,(nth expr-selector spec))))
                     *special-form-constantp-defs*)
              (t
               ,default-clause))))))

  (defun constant-special-form-p (form environment envp)
    (let (result)
      (tagbody (setq result (expand-cases 1 :maybe)) fail)
      result))

  (defun constant-special-form-value (form environment envp)
    (let ((result))
      (tagbody
       (setq result (expand-cases 2 (return-from constant-special-form-value
                                      (values nil nil))))
       (return-from constant-special-form-value (values t result))
       fail))
    ;; Mutatation of FORM could cause failure. It's user error, not a bug.
    (error "CONSTANT-FORM-VALUE called with invalid expression ~S" form)))
