#!/usr/bin/python
#
# Simple Backup suit
#
# Running this command will restore a file or directory from backup.
# This is also a backend for simple-restore-gnome GUI.
#
# Author: Aigars Mahinovs <aigarius@debian.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import tarfile, sys, re, os, os.path, shutil, datetime, filecmp, gzip, tempfile, zlib
import cPickle as pickle
try:
    import gnomevfs
except ImportError:
    import gnome.vfs as gnomevfs


class MyGzipFile(gzip.GzipFile):
    def _read(self, size=1024):
        if self.fileobj is None:
            raise EOFError, "Reached EOF"

        if self._new_member:
            # If the _new_member flag is set, we have to
            # jump to the next member, if there is one.
            #
            # First, check if we're at the end of the file;
            # if so, it's time to stop; no more members to read.
            pos = self.fileobj.tell()   # Save current position
            self.fileobj.seek(0, 2)     # Seek to end of file
            if pos == self.fileobj.tell():
                raise EOFError, "Reached EOF"
            else:
                self.fileobj.seek( pos ) # Return to original position

            self._init_read()
            self._read_gzip_header()
            self.decompress = zlib.decompressobj(-zlib.MAX_WBITS)
            self._new_member = False

        # Read a chunk of data from the file
        try:
                buf = self.fileobj.read(size)
        except:
                buf = ""

        # If the EOF has been reached, flush the decompression object
        # and mark this object as finished.

        if buf == "":
            uncompress = self.decompress.flush()
            #self._read_eof()
            self._add_read_data( uncompress )
            raise EOFError, 'Reached EOF'

        uncompress = self.decompress.decompress(buf)
        self._add_read_data( uncompress )

        if self.decompress.unused_data != "":
            # Ending case: we've come to the end of a member in the file,
            # so seek back to the start of the unused data, finish up
            # this member, and read a new gzip header.
            # (The number of bytes to seek back is the length of the unused
            # data, minus 8 because _read_eof() will rewind a further 8 bytes)
            self.fileobj.seek( -len(self.decompress.unused_data)+8, 1)

            # Check the CRC and file size, and set the flag so we read
            # a new member on the next call
            self._read_eof()
            self._new_member = True



class MyTarFile(tarfile.TarFile):
    posix = False
    def gzopen(cls, name, mode="r", fileobj=None, compresslevel=9):
        """Open gzip compressed tar archive name for reading or writing.
           Appending is not allowed.
        """
        if len(mode) > 1 or mode not in "rw":
            raise ValueError, "mode must be 'r' or 'w'"

        pre, ext = os.path.splitext(name)
        pre = os.path.basename(pre)
        if ext == ".tgz":
            ext = ".tar"
        if ext == ".gz":
            ext = ""
        tarname = pre + ext

        if fileobj is None:
            fileobj = file(name, mode + "b")

        if mode != "r":
            name = tarname

        try:
            t = cls.taropen(tarname, mode,
                MyGzipFile(name, mode, compresslevel, fileobj)
            )
        except IOError:
            raise ReadError, "not a gzip file"
        t._extfileobj = False
        return t

    gzopen = classmethod(gzopen)


class SRestore:
	def __init__(self):
		pass

	def restore( self, tdir, spath, dpath = None ):
		"""
		Restore one file or directory from the backup tdir with name
		spath to dpath (or to its old location).
		All existing files must be moved to a "*.before_restore_$time" files.
		"""
		
		# Detect backup format
		# TODO in later versions
		
		format= 1

		local = True

		try:
		    if gnomevfs.URI( tdir ).is_local:
		        tdir = gnomevfs.get_local_path_from_uri( tdir )
		    else:
		        local = False
		except:
		    pass

		try:
			if local:
	       			self.tar = MyTarFile.open( tdir+"/files.tgz", "r:gz" )
	       		else:
	       			turi = gnomevfs.URI( tdir+"/files.tgz" )
	       			thandle = gnomevfs.open( turi, 1 )
	     			self.tar = MyTarFile.open( "files.tgz", "r:gz", thandle )
		except: return False

		if not dpath:
			dpath = spath

		# Gather spath and dpath information
		if spath[0] == "/": spath = spath[1:]
		spath = os.path.normpath( spath )
		(sparent, sname) = os.path.split( spath )
		if not sname:
			(sparent, sname) = os.path.split( sparent )
		dpath = os.path.normpath( dpath )
		(dparent, dname) = os.path.split( dpath )
		if not dname:
			dpath = dparent
			(dparent, dname) = os.path.split( dpath )
		
		now = datetime.datetime.now().isoformat("_").replace( ":", "." )
		
		self.tree = self.tar.getmembers()

                self.childlist = []
		
		for file in self.tree:
			if file.name == spath:
				self.childlist.append( file )
			if str(file.name)[0:len(spath)+1] == spath+"/":
				self.childlist.append( file )
		del self.tree

		if len(self.childlist) > 1:
			s_isdir = True
		elif len(self.childlist) == 1:
			s_isdir = False
		else:
			return False

		if s_isdir: # spath is a directory
			if os.path.exists(dpath):
				if os.path.isdir(dpath):
					tdir = tempfile.mkdtemp( dir=dpath )
					self.extract( tdir )
					for file in self.childlist:
						bname = file.name[len(spath)+1:]
						src = os.path.join( tdir, spath, bname )
						dst = os.path.join( dpath, bname )
						if file.isdir():
							if not os.path.exists(dst):
								os.makedirs(dst)
							os.chown( dst, file.uid, file.gid )
							os.chmod( dst, file.mode )
						else:
							if os.path.exists(dst) and not filecmp.cmp( src, dst):
								shutil.move( dst, dst+".before_restore_"+now )
							if not os.path.exists(dst):
								shutil.move( src, dst )
					shutil.rmtree( tdir )
				else:
					tdir = tempfile.mkdtemp( dir=dparent )
					self.extract( tdir )
					shutil.move( dpath, dpath+".before_restore_"+now )
					shutil.move( os.path.join(tdir,spath), dpath )
					shutil.rmtree( tdir )
					
			else:
				tdir = tempfile.mkdtemp( dir=dparent )
				self.extract( tdir )
				shutil.move( os.path.join(tdir,spath), dpath )
				shutil.rmtree( tdir )
				
		else: # Is a regular file
			if os.path.exists(dpath) and os.path.isdir(dpath):
				dpath = os.path.join(dpath, sname)
				if os.path.exists(dpath):
					shutil.move( dpath, dpath+".before_restore_"+now )
					self.tar._extract_member( self.childlist[0], dpath )
				else:
					self.tar._extract_member( self.childlist[0], dpath )
			else:
				if os.path.exists(dpath):
					shutil.move( dpath, dpath+".before_restore_"+now )
					self.tar._extract_member( self.childlist[0], dpath )
				else:
					self.tar._extract_member( self.childlist[0], dpath )
		# Done
		self.tar.close()

		return True

	def extract( self, dst ):
		for m in self.childlist:
			self.tar.extract( m, dst )

	
	

if __name__ == "__main__":
	r = SRestore()
	if not len(sys.argv) in [3,4]:
		print "Simple Backup suit command line restore utility"
		print " Usage: simple-restore backup-url file-or-dir-to-restore [target-file-or-dir]"
		print " Note: backup-url must include the snapshot subdirectory name, for example:"
		print "  /var/backup/2005-08-09_14:59:38.441112.myhost.ful/"
		print " Use simple-restore-gnome for more ease of use."
		sys.exit(1)
	
	if len(sys.argv) == 3:
		ret = r.restore( sys.argv[1], sys.argv[2] )
	else:
		ret = r.restore( sys.argv[1], sys.argv[2], sys.argv[3] )

	if not ret:
		print "Restore FAILED! Please check you parameters."
