#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) 2017-2019 Satpy developers
#
# This file is part of satpy.
#
# satpy is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# satpy is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# satpy.  If not, see <http://www.gnu.org/licenses/>.
"""Module for testing the satpy.readers.viirs_compact module."""

import numpy as np
import unittest
import h5py
import tempfile
import os


class TestCompact(unittest.TestCase):
    """Test class for reading compact viirs format."""

    def setUp(self):
        """Create a fake file from scratch."""
        fake_dnb = {
            "All_Data": {
                "ModeGran": {"value": 0},
                "ModeScan": {
                    "value": np.array(
                        [
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            0,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            254,
                            249,
                        ],
                        dtype=np.uint8,
                    )
                },
                "NumberOfScans": {"value": np.array([47])},
                "VIIRS-DNB-GEO_All": {
                    "AlignmentCoefficient": {
                        "value": np.array(
                            [
                                2.11257413e-02,
                                2.11152732e-02,
                                2.11079046e-02,
                                2.10680142e-02,
                                1.80840008e-02,
                                1.80402063e-02,
                                1.79968309e-02,
                                1.79477539e-02,
                                2.20463774e-03,
                                2.17431062e-03,
                                2.14360282e-03,
                                2.11503846e-03,
                                2.08630669e-03,
                                2.05924874e-03,
                                2.03177333e-03,
                                2.00573727e-03,
                                1.98072987e-03,
                                1.95503305e-03,
                                1.93077011e-03,
                                1.90702057e-03,
                                1.88353716e-03,
                                1.86104013e-03,
                                1.83863181e-03,
                                1.81696517e-03,
                                1.79550308e-03,
                                1.77481642e-03,
                                1.75439729e-03,
                                1.73398503e-03,
                                1.71459839e-03,
                                1.69516564e-03,
                                1.67622324e-03,
                                1.65758410e-03,
                                1.63990213e-03,
                                1.62128301e-03,
                                1.60375470e-03,
                                1.58667017e-03,
                                1.61543000e-03,
                                1.59775047e-03,
                                1.50719041e-03,
                                1.48937735e-03,
                                1.47257745e-03,
                                1.50070526e-03,
                                1.48288533e-03,
                                9.29064234e-04,
                                9.12246935e-04,
                                8.95748264e-04,
                                8.71886965e-04,
                                8.55044520e-04,
                                8.38686305e-04,
                                8.18263041e-04,
                                8.01501446e-04,
                                7.85346841e-04,
                                1.15984806e-03,
                                1.14326552e-03,
                                1.12648588e-03,
                                1.11018715e-03,
                                1.09399087e-03,
                                1.19698711e-03,
                                1.18051842e-03,
                                1.16404379e-03,
                                1.14832399e-03,
                                9.92591376e-04,
                                9.75896895e-04,
                                9.59663419e-04,
                                9.43415158e-04,
                                9.27662419e-04,
                                8.92253709e-04,
                                8.75947590e-04,
                                8.60177504e-04,
                                8.44484195e-04,
                                8.35279003e-04,
                                8.19236680e-04,
                                8.03303672e-04,
                                7.87482015e-04,
                                7.60449213e-04,
                                7.44239136e-04,
                                7.28625571e-04,
                                7.12990935e-04,
                                6.89090986e-04,
                                6.73000410e-04,
                                6.57248020e-04,
                                6.41623745e-04,
                                6.20219158e-04,
                                6.04308851e-04,
                                5.88596100e-04,
                                5.73108089e-04,
                                3.65344196e-04,
                                3.49639275e-04,
                                3.34273063e-04,
                                4.81286290e-04,
                                4.65485587e-04,
                                4.49862011e-04,
                                4.34543617e-04,
                                4.19324206e-04,
                                2.60536268e-04,
                                2.45052564e-04,
                                2.29740850e-04,
                                2.34466774e-04,
                                2.18822126e-04,
                                2.03370175e-04,
                                1.88058810e-04,
                                1.60192372e-04,
                                1.44485937e-04,
                                1.28920830e-04,
                                3.45615146e-04,
                                3.30171984e-04,
                                3.14682693e-04,
                                2.99300562e-04,
                                2.83925037e-04,
                                2.68518896e-04,
                                2.53254839e-04,
                                2.37950648e-04,
                                2.22716670e-04,
                                2.07562072e-04,
                                1.92296386e-04,
                                1.77147449e-04,
                                1.61994336e-04,
                                1.46895778e-04,
                                1.31844325e-04,
                                1.16730320e-04,
                                1.01757469e-04,
                                8.67861963e-05,
                                7.18669180e-05,
                                5.70719567e-05,
                                4.24701866e-05,
                                2.84846719e-05,
                                1.70599415e-05,
                                -1.47213286e-05,
                                -2.33691408e-05,
                                -3.68025649e-05,
                                -5.12388433e-05,
                                -6.59972284e-05,
                                -8.08926561e-05,
                                -9.58433884e-05,
                                -1.10882705e-04,
                                -1.25976600e-04,
                                -1.41044657e-04,
                                -1.56166439e-04,
                                -1.71307023e-04,
                                -1.86516074e-04,
                                -2.01731804e-04,
                                -2.16980450e-04,
                                -2.32271064e-04,
                                -2.47527263e-04,
                                -2.62940506e-04,
                                -2.78283434e-04,
                                -2.93711084e-04,
                                -3.09180934e-04,
                                -3.24661058e-04,
                                -3.40237195e-04,
                                -1.27807143e-04,
                                -1.43646437e-04,
                                -1.59638614e-04,
                                -1.87593061e-04,
                                -2.03169184e-04,
                                -2.18941437e-04,
                                -2.34920750e-04,
                                -2.30605408e-04,
                                -2.46262236e-04,
                                -2.62226094e-04,
                                -4.19838558e-04,
                                -4.35510388e-04,
                                -4.51152271e-04,
                                -4.67120990e-04,
                                -4.83241311e-04,
                                -3.37647041e-04,
                                -3.53568990e-04,
                                -3.69836489e-04,
                                -5.76354389e-04,
                                -5.92070050e-04,
                                -6.08178903e-04,
                                -6.24440494e-04,
                                -6.45648804e-04,
                                -6.61431870e-04,
                                -6.77491073e-04,
                                -6.93967624e-04,
                                -7.17683870e-04,
                                -7.33471534e-04,
                                -7.49999890e-04,
                                -7.66390527e-04,
                                -7.93468382e-04,
                                -8.09502264e-04,
                                -8.25728697e-04,
                                -8.42282083e-04,
                                -8.51265620e-04,
                                -8.67322611e-04,
                                -8.83649045e-04,
                                -9.00280487e-04,
                                -9.35055199e-04,
                                -9.51097580e-04,
                                -9.67527216e-04,
                                -9.84144746e-04,
                                -1.00128003e-03,
                                -1.15522649e-03,
                                -1.17168750e-03,
                                -1.18826574e-03,
                                -1.20496599e-03,
                                -1.10272120e-03,
                                -1.11865194e-03,
                                -1.13539130e-03,
                                -1.15241797e-03,
                                -1.16964686e-03,
                                -7.97322951e-04,
                                -8.14269355e-04,
                                -8.31696263e-04,
                                -8.51555436e-04,
                                -8.68656265e-04,
                                -8.86220601e-04,
                                -9.09406052e-04,
                                -9.26509325e-04,
                                -9.44124535e-04,
                                -1.49479776e-03,
                                -1.51314179e-03,
                                -1.48387800e-03,
                                -1.50146009e-03,
                                -1.51945755e-03,
                                -1.61006744e-03,
                                -1.62846781e-03,
                                -1.59783731e-03,
                                -1.61545863e-03,
                                -1.63336343e-03,
                                -1.65167439e-03,
                                -1.67034590e-03,
                                -1.68956630e-03,
                                -1.70884258e-03,
                                -1.72863202e-03,
                                -1.74859120e-03,
                                -1.76901231e-03,
                                -1.79015659e-03,
                                -1.81144674e-03,
                                -1.83329231e-03,
                                -1.85552111e-03,
                                -1.87840930e-03,
                                -1.90151483e-03,
                                -1.92550803e-03,
                                -1.94982730e-03,
                                -1.97511422e-03,
                                -2.00066133e-03,
                                -2.02709576e-03,
                                -2.05422146e-03,
                                -2.08215159e-03,
                                -2.11093877e-03,
                                -2.14011059e-03,
                                -2.17073411e-03,
                                -2.20196834e-03,
                                -2.23409734e-03,
                                -2.26700748e-03,
                                -2.30150856e-03,
                                -2.33719964e-03,
                                -2.37406371e-03,
                                -2.41223071e-03,
                                -2.45184498e-03,
                                -2.49327719e-03,
                                -2.53651105e-03,
                                -2.58166087e-03,
                                -2.62895599e-03,
                                -2.67871981e-03,
                                -2.73117283e-03,
                                -5.49861044e-03,
                                -5.55437338e-03,
                                -5.61159104e-03,
                                -5.67073002e-03,
                                -5.73173212e-03,
                                -5.79498662e-03,
                                -5.85969677e-03,
                                -5.92768658e-03,
                                -5.99809457e-03,
                                -6.07080618e-03,
                                -6.14715228e-03,
                                -6.22711331e-03,
                            ],
                            dtype=np.float32,
                        )
                    },
                    "ExpansionCoefficient": {
                        "value": np.array(
                            [
                                1.17600127e-03,
                                1.17271533e-03,
                                1.17000856e-03,
                                1.16674276e-03,
                                2.11251900e-03,
                                2.10516527e-03,
                                2.09726905e-03,
                                2.08941335e-03,
                                1.63907595e-02,
                                1.58577170e-02,
                                1.53679820e-02,
                                1.49007449e-02,
                                1.44708352e-02,
                                1.40612368e-02,
                                1.36818690e-02,
                                1.33193973e-02,
                                1.29744308e-02,
                                1.26568424e-02,
                                1.23488475e-02,
                                1.20567940e-02,
                                1.17803067e-02,
                                1.15150018e-02,
                                1.12629030e-02,
                                1.10203745e-02,
                                1.07905651e-02,
                                1.05690639e-02,
                                1.03563424e-02,
                                1.01526314e-02,
                                9.95650515e-03,
                                9.76785459e-03,
                                9.58597753e-03,
                                9.41115711e-03,
                                9.23914276e-03,
                                9.07964632e-03,
                                8.92116502e-03,
                                8.76654685e-03,
                                9.04925726e-03,
                                8.88936501e-03,
                                9.14804544e-03,
                                8.98920093e-03,
                                8.83030891e-03,
                                9.06952657e-03,
                                8.90891161e-03,
                                1.36343827e-02,
                                1.32706892e-02,
                                1.29242949e-02,
                                1.36271119e-02,
                                1.32572902e-02,
                                1.29025253e-02,
                                1.35165229e-02,
                                1.31412474e-02,
                                1.27808526e-02,
                                8.91761761e-03,
                                8.74674786e-03,
                                8.58181808e-03,
                                8.42147414e-03,
                                8.26664641e-03,
                                7.81304855e-03,
                                7.67400907e-03,
                                7.54208490e-03,
                                7.40892906e-03,
                                8.81091598e-03,
                                8.62924196e-03,
                                8.45206063e-03,
                                8.28018785e-03,
                                8.11239891e-03,
                                8.62185098e-03,
                                8.43446422e-03,
                                8.25031102e-03,
                                8.07087123e-03,
                                8.30837712e-03,
                                8.11944436e-03,
                                7.93648325e-03,
                                7.75875151e-03,
                                8.14332347e-03,
                                7.94676598e-03,
                                7.75293307e-03,
                                7.56529858e-03,
                                7.88933039e-03,
                                7.68536143e-03,
                                7.48489471e-03,
                                7.28917075e-03,
                                7.55438488e-03,
                                7.34063145e-03,
                                7.13229552e-03,
                                6.92783622e-03,
                                1.06161544e-02,
                                1.01234140e-02,
                                9.64432582e-03,
                                6.52031973e-03,
                                6.29310543e-03,
                                6.06948463e-03,
                                5.84984245e-03,
                                5.63343242e-03,
                                8.61937553e-03,
                                8.08268972e-03,
                                7.55874207e-03,
                                6.79610623e-03,
                                6.32849289e-03,
                                5.86955249e-03,
                                5.41723240e-03,
                                5.56734810e-03,
                                5.01116784e-03,
                                4.46233014e-03,
                                1.40874484e-03,
                                1.34475902e-03,
                                1.28140685e-03,
                                1.21824886e-03,
                                1.15505024e-03,
                                1.09222531e-03,
                                1.02962845e-03,
                                9.67168540e-04,
                                9.04808170e-04,
                                8.42478999e-04,
                                7.80681905e-04,
                                7.18652213e-04,
                                6.56902499e-04,
                                5.95146266e-04,
                                5.33432467e-04,
                                4.72071581e-04,
                                4.10460081e-04,
                                3.49062117e-04,
                                2.87777104e-04,
                                2.26464268e-04,
                                1.65259655e-04,
                                1.03993290e-04,
                                4.27830964e-05,
                                -1.84028686e-05,
                                -7.95840388e-05,
                                -1.40780976e-04,
                                -2.01987947e-04,
                                -2.63233029e-04,
                                -3.24499299e-04,
                                -3.85862397e-04,
                                -4.47216793e-04,
                                -5.08567959e-04,
                                -5.70152479e-04,
                                -6.31901203e-04,
                                -6.93684444e-04,
                                -7.55490037e-04,
                                -8.17523745e-04,
                                -8.79664498e-04,
                                -9.41973762e-04,
                                -1.00450485e-03,
                                -1.06710335e-03,
                                -1.12990546e-03,
                                -1.19290419e-03,
                                -1.25615683e-03,
                                -1.31971564e-03,
                                -1.38323894e-03,
                                -4.38789371e-03,
                                -4.93527949e-03,
                                -5.48970094e-03,
                                -5.34658274e-03,
                                -5.79780247e-03,
                                -6.25621388e-03,
                                -6.72366377e-03,
                                -7.48283789e-03,
                                -8.00681766e-03,
                                -8.54192488e-03,
                                -5.58420410e-03,
                                -5.79793099e-03,
                                -6.01683883e-03,
                                -6.23886706e-03,
                                -6.46463828e-03,
                                -9.56355780e-03,
                                -1.00387875e-02,
                                -1.05282217e-02,
                                -6.87109074e-03,
                                -7.07587786e-03,
                                -7.28309387e-03,
                                -7.49528036e-03,
                                -7.23363785e-03,
                                -7.42882164e-03,
                                -7.62982434e-03,
                                -7.83343613e-03,
                                -7.51076965e-03,
                                -7.69859226e-03,
                                -7.88733363e-03,
                                -8.08352232e-03,
                                -7.69890239e-03,
                                -7.87641760e-03,
                                -8.05852562e-03,
                                -8.24564695e-03,
                                -8.00882280e-03,
                                -8.18727538e-03,
                                -8.36882368e-03,
                                -8.55544209e-03,
                                -8.04922916e-03,
                                -8.21674801e-03,
                                -8.38823151e-03,
                                -8.56383517e-03,
                                -8.74411128e-03,
                                -7.35407788e-03,
                                -7.48245185e-03,
                                -7.61653157e-03,
                                -7.75389513e-03,
                                -8.20003450e-03,
                                -8.35770369e-03,
                                -8.51695240e-03,
                                -8.67962278e-03,
                                -8.84699915e-03,
                                -1.26767000e-02,
                                -1.30308550e-02,
                                -1.34020159e-02,
                                -1.27902590e-02,
                                -1.31374933e-02,
                                -1.35022206e-02,
                                -1.28020663e-02,
                                -1.31427627e-02,
                                -1.35003338e-02,
                                -8.81921593e-03,
                                -8.97676684e-03,
                                -8.73885304e-03,
                                -8.89289286e-03,
                                -9.05076787e-03,
                                -8.79113190e-03,
                                -8.94579384e-03,
                                -8.66949651e-03,
                                -8.81993212e-03,
                                -8.97467043e-03,
                                -9.13402718e-03,
                                -9.29924846e-03,
                                -9.47104022e-03,
                                -9.64829233e-03,
                                -9.83224157e-03,
                                -1.00242840e-02,
                                -1.02243433e-02,
                                -1.04304748e-02,
                                -1.06464764e-02,
                                -1.08723603e-02,
                                -1.11076497e-02,
                                -1.13517633e-02,
                                -1.16107482e-02,
                                -1.18797245e-02,
                                -1.21643478e-02,
                                -1.24597261e-02,
                                -1.27725713e-02,
                                -1.31026637e-02,
                                -1.34509858e-02,
                                -1.38195883e-02,
                                -1.42097492e-02,
                                -1.46267340e-02,
                                -1.50670996e-02,
                                -1.55417984e-02,
                                -1.60482023e-02,
                                -1.65943075e-02,
                                -1.71795618e-02,
                                -1.78127103e-02,
                                -1.84999816e-02,
                                -1.92504879e-02,
                                -2.00698171e-02,
                                -2.09702197e-02,
                                -2.19654124e-02,
                                -2.30720937e-02,
                                -2.43106075e-02,
                                -2.57069822e-02,
                                -2.72962451e-02,
                                -1.43178934e-02,
                                -1.48085468e-02,
                                -1.53383436e-02,
                                -1.59113277e-02,
                                -1.65353119e-02,
                                -1.72161739e-02,
                                -1.79625414e-02,
                                -1.87847745e-02,
                                -1.96950957e-02,
                                -2.07099430e-02,
                                -2.18482167e-02,
                                -2.31328830e-02,
                            ],
                            dtype=np.float32,
                        )
                    },
                    "Latitude": {"value": np.random.rand(96, 332).astype(np.float32)},
                    "Longitude": {"value": np.random.rand(96, 332).astype(np.float32)},
                    "LunarAzimuthAngle": {
                        "value": np.random.rand(96, 332).astype(np.float32)
                    },
                    "LunarZenithAngle": {
                        "value": np.random.rand(96, 332).astype(np.float32)
                    },
                    "MidTime": {
                        "value": np.array(
                            [
                                1950675122400462,
                                1950675124187044,
                                1950675125973621,
                                1950675127760200,
                                1950675129546777,
                                1950675131333401,
                                1950675133119981,
                                1950675134906559,
                                1950675136693138,
                                1950675138479716,
                                1950675140266341,
                                1950675142052918,
                                1950675143839498,
                                1950675145626075,
                                1950675147412654,
                                1950675149199278,
                                1950675150985857,
                                1950675152772434,
                                1950675154559014,
                                1950675156345591,
                                1950675158132216,
                                1950675159918795,
                                1950675161705373,
                                1950675163491595,
                                1950675165278173,
                                1950675167064395,
                                1950675168850973,
                                1950675170637195,
                                1950675172423773,
                                1950675174209995,
                                1950675175996573,
                                1950675177782795,
                                1950675179569373,
                                1950675181355595,
                                1950675183142173,
                                1950675184928395,
                                1950675186714973,
                                1950675188501195,
                                1950675190287773,
                                1950675192073995,
                                1950675193860573,
                                1950675195646795,
                                1950675197433373,
                                1950675199219595,
                                1950675201006173,
                                1950675202792395,
                                1950675204578973,
                                -993,
                            ]
                        )
                    },
                    "MoonIllumFraction": {"value": 11.518141746520996},
                    "MoonPhaseAngle": {"value": 140.32131958007812},
                    "NumberOfTiePointZoneGroupsScan": {"value": 62},
                    "NumberOfTiePointZoneGroupsTrack": {"value": 1},
                    "NumberOfTiePointZonesScan": {
                        "value": np.array(
                            [
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                28,
                                2,
                                3,
                                2,
                                3,
                                3,
                                3,
                                5,
                                4,
                                5,
                                4,
                                4,
                                4,
                                4,
                                4,
                                3,
                                5,
                                3,
                                4,
                                3,
                                23,
                                23,
                                3,
                                4,
                                3,
                                5,
                                3,
                                4,
                                4,
                                4,
                                4,
                                4,
                                5,
                                4,
                                5,
                                3,
                                3,
                                3,
                                2,
                                3,
                                2,
                                40,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                                1,
                            ],
                            dtype=np.int32,
                        )
                    },
                    "NumberOfTiePointZonesTrack": {"value": 1},
                    "PadByte1": {"value": np.array([0, 0, 0], dtype=np.uint8)},
                    "QF1_SCAN_VIIRSSDRGEO": {
                        "value": np.array(
                            [
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                0,
                                128,
                                2,
                                130,
                                2,
                                130,
                                2,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                142,
                                14,
                                0,
                            ],
                            dtype=np.uint8,
                        )
                    },
                    "QF2_SCAN_VIIRSSDRGEO": {
                        "value": np.array(
                            [
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                2,
                            ],
                            dtype=np.uint8,
                        )
                    },
                    "SCAttitude": {
                        "value": np.array(
                            [
                                [-9.22587514e-01, 3.92340779e00, 5.93621433e-01],
                                [-2.82428920e-01, 3.98425841e00, 7.05978215e-01],
                                [5.63421488e-01, 3.83695555e00, 3.93174857e-01],
                                [-3.16407561e-01, 3.85351181e00, 5.33868372e-01],
                                [-1.10977542e00, 3.82791996e00, 6.06707633e-01],
                                [-1.46703672e00, 3.94862103e00, 6.45296216e-01],
                                [-1.14162290e00, 3.79930806e00, 7.45548725e-01],
                                [-1.56181908e00, 3.68108273e00, 6.49301231e-01],
                                [-1.46823406e00, 3.63365412e00, 5.03535330e-01],
                                [-1.02590537e00, 3.64477968e00, 5.22250295e-01],
                                [-5.35379410e-01, 3.69151831e00, 4.32526857e-01],
                                [-5.78065366e-02, 3.37806726e00, 4.95986529e-02],
                                [-2.40110800e-01, 3.22970843e00, -9.55391768e-03],
                                [-6.54527247e-01, 3.16465378e00, 1.89672917e-01],
                                [-1.35780311e00, 3.24750924e00, 1.63008988e-01],
                                [-1.47417045e00, 3.39788198e00, 1.84387550e-01],
                                [-1.74577117e00, 3.53278613e00, 1.89606979e-01],
                                [-1.46304774e00, 3.22666740e00, 1.59070507e-01],
                                [-4.05473042e00, 3.06258607e00, 1.10443914e00],
                                [-5.91582203e00, 2.83895302e00, 1.79846287e00],
                                [-7.04713678e00, 2.55699897e00, 2.23985386e00],
                                [-7.43741798e00, 2.21711683e00, 2.42266488e00],
                                [-7.06249666e00, 1.81872594e00, 2.33713675e00],
                                [-5.96051836e00, 1.36609375e00, 1.99506497e00],
                                [-4.13137341e00, 8.60225558e-01, 1.39551389e00],
                                [-1.57741416e00, 3.02793205e-01, 5.36690295e-01],
                                [7.63817742e-12, 1.11727738e-10, 2.74194088e-11],
                                [-1.24213686e-11, 8.01499769e-11, -1.34056446e-11],
                                [1.78272761e-11, 9.04948685e-11, 1.77389995e-11],
                                [-1.47259357e-11, 9.37734057e-11, -3.89882709e-11],
                                [-1.94052344e-11, 1.49411969e-10, -2.48492286e-11],
                                [3.40418752e-12, 1.25333730e-10, 1.14499972e-11],
                                [5.64890669e-12, 1.35170833e-10, 2.27858565e-11],
                                [8.78361273e-12, 1.02109009e-10, -5.92111386e-12],
                                [1.47398396e-11, 8.59943505e-11, -8.54686872e-13],
                                [-5.35027361e-12, 1.25450331e-10, -1.54262800e-11],
                                [2.12667054e-11, 1.57356642e-10, 2.54392306e-11],
                                [-6.39285022e-12, 1.42791029e-10, -8.58749790e-12],
                                [-2.18451160e-11, 9.94347313e-11, -2.18451160e-11],
                                [1.77587389e-11, 1.16834944e-10, 3.09037483e-11],
                                [5.09583955e-12, 1.06878555e-10, 1.30452402e-11],
                                [-1.25895900e-11, 1.06217646e-10, -1.07971496e-11],
                                [1.45264981e-11, 1.03935242e-10, 1.73963136e-11],
                                [-1.41730258e-12, 7.72037989e-11, 1.15057850e-11],
                                [1.99397634e-11, 1.36618120e-10, 4.70010628e-11],
                                [1.24784124e-11, 1.14499965e-10, 4.69658253e-12],
                                [-1.83001236e-11, 5.19546177e-11, -1.31873679e-11],
                                [-9.99299988e02, -9.99299988e02, -9.99299988e02],
                            ],
                            dtype=np.float32,
                        )
                    },
                    "SCPosition": {
                        "value": np.array(
                            [
                                [2.3191672e06, -4.5127075e06, 5.1096645e06],
                                [2.3202438e06, -4.5225140e06, 5.1005205e06],
                                [2.3213098e06, -4.5323050e06, 5.0913595e06],
                                [2.3223650e06, -4.5420810e06, 5.0821800e06],
                                [2.3234100e06, -4.5518415e06, 5.0729835e06],
                                [2.3244445e06, -4.5615875e06, 5.0637700e06],
                                [2.3254692e06, -4.5713185e06, 5.0545390e06],
                                [2.3264830e06, -4.5810340e06, 5.0452915e06],
                                [2.3274862e06, -4.5907340e06, 5.0360255e06],
                                [2.3284792e06, -4.6004185e06, 5.0267430e06],
                                [2.3294620e06, -4.6100885e06, 5.0174430e06],
                                [2.3304345e06, -4.6197430e06, 5.0081270e06],
                                [2.3313962e06, -4.6293820e06, 4.9987935e06],
                                [2.3323475e06, -4.6390050e06, 4.9894420e06],
                                [2.3332888e06, -4.6486130e06, 4.9800740e06],
                                [2.3342195e06, -4.6582060e06, 4.9706890e06],
                                [2.3351398e06, -4.6677835e06, 4.9612880e06],
                                [2.3360495e06, -4.6773440e06, 4.9518685e06],
                                [2.3369522e06, -4.6868750e06, 4.9424430e06],
                                [2.3378502e06, -4.6963695e06, 4.9330150e06],
                                [2.3387432e06, -4.7058270e06, 4.9235845e06],
                                [2.3396312e06, -4.7152475e06, 4.9141520e06],
                                [2.3405140e06, -4.7246290e06, 4.9047175e06],
                                [2.3413915e06, -4.7339725e06, 4.8952825e06],
                                [2.3422642e06, -4.7432805e06, 4.8858430e06],
                                [2.3431318e06, -4.7525505e06, 4.8764035e06],
                                [2.3439710e06, -4.7618790e06, 4.8668965e06],
                                [2.3447770e06, -4.7712820e06, 4.8573130e06],
                                [2.3455728e06, -4.7806710e06, 4.8477115e06],
                                [2.3463582e06, -4.7900425e06, 4.8380950e06],
                                [2.3471335e06, -4.7994005e06, 4.8284610e06],
                                [2.3478980e06, -4.8087395e06, 4.8188110e06],
                                [2.3486522e06, -4.8180645e06, 4.8091435e06],
                                [2.3493960e06, -4.8273715e06, 4.7994615e06],
                                [2.3501298e06, -4.8366645e06, 4.7897610e06],
                                [2.3508530e06, -4.8459395e06, 4.7800465e06],
                                [2.3515658e06, -4.8552000e06, 4.7703130e06],
                                [2.3522680e06, -4.8644420e06, 4.7605655e06],
                                [2.3529602e06, -4.8736700e06, 4.7508000e06],
                                [2.3536420e06, -4.8828800e06, 4.7410205e06],
                                [2.3543132e06, -4.8920755e06, 4.7312230e06],
                                [2.3549740e06, -4.9012520e06, 4.7214105e06],
                                [2.3556248e06, -4.9104145e06, 4.7115800e06],
                                [2.3562650e06, -4.9195590e06, 4.7017360e06],
                                [2.3568952e06, -4.9286890e06, 4.6918745e06],
                                [2.3575145e06, -4.9378000e06, 4.6819980e06],
                                [2.3581235e06, -4.9468960e06, 4.6721035e06],
                                [-9.9929999e02, -9.9929999e02, -9.9929999e02],
                            ],
                            dtype=np.float32,
                        )
                    },
                    "SCSolarAzimuthAngle": {
                        "value": np.array(
                            [
                                -140.6137,
                                -140.54446,
                                -140.47484,
                                -140.40486,
                                -140.33464,
                                -140.26427,
                                -140.19333,
                                -140.12198,
                                -140.05042,
                                -139.97855,
                                -139.90648,
                                -139.83394,
                                -139.76117,
                                -139.68803,
                                -139.61465,
                                -139.54103,
                                -139.46695,
                                -139.3923,
                                -139.31741,
                                -139.2424,
                                -139.16727,
                                -139.09201,
                                -139.01662,
                                -138.94112,
                                -138.86546,
                                -138.78972,
                                -138.71251,
                                -138.63487,
                                -138.5569,
                                -138.4786,
                                -138.39995,
                                -138.32097,
                                -138.24161,
                                -138.16193,
                                -138.0819,
                                -138.00153,
                                -137.92078,
                                -137.8397,
                                -137.75827,
                                -137.67648,
                                -137.59433,
                                -137.51183,
                                -137.42896,
                                -137.34573,
                                -137.26213,
                                -137.17819,
                                -137.09386,
                                -999.3,
                            ],
                            dtype=np.float32,
                        )
                    },
                    "SCSolarZenithAngle": {
                        "value": np.array(
                            [
                                135.88528,
                                135.96703,
                                136.04868,
                                136.1302,
                                136.21165,
                                136.2931,
                                136.37451,
                                136.4556,
                                136.53659,
                                136.61748,
                                136.69843,
                                136.77931,
                                136.86021,
                                136.94092,
                                137.02148,
                                137.10208,
                                137.18248,
                                137.26239,
                                137.34204,
                                137.42155,
                                137.50092,
                                137.58014,
                                137.65923,
                                137.73816,
                                137.81696,
                                137.8956,
                                137.97507,
                                138.05447,
                                138.13382,
                                138.21303,
                                138.29218,
                                138.37122,
                                138.45016,
                                138.529,
                                138.60777,
                                138.68642,
                                138.76498,
                                138.84343,
                                138.9218,
                                139.00005,
                                139.07823,
                                139.15627,
                                139.23422,
                                139.31207,
                                139.38983,
                                139.46748,
                                139.54503,
                                -999.3,
                            ],
                            dtype=np.float32,
                        )
                    },
                    "SCVelocity": {
                        "value": np.array(
                            [
                                [605.31726, -5492.9614, -5113.397],
                                [599.4935, -5484.5615, -5123.1396],
                                [593.66986, -5476.142, -5132.8657],
                                [587.8464, -5467.7017, -5142.573],
                                [582.02313, -5459.241, -5152.263],
                                [576.19995, -5450.7607, -5161.936],
                                [570.37714, -5442.2607, -5171.592],
                                [564.5546, -5433.741, -5181.2295],
                                [558.73236, -5425.2, -5190.849],
                                [552.9104, -5416.6396, -5200.4517],
                                [547.0887, -5408.06, -5210.0366],
                                [541.26746, -5399.4604, -5219.6035],
                                [535.44666, -5390.841, -5229.153],
                                [529.6263, -5382.201, -5238.684],
                                [523.8063, -5373.5415, -5248.1978],
                                [517.9866, -5364.863, -5257.694],
                                [512.16754, -5356.1646, -5267.1724],
                                [506.34906, -5347.446, -5276.632],
                                [500.53455, -5338.72, -5286.0645],
                                [494.72552, -5329.993, -5295.466],
                                [488.9218, -5321.265, -5304.8364],
                                [483.1238, -5312.536, -5314.1743],
                                [477.33157, -5303.806, -5323.4795],
                                [471.546, -5295.0767, -5332.7515],
                                [465.7647, -5286.344, -5341.9937],
                                [459.99005, -5277.613, -5351.2026],
                                [454.19785, -5268.798, -5360.442],
                                [448.38614, -5259.887, -5369.7207],
                                [442.57404, -5250.955, -5378.983],
                                [436.7639, -5242.0063, -5388.225],
                                [430.9534, -5233.0366, -5397.4517],
                                [425.145, -5224.0483, -5406.6567],
                                [419.33627, -5215.0396, -5415.845],
                                [413.52963, -5206.013, -5425.014],
                                [407.72275, -5196.9663, -5434.1665],
                                [401.91797, -5187.9023, -5443.299],
                                [396.11307, -5178.8164, -5452.4136],
                                [390.3103, -5169.7134, -5461.508],
                                [384.50742, -5160.59, -5470.586],
                                [378.70673, -5151.4497, -5479.644],
                                [372.90598, -5142.288, -5488.6846],
                                [367.1075, -5133.109, -5497.7046],
                                [361.309, -5123.9097, -5506.708],
                                [355.5128, -5114.6934, -5515.691],
                                [349.71658, -5105.4565, -5524.657],
                                [343.9228, -5096.202, -5533.602],
                                [338.12906, -5086.927, -5542.53],
                                [-999.3, -999.3, -999.3],
                            ],
                            dtype=np.float32,
                        )
                    },
                    "SatelliteAzimuthAngle": {
                        "value": np.random.rand(96, 332).astype(np.float32)
                    },
                    "SatelliteZenithAngle": {
                        "value": np.random.rand(96, 332).astype(np.float32)
                    },
                    "SolarAzimuthAngle": {
                        "value": np.random.rand(96, 332).astype(np.float32)
                    },
                    "SolarZenithAngle": {
                        "value": np.random.rand(96, 332).astype(np.float32)
                    },
                    "StartTime": {
                        "value": np.array(
                            [
                                1950675122120971,
                                1950675123907557,
                                1950675125694139,
                                1950675127480722,
                                1950675129267304,
                                1950675131053910,
                                1950675132840494,
                                1950675134627077,
                                1950675136413660,
                                1950675138200243,
                                1950675139986850,
                                1950675141773433,
                                1950675143560016,
                                1950675145346598,
                                1950675147133181,
                                1950675148919788,
                                1950675150706371,
                                1950675152492953,
                                1950675154279537,
                                1950675156066119,
                                1950675157852726,
                                1950675159639309,
                                1950675161425892,
                                1950675163212109,
                                1950675164998692,
                                1950675166784909,
                                1950675168571492,
                                1950675170357709,
                                1950675172144292,
                                1950675173930509,
                                1950675175717092,
                                1950675177503309,
                                1950675179289892,
                                1950675181076109,
                                1950675182862692,
                                1950675184648909,
                                1950675186435492,
                                1950675188221709,
                                1950675190008292,
                                1950675191794509,
                                1950675193581092,
                                1950675195367309,
                                1950675197153892,
                                1950675198940109,
                                1950675200726692,
                                1950675202512909,
                                1950675204299492,
                                -993,
                            ]
                        )
                    },
                    "TiePointZoneGroupLocationScanCompact": {
                        "value": np.array(
                            [
                                0,
                                2,
                                4,
                                6,
                                8,
                                10,
                                12,
                                14,
                                16,
                                45,
                                48,
                                52,
                                55,
                                59,
                                63,
                                67,
                                73,
                                78,
                                84,
                                89,
                                94,
                                99,
                                104,
                                109,
                                113,
                                119,
                                123,
                                128,
                                132,
                                156,
                                180,
                                184,
                                189,
                                193,
                                199,
                                203,
                                208,
                                213,
                                218,
                                223,
                                228,
                                234,
                                239,
                                245,
                                249,
                                253,
                                257,
                                260,
                                264,
                                267,
                                308,
                                310,
                                312,
                                314,
                                316,
                                318,
                                320,
                                322,
                                324,
                                326,
                                328,
                                330,
                            ],
                            dtype=np.int32,
                        )
                    },
                    "TiePointZoneGroupLocationTrackCompact": {"value": 0},
                    "attrs": {
                        "OriginalFilename": np.array(
                            [
                                [
                                    b"GDNBO_j01_d20191025_t0611251_e0612478_b10015_c20191025062405837630_cspp_dev.h5"
                                ]
                            ],
                            dtype="|S78",
                        )
                    },
                },
                "VIIRS-DNB-SDR_All": {
                    "NumberOfBadChecksums": {
                        "value": np.array(
                            [
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                -993,
                            ],
                            dtype=np.int32,
                        )
                    },
                    "NumberOfDiscardedPkts": {
                        "value": np.array(
                            [
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                -993,
                            ],
                            dtype=np.int32,
                        )
                    },
                    "NumberOfMissingPkts": {
                        "value": np.array(
                            [
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                17,
                                18,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                479,
                                -993,
                            ],
                            dtype=np.int32,
                        )
                    },
                    "PadByte1": {"value": np.array([0, 0, 0], dtype=np.uint8)},
                    "QF1_VIIRSDNBSDR": {
                        "value": (np.random.rand(768, 4064) * 255).astype(np.uint8)
                    },
                    "QF2_SCAN_SDR": {
                        "value": np.array(
                            [
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                1,
                                0,
                                0,
                            ],
                            dtype=np.uint8,
                        )
                    },
                    "QF3_SCAN_RDR": {
                        "value": np.array(
                            [
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                                64,
                            ],
                            dtype=np.uint8,
                        )
                    },
                    "Radiance": {"value": np.random.rand(768, 4064).astype(np.float32)},
                    "attrs": {
                        "OriginalFilename": np.array(
                            [
                                [
                                    b"SVDNB_j01_d20191025_t0611251_e0612478_b10015_c20191025062427398006_cspp_dev.h5"
                                ]
                            ],
                            dtype="|S78",
                        ),
                        "PixelOffsetScan": np.array([[0.5]], dtype=np.float32),
                        "PixelOffsetTrack": np.array([[0.5]], dtype=np.float32),
                        "TiePointZoneGroupLocationScan": np.array(
                            [
                                [0],
                                [2],
                                [4],
                                [6],
                                [8],
                                [10],
                                [12],
                                [14],
                                [16],
                                [464],
                                [496],
                                [544],
                                [576],
                                [648],
                                [720],
                                [792],
                                [872],
                                [928],
                                [1008],
                                [1072],
                                [1136],
                                [1200],
                                [1264],
                                [1328],
                                [1400],
                                [1480],
                                [1552],
                                [1640],
                                [1712],
                                [1896],
                                [2080],
                                [2152],
                                [2240],
                                [2312],
                                [2392],
                                [2464],
                                [2528],
                                [2592],
                                [2656],
                                [2720],
                                [2784],
                                [2864],
                                [2920],
                                [3000],
                                [3072],
                                [3144],
                                [3216],
                                [3248],
                                [3296],
                                [3328],
                                [3968],
                                [3976],
                                [3984],
                                [3992],
                                [4000],
                                [4008],
                                [4016],
                                [4024],
                                [4032],
                                [4040],
                                [4048],
                                [4056],
                            ],
                            dtype=np.int32,
                        ),
                        "TiePointZoneGroupLocationTrack": np.array(
                            [[0]], dtype=np.int32
                        ),
                        "TiePointZoneSizeScan": np.array(
                            [
                                [2],
                                [2],
                                [2],
                                [2],
                                [2],
                                [2],
                                [2],
                                [2],
                                [16],
                                [16],
                                [16],
                                [16],
                                [24],
                                [24],
                                [24],
                                [16],
                                [14],
                                [16],
                                [16],
                                [16],
                                [16],
                                [16],
                                [16],
                                [24],
                                [16],
                                [24],
                                [22],
                                [24],
                                [8],
                                [8],
                                [24],
                                [22],
                                [24],
                                [16],
                                [24],
                                [16],
                                [16],
                                [16],
                                [16],
                                [16],
                                [16],
                                [14],
                                [16],
                                [24],
                                [24],
                                [24],
                                [16],
                                [16],
                                [16],
                                [16],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                                [8],
                            ],
                            dtype=np.int32,
                        ),
                        "TiePointZoneSizeTrack": np.array([[16]], dtype=np.int32),
                    },
                },
                "attrs": {"MissionStartTime": np.array([[1698019234000000]])},
            },
            "Data_Products": {
                "VIIRS-DNB-GEO": {
                    "VIIRS-DNB-GEO_Aggr": {
                        "attrs": {
                            "AggregateBeginningDate": np.array(
                                [[b"20191025"]], dtype="|S9"
                            ),
                            "AggregateBeginningGranuleID": np.array(
                                [[b"J01002526558865"]], dtype="|S16"
                            ),
                            "AggregateBeginningOrbitNumber": np.array(
                                [[10015]], dtype=np.uint64
                            ),
                            "AggregateBeginningTime": np.array(
                                [[b"061125.120971Z"]], dtype="|S15"
                            ),
                            "AggregateEndingDate": np.array(
                                [[b"20191025"]], dtype="|S9"
                            ),
                            "AggregateEndingGranuleID": np.array(
                                [[b"J01002526558865"]], dtype="|S16"
                            ),
                            "AggregateEndingOrbitNumber": np.array(
                                [[10015]], dtype=np.uint64
                            ),
                            "AggregateEndingTime": np.array(
                                [[b"061247.849492Z"]], dtype="|S15"
                            ),
                            "AggregateNumberGranules": np.array([[1]], dtype=np.uint64),
                        }
                    },
                    "VIIRS-DNB-GEO_Gran_0": {
                        "attrs": {
                            "Ascending/Descending_Indicator": np.array(
                                [[1]], dtype=np.uint8
                            ),
                            "Beginning_Date": np.array([[b"20191025"]], dtype="|S9"),
                            "Beginning_Time": np.array(
                                [[b"061125.120971Z"]], dtype="|S15"
                            ),
                            "East_Bounding_Coordinate": np.array(
                                [[-45.09228]], dtype=np.float32
                            ),
                            "Ending_Date": np.array([[b"20191025"]], dtype="|S9"),
                            "Ending_Time": np.array(
                                [[b"061247.849492Z"]], dtype="|S15"
                            ),
                            "G-Ring_Latitude": np.array(
                                [
                                    [41.84151],
                                    [44.31062],
                                    [46.78565],
                                    [45.41409],
                                    [41.07657],
                                    [38.81504],
                                    [36.53401],
                                    [40.55788],
                                ],
                                dtype=np.float32,
                            ),
                            "G-Ring_Longitude": np.array(
                                [
                                    [-82.66234],
                                    [-82.55624],
                                    [-82.48891],
                                    [-62.80042],
                                    [-45.09228],
                                    [-46.58502],
                                    [-47.95933],
                                    [-64.54196],
                                ],
                                dtype=np.float32,
                            ),
                            "LeapSecondsGranuleStart": np.array([[37]], dtype=np.int32),
                            "N_Algorithm_Version": np.array(
                                [[b"1.O.000.014"]], dtype="|S12"
                            ),
                            "N_Anc_Filename": np.array(
                                [
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0691_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0692_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0693_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0719_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0720_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0721_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0722_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0723_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0724_1.O.0.0"
                                    ],
                                    [
                                        b"Terrain-Eco-ANC-Tile_20030125000000Z_ee00000000000000Z_NA_NA_N0725_1.O.0.0"
                                    ],
                                    [
                                        b"off_Planet-Eph-ANC_Static_JPL_000f_20151008_200001010000Z_20000101000000Z_ee00000000000000Z_np"  # noqa
                                    ],
                                    [
                                        b"off_USNO-PolarWander-UT1-ANC_Ser7_USNO_000f_20191025_201910250000Z_20191025000109Z_ee20191101120000Z_np"  # noqa
                                    ],
                                ],
                                dtype="|S104",
                            ),
                            "N_Aux_Filename": np.array(
                                [
                                    [
                                        b"CMNGEO-PARAM-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"CmnGeo-SAA-AC_j01_20151008180000Z_20170807130000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"TLE-AUX_j01_20191024053224Z_20191024000000Z_ee00000000000000Z_-_nobc_ops_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-GEO-DNB-PARAM-LUT_j01_20180507121508Z_20180315000000Z_ee00000000000000Z_PS-1-O-CCR-3963-006-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-GEO-IMG-PARAM-LUT_j01_20180430182354Z_20180315000000Z_ee00000000000000Z_PS-1-O-CCR-3963-006-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-GEO-MOD-PARAM-LUT_j01_20180430182652Z_20180315000000Z_ee00000000000000Z_PS-1-O-CCR-3963-006-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-QA-LUT_j01_20180109121411Z_20180409000000Z_ee00000000000000Z_PS-1-O-CCR-3742-003-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                ],
                                dtype="|S126",
                            ),
                            "N_Beginning_Orbit_Number": np.array(
                                [[10015]], dtype=np.uint64
                            ),
                            "N_Beginning_Time_IET": np.array(
                                [[1950675122120971]], dtype=np.uint64
                            ),
                            "N_Creation_Date": np.array([[b"20191025"]], dtype="|S9"),
                            "N_Creation_Time": np.array(
                                [[b"062136.412867Z"]], dtype="|S15"
                            ),
                            "N_Day_Night_Flag": np.array([[b"Night"]], dtype="|S6"),
                            "N_Ending_Time_IET": np.array(
                                [[1950675204849492]], dtype=np.uint64
                            ),
                            "N_Granule_ID": np.array(
                                [[b"J01002526558865"]], dtype="|S16"
                            ),
                            "N_Granule_Status": np.array([[b"N/A"]], dtype="|S4"),
                            "N_Granule_Version": np.array([[b"A1"]], dtype="|S3"),
                            "N_IDPS_Mode": np.array([[b"N/A"]], dtype="|S4"),
                            "N_Input_Prod": np.array(
                                [
                                    [b"SPACECRAFT-DIARY-RDR:J01002526558800:A1"],
                                    [b"SPACECRAFT-DIARY-RDR:J01002526559000:A1"],
                                    [b"VIIRS-SCIENCE-RDR:J01002526558865:A1"],
                                ],
                                dtype="|S40",
                            ),
                            "N_JPSS_Document_Ref": np.array(
                                [
                                    [
                                        b"474-00448-02-06_JPSS-DD-Vol-II-Part-6_0200H.pdf"
                                    ],
                                    [
                                        b"474-00448-02-06_JPSS-VIIRS-SDR-DD-Part-6_0200H_VIIRS-DNB-GEO-PP.xml"
                                    ],
                                    [
                                        b"474-00448-03-06_JPSS-OAD-Vol-III-Part-6-VIIRS-RDR-SDR_-1.pdf"
                                    ],
                                ],
                                dtype="|S68",
                            ),
                            "N_LEOA_Flag": np.array([[b"On"]], dtype="|S3"),
                            "N_Nadir_Latitude_Max": np.array(
                                [[45.3722]], dtype=np.float32
                            ),
                            "N_Nadir_Latitude_Min": np.array(
                                [[40.6172]], dtype=np.float32
                            ),
                            "N_Nadir_Longitude_Max": np.array(
                                [[-62.80047]], dtype=np.float32
                            ),
                            "N_Nadir_Longitude_Min": np.array(
                                [[-64.51342]], dtype=np.float32
                            ),
                            "N_Number_Of_Scans": np.array([[47]], dtype=np.int32),
                            "N_Primary_Label": np.array(
                                [[b"Non-Primary"]], dtype="|S12"
                            ),
                            "N_Quality_Summary_Names": np.array(
                                [
                                    [b"Automatic Quality Flag"],
                                    [b"Percent Missing Data"],
                                    [b"Percent Out of Bounds"],
                                ],
                                dtype="|S23",
                            ),
                            "N_Quality_Summary_Values": np.array(
                                [[1], [61], [0]], dtype=np.int32
                            ),
                            "N_Reference_ID": np.array(
                                [[b"VIIRS-DNB-GEO:J01002526558865:A1"]], dtype="|S33"
                            ),
                            "N_Software_Version": np.array(
                                [[b"CSPP_SDR_3_1_3"]], dtype="|S15"
                            ),
                            "N_Spacecraft_Maneuver": np.array(
                                [[b"Normal Operations"]], dtype="|S18"
                            ),
                            "North_Bounding_Coordinate": np.array(
                                [[46.8018]], dtype=np.float32
                            ),
                            "South_Bounding_Coordinate": np.array(
                                [[36.53401]], dtype=np.float32
                            ),
                            "West_Bounding_Coordinate": np.array(
                                [[-82.66234]], dtype=np.float32
                            ),
                        }
                    },
                    "attrs": {
                        "Instrument_Short_Name": np.array([[b"VIIRS"]], dtype="|S6"),
                        "N_Anc_Type_Tasked": np.array([[b"Official"]], dtype="|S9"),
                        "N_Collection_Short_Name": np.array(
                            [[b"VIIRS-DNB-GEO"]], dtype="|S14"
                        ),
                        "N_Dataset_Type_Tag": np.array([[b"GEO"]], dtype="|S4"),
                        "N_Processing_Domain": np.array([[b"ops"]], dtype="|S4"),
                        "Operational_Mode": np.array(
                            [[b"J01 Normal Operations, VIIRS Operational"]],
                            dtype="|S41",
                        ),
                    },
                },
                "VIIRS-DNB-SDR": {
                    "VIIRS-DNB-SDR_Aggr": {
                        "attrs": {
                            "AggregateBeginningDate": np.array(
                                [[b"20191025"]], dtype="|S9"
                            ),
                            "AggregateBeginningGranuleID": np.array(
                                [[b"J01002526558865"]], dtype="|S16"
                            ),
                            "AggregateBeginningOrbitNumber": np.array(
                                [[10015]], dtype=np.uint64
                            ),
                            "AggregateBeginningTime": np.array(
                                [[b"061125.120971Z"]], dtype="|S15"
                            ),
                            "AggregateEndingDate": np.array(
                                [[b"20191025"]], dtype="|S9"
                            ),
                            "AggregateEndingGranuleID": np.array(
                                [[b"J01002526558865"]], dtype="|S16"
                            ),
                            "AggregateEndingOrbitNumber": np.array(
                                [[10015]], dtype=np.uint64
                            ),
                            "AggregateEndingTime": np.array(
                                [[b"061247.849492Z"]], dtype="|S15"
                            ),
                            "AggregateNumberGranules": np.array([[1]], dtype=np.uint64),
                        }
                    },
                    "VIIRS-DNB-SDR_Gran_0": {
                        "attrs": {
                            "Ascending/Descending_Indicator": np.array(
                                [[1]], dtype=np.uint8
                            ),
                            "Band_ID": np.array([[b"N/A"]], dtype="|S4"),
                            "Beginning_Date": np.array([[b"20191025"]], dtype="|S9"),
                            "Beginning_Time": np.array(
                                [[b"061125.120971Z"]], dtype="|S15"
                            ),
                            "East_Bounding_Coordinate": np.array(
                                [[-45.09281]], dtype=np.float32
                            ),
                            "Ending_Date": np.array([[b"20191025"]], dtype="|S9"),
                            "Ending_Time": np.array(
                                [[b"061247.849492Z"]], dtype="|S15"
                            ),
                            "G-Ring_Latitude": np.array(
                                [
                                    [41.84157],
                                    [44.31069],
                                    [46.78591],
                                    [45.41409],
                                    [41.07675],
                                    [38.81512],
                                    [36.53402],
                                    [40.55788],
                                ],
                                dtype=np.float32,
                            ),
                            "G-Ring_Longitude": np.array(
                                [
                                    [-82.65787],
                                    [-82.55148],
                                    [-82.47269],
                                    [-62.80042],
                                    [-45.09281],
                                    [-46.58528],
                                    [-47.95936],
                                    [-64.54196],
                                ],
                                dtype=np.float32,
                            ),
                            "N_Algorithm_Version": np.array(
                                [[b"1.O.000.015"]], dtype="|S12"
                            ),
                            "N_Anc_Filename": np.array(
                                [
                                    [
                                        b"off_Planet-Eph-ANC_Static_JPL_000f_20151008_200001010000Z_20000101000000Z_ee00000000000000Z_np"  # noqa
                                    ],
                                    [
                                        b"off_USNO-PolarWander-UT1-ANC_Ser7_USNO_000f_20191025_201910250000Z_20191025000109Z_ee20191101120000Z_np"  # noqa
                                    ],
                                ],
                                dtype="|S104",
                            ),
                            "N_Aux_Filename": np.array(
                                [
                                    [
                                        b"CMNGEO-PARAM-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-DNB-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-I1-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-I2-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-I3-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-I4-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-I5-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M1-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M10-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M11-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M12-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M13-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M14-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M15-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M16-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M2-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M3-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M4-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M5-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M6-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M7-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M8-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-M9-SDR-DQTT_j01_20151008180000Z_20020101010000Z_ee00000000000000Z_PS-1-O-NPP-1-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-RSBAUTOCAL-HISTORY-AUX_j01_20191024021527Z_20191024000000Z_ee00000000000000Z_-_nobc_ops_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-RSBAUTOCAL-VOLT-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-EDD154640-109C-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-BB-TEMP-COEFFS-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-CAL-AUTOMATE-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-Pred-SideA-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-COEFF-A-LUT_j01_20180109114311Z_20180409000000Z_ee00000000000000Z_PS-1-O-CCR-3742-003-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-COEFF-B-LUT_j01_20180109101739Z_20180409000000Z_ee00000000000000Z_PS-1-O-CCR-3742-004-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DELTA-C-LUT_j01_20180109000000Z_20180409000000Z_ee00000000000000Z_PS-1-O-CCR-3742-003-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DG-ANOMALY-DN-LIMITS-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-SideA-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DNB-DN0-LUT_j01_20190930000000Z_20190928000000Z_ee00000000000000Z_PS-1-O-CCR-4262-026-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DNB-FRAME-TO-ZONE-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-Op21-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DNB-GAIN-RATIOS-LUT_j01_20190930000000Z_20190928000000Z_ee00000000000000Z_PS-1-O-CCR-4262-025-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DNB-LGS-GAINS-LUT_j01_20180413122703Z_20180412000000Z_ee00000000000000Z_PS-1-O-CCR-3918-005-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DNB-RVF-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-Op21-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-DNB-STRAY-LIGHT-CORRECTION-LUT_j01_20190930160523Z_20191001000000Z_ee00000000000000Z_PS-1-O-CCR-4322-024-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-EBBT-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-EMISSIVE-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-F-PREDICTED-LUT_j01_20180413123333Z_20180412000000Z_ee00000000000000Z_PS-1-O-CCR-3918-006-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-GAIN-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-HAM-ER-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-OBC-ER-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-OBC-RR-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-OBS-TO-PIXELS-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-SameAsSNPP-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-QA-LUT_j01_20180109121411Z_20180409000000Z_ee00000000000000Z_PS-1-O-CCR-3742-003-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-RADIOMETRIC-PARAM-V3-LUT_j01_20161117000000Z_20180111000000Z_ee00000000000000Z_PS-1-O-CCR-17-3436-v003-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-REFLECTIVE-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-SameAsSNPP-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-RELATIVE-SPECTRAL-RESPONSE-LUT_j01_20161031000000Z_20180111000000Z_ee00000000000000Z_PS-1-O-CCR-17-3436-v003-FusedM9-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-RTA-ER-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-RVF-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-M16-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-SOLAR-IRAD-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-Thuillier2002-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                    [
                                        b"VIIRS-SDR-TELE-COEFFS-LUT_j01_20160331000000Z_20170807130000Z_ee00000000000000Z_PS-1-O-CCR-16-2859-v002-SideA-LE-PE_all-_all_all-_ops"  # noqa
                                    ],
                                ],
                                dtype="|S151",
                            ),
                            "N_Beginning_Orbit_Number": np.array(
                                [[10015]], dtype=np.uint64
                            ),
                            "N_Beginning_Time_IET": np.array(
                                [[1950675122120971]], dtype=np.uint64
                            ),
                            "N_Creation_Date": np.array([[b"20191025"]], dtype="|S9"),
                            "N_Creation_Time": np.array(
                                [[b"062411.116253Z"]], dtype="|S15"
                            ),
                            "N_Day_Night_Flag": np.array([[b"Night"]], dtype="|S6"),
                            "N_Ending_Time_IET": np.array(
                                [[1950675204849492]], dtype=np.uint64
                            ),
                            "N_Graceful_Degradation": np.array([[b"No"]], dtype="|S3"),
                            "N_Granule_ID": np.array(
                                [[b"J01002526558865"]], dtype="|S16"
                            ),
                            "N_Granule_Status": np.array([[b"N/A"]], dtype="|S4"),
                            "N_Granule_Version": np.array([[b"A1"]], dtype="|S3"),
                            "N_IDPS_Mode": np.array([[b"N/A"]], dtype="|S4"),
                            "N_Input_Prod": np.array(
                                [
                                    [b"GEO-VIIRS-OBC-IP:J01002526558865:A1"],
                                    [b"SPACECRAFT-DIARY-RDR:J01002526558800:A1"],
                                    [b"SPACECRAFT-DIARY-RDR:J01002526559000:A1"],
                                    [b"VIIRS-DNB-GEO:J01002526558865:A1"],
                                    [b"VIIRS-IMG-RGEO-TC:J01002526558865:A1"],
                                    [b"VIIRS-MOD-RGEO-TC:J01002526558865:A1"],
                                    [b"VIIRS-SCIENCE-RDR:J01002526558012:A1"],
                                    [b"VIIRS-SCIENCE-RDR:J01002526558865:A1"],
                                ],
                                dtype="|S40",
                            ),
                            "N_JPSS_Document_Ref": np.array(
                                [
                                    [
                                        b"474-00448-02-06_JPSS-DD-Vol-II-Part-6_0200H.pdf"
                                    ],
                                    [
                                        b"474-00448-02-06_JPSS-VIIRS-SDR-DD-Part-6_0200H_VIIRS-DNB-SDR-PP.xml"
                                    ],
                                    [
                                        b"474-00448-03-06_JPSS-OAD-Vol-III-Part-6-VIIRS-RDR-SDR_-1.pdf"
                                    ],
                                ],
                                dtype="|S68",
                            ),
                            "N_LEOA_Flag": np.array([[b"On"]], dtype="|S3"),
                            "N_Nadir_Latitude_Max": np.array(
                                [[45.3722]], dtype=np.float32
                            ),
                            "N_Nadir_Latitude_Min": np.array(
                                [[40.6172]], dtype=np.float32
                            ),
                            "N_Nadir_Longitude_Max": np.array(
                                [[-62.80047]], dtype=np.float32
                            ),
                            "N_Nadir_Longitude_Min": np.array(
                                [[-64.51342]], dtype=np.float32
                            ),
                            "N_Number_Of_Scans": np.array([[47]], dtype=np.int32),
                            "N_Percent_Erroneous_Data": np.array(
                                [[0.0]], dtype=np.float32
                            ),
                            "N_Percent_Missing_Data": np.array(
                                [[51.05127]], dtype=np.float32
                            ),
                            "N_Percent_Not-Applicable_Data": np.array(
                                [[0.0]], dtype=np.float32
                            ),
                            "N_Primary_Label": np.array(
                                [[b"Non-Primary"]], dtype="|S12"
                            ),
                            "N_Quality_Summary_Names": np.array(
                                [
                                    [b"Scan Quality Exclusion"],
                                    [b"Summary VIIRS SDR Quality"],
                                ],
                                dtype="|S26",
                            ),
                            "N_Quality_Summary_Values": np.array(
                                [[24], [49]], dtype=np.int32
                            ),
                            "N_RSB_Index": np.array([[17]], dtype=np.int32),
                            "N_Reference_ID": np.array(
                                [[b"VIIRS-DNB-SDR:J01002526558865:A1"]], dtype="|S33"
                            ),
                            "N_Satellite/Local_Azimuth_Angle_Max": np.array(
                                [[179.9995]], dtype=np.float32
                            ),
                            "N_Satellite/Local_Azimuth_Angle_Min": np.array(
                                [[-179.9976]], dtype=np.float32
                            ),
                            "N_Satellite/Local_Zenith_Angle_Max": np.array(
                                [[69.83973]], dtype=np.float32
                            ),
                            "N_Satellite/Local_Zenith_Angle_Min": np.array(
                                [[0.00898314]], dtype=np.float32
                            ),
                            "N_Software_Version": np.array(
                                [[b"CSPP_SDR_3_1_3"]], dtype="|S15"
                            ),
                            "N_Solar_Azimuth_Angle_Max": np.array(
                                [[73.93496]], dtype=np.float32
                            ),
                            "N_Solar_Azimuth_Angle_Min": np.array(
                                [[23.83542]], dtype=np.float32
                            ),
                            "N_Solar_Zenith_Angle_Max": np.array(
                                [[147.5895]], dtype=np.float32
                            ),
                            "N_Solar_Zenith_Angle_Min": np.array(
                                [[126.3929]], dtype=np.float32
                            ),
                            "N_Spacecraft_Maneuver": np.array(
                                [[b"Normal Operations"]], dtype="|S18"
                            ),
                            "North_Bounding_Coordinate": np.array(
                                [[46.8018]], dtype=np.float32
                            ),
                            "South_Bounding_Coordinate": np.array(
                                [[36.53402]], dtype=np.float32
                            ),
                            "West_Bounding_Coordinate": np.array(
                                [[-82.65787]], dtype=np.float32
                            ),
                        }
                    },
                    "attrs": {
                        "Instrument_Short_Name": np.array([[b"VIIRS"]], dtype="|S6"),
                        "N_Collection_Short_Name": np.array(
                            [[b"VIIRS-DNB-SDR"]], dtype="|S14"
                        ),
                        "N_Dataset_Type_Tag": np.array([[b"SDR"]], dtype="|S4"),
                        "N_Instrument_Flight_SW_Version": np.array(
                            [[20], [65534]], dtype=np.int32
                        ),
                        "N_Processing_Domain": np.array([[b"ops"]], dtype="|S4"),
                        "Operational_Mode": np.array(
                            [[b"J01 Normal Operations, VIIRS Operational"]],
                            dtype="|S41",
                        ),
                    },
                },
            },
            "attrs": {
                "CVIIRS_Version": np.array([[b"2.0.1"]], dtype="|S5"),
                "Compact_VIIRS_SDR_Version": np.array([[b"3.1"]], dtype="|S3"),
                "Distributor": np.array([[b"cspp"]], dtype="|S5"),
                "Mission_Name": np.array([[b"JPSS-1"]], dtype="|S7"),
                "N_Dataset_Source": np.array([[b"all-"]], dtype="|S5"),
                "N_GEO_Ref": np.array(
                    [
                        [
                            b"GDNBO_j01_d20191025_t0611251_e0612478_b10015_c20191025062405837630_cspp_dev.h5"
                        ]
                    ],
                    dtype="|S78",
                ),
                "N_HDF_Creation_Date": np.array([[b"20191025"]], dtype="|S8"),
                "N_HDF_Creation_Time": np.array([[b"062502.927000Z"]], dtype="|S14"),
                "Platform_Short_Name": np.array([[b"J01"]], dtype="|S4"),
                "Satellite_Id_Filename": np.array([[b"j01"]], dtype="|S3"),
            },
        }
        self.filename = os.path.join(
            tempfile.gettempdir(),
            "SVDNBC_j01_d20191025_t0611251_e0612478_b10015_c20191025062459000870_eum_ops.h5",
        )
        h5f = h5py.File(self.filename, mode="w")

        def fill_h5(root, stuff):
            for key, val in stuff.items():
                if key in ["value", "attrs"]:
                    continue
                if "value" in val:
                    root[key] = val["value"]
                else:
                    grp = root.create_group(key)
                    fill_h5(grp, stuff[key])
                if "attrs" in val:
                    for attrs, val in val["attrs"].items():
                        root[key].attrs[attrs] = val

        fill_h5(h5f, fake_dnb)
        for attr, val in fake_dnb["attrs"].items():
            h5f.attrs[attr] = val
        h5f.close()

    def test_get_dataset(self):
        """Retrieve datasets from a DNB file."""
        from satpy.readers.viirs_compact import VIIRSCompactFileHandler
        from satpy import DatasetID

        filename_info = {}
        filetype_info = {'file_type': 'compact_dnb'}
        dsid = DatasetID(name='DNB', calibration='radiance')
        test = VIIRSCompactFileHandler(self.filename, filename_info, filetype_info)
        ds = test.get_dataset(dsid, {})
        self.assertEqual(ds.shape, (752, 4064))
        self.assertEqual(ds.dtype, np.float32)

        dsid = DatasetID(name='longitude')
        ds = test.get_dataset(dsid, {'standard_name': 'longitude'})
        self.assertEqual(ds.shape, (752, 4064))
        self.assertEqual(ds.dtype, np.float32)
        self.assertEqual(ds.compute().shape, (752, 4064))

    def tearDown(self):
        """Destroy."""
        try:
            os.remove(self.filename)
        except OSError:
            pass
