 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@t-online.de)

   This file was initially copied from the hp3300 backend.
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RTS88xx chipset,
    such as HP3500C, 3530C, and HP ScanJet 3570C.
    Parts of this source were inspired by other backends.

		History:

		Version 0.18  21.11.04 13.alpha,
				- source sorted,
				- now only SANEI_USB_SUPPORT for a better overview(xfermodules removed)
				- read and verify the MainBoardID 
		Version 0.17p 02.11.04 12.alpha, source sorted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17p 02.11.04 12.alpha, source sourted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
		Version 0.17  09.03.04 9. alpha, HP3500 included
		Version 0.16  06.02.04 8. alpha, wait counting on LCD
		Version 0.15a 29.01.04 7. alpha, CCD switch moved to config file
		Version 0.15  11.01.04 6. alpha, a second CCD implemented
		Version 0.13a 21.11.04 4. alpha, an little fix included
		Version 0.12  22.10.03 third alpha, Backend name changed to HP_RTS88xx
		Version 0.11  30.08.03 second alpha
		Version 0.10  19.07.03 first alpha
*/

/*
    Core HP35x0c functions.
*/

#include <stdio.h>  /* fopen, fread, fwrite, fclose etc */
#include <stdarg.h> /* va_list for vfprintf */
#include <string.h> /* memcpy, memset */
#include <unistd.h> /* unlink */
#include <stdlib.h> /* malloc, free */
#include <math.h>   /* exp, pow */
#include <ctype.h>

#include "hp_rts_xfer.h"
#include "hp_rts_44x0c.h"
#include "hp_rts_35x0c.h"

static SANE_Byte sram_access_method = 0;
static SANE_Word sram_size = 0;

/****************************************************************************/
SANE_Int
Hp35x0c_queue_read_register( SANE_Int iHandle, SANE_Byte reg, SANE_Int bytes, SANE_Byte *data)
/****************************************************************************/
{
	if (bytes == 1)
		return ( Hp_rts_RegRead(iHandle,reg, data));
	else
		return ( Hp_rts_BulkRead( iHandle, reg, data, bytes, SANE_TRUE) );
  }

/****************************************************************************/
SANE_Int
Hp35x0c_read_register_immediate( SANE_Int iHandle, SANE_Int reg, SANE_Int bytes, SANE_Byte *data)
/****************************************************************************/
{
	if (bytes == 1)
		return ( Hp_rts_RegRead(iHandle,reg, data) );
	else
		return ( Hp_rts_BulkRead( iHandle, reg, data, bytes, SANE_TRUE) );
}

/****************************************************************************
SANE_Int
Hp35x0c_queue_set_register( SANE_Int iHandle, SANE_Int reg, SANE_Int bytes, void *data)
****************************************************************************
{
	if (bytes == 1)
		return( Hp_rts_RegWrite (iHandle, reg, data));
	else
		return( Hp_rts_BulkWrite(iHandle, reg, data, bytes, SANE_TRUE));
}*/

/****************************************************************************/
SANE_Int
Hp35x0c_set_register_immediate( SANE_Int iHandle, SANE_Int reg, SANE_Int bytes, SANE_Byte *data)
/****************************************************************************/
{
	if (reg < 0xb3 && reg + bytes > 0xb3)
	{
		SANE_Int	bytes_in_first_block = 0xb3 - reg;

		if (Hp35x0c_set_register_immediate(iHandle, reg, bytes_in_first_block, data) < 0 ||
		    Hp35x0c_set_register_immediate(iHandle, 0xb4, bytes - bytes_in_first_block - 1,
							data + bytes_in_first_block + 1) < 0)
			return -1;
		return 0;
	}
	if (bytes == 1){
		return( Hp_rts_RegWrite (iHandle, reg, *data));
	}else
		return( Hp_rts_BulkWrite(iHandle, reg, (SANE_Byte *)data, bytes, SANE_TRUE));
}


/****************************************************************************/
SANE_Int
Hp35x0c_send_command_immediate( SANE_Int iHandle, SANE_Int command,
/****************************************************************************/
				SANE_Byte	reg,
				SANE_Int	count,
				SANE_Int	bytes,
				SANE_Byte	*data,
				SANE_Int	readbytes,
				SANE_Byte	*readdata)
{
	callibration_buffer[0] = command;
	callibration_buffer[1] = reg;
	callibration_buffer[2] = count >> 8;
	callibration_buffer[3] = count;
	memcpy(callibration_buffer + 4, data, bytes);
	Hp_rts_BulkWrite(iHandle, 0,callibration_buffer, bytes+4, SANE_FALSE);
	if (readbytes)
	{
		return(Hp_rts_BulkRead(iHandle, 0,readdata,readbytes,SANE_FALSE));
	}
	return( 0 );
}


/****************************************************************************/
SANE_Int
Hp35x0c_write_sram( SANE_Int iHandle, SANE_Word	bytes, SANE_Byte	*data)
/****************************************************************************/
{
	callibration_buffer[0] = RTCMD_WRITESRAM;
	callibration_buffer[1] = 0x00;
	callibration_buffer[2] = bytes >> 8;
	callibration_buffer[3] = bytes;
	memcpy(callibration_buffer + 4, data, bytes);
	return( Hp_rts_BulkWrite(iHandle, 0,
		callibration_buffer, bytes+4, SANE_FALSE));
}

/****************************************************************************/
SANE_Int
Hp35x0c_read_sram( SANE_Int iHandle, SANE_Int	bytes, SANE_Byte *data)
/****************************************************************************/
{
	return Hp35x0c_send_command_immediate(iHandle,RTCMD_READSRAM, 0, bytes, 0, 0, bytes, data);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_sram_page( SANE_Int iHandle, SANE_Byte	page)
/****************************************************************************/
{
	return(Hp_rts_Set_double_reg(iHandle,0x91,  page, page >> 8));
}

/****************************************************************************/
SANE_Int
Hp35x0c_detect_sram( SANE_Int iHandle, SANE_Word *totalbytes, SANE_Byte *r93setting)
/****************************************************************************/
{
	SANE_Byte	data[0x818];
	SANE_Byte	testbuf[0x818];
	SANE_Int	i;
	SANE_Int	test_values[] = { 6, 2, 1, -1 };

	for (i = 0; i < (SANE_Int)sizeof(data); ++i)
		data[i] = i % 0x61;


	for (i = 0; test_values[i] != -1; ++i)
	{
		if (Hp_rts_RegWrite(iHandle,0x93, test_values[i]) ||
			Hp35x0c_set_sram_page(iHandle,0x81) ||
			Hp35x0c_write_sram(iHandle,0x818, data) ||
			Hp35x0c_set_sram_page(iHandle,0x81) ||
			Hp_rts_Read_Sram (iHandle, testbuf,0x818) )
			return -1;
		if (!memcmp(testbuf, data, 0x818))
		{
			sram_access_method = test_values[i];
			if (r93setting)
				*r93setting = sram_access_method;
			break;
		}
	}
	if (!sram_access_method)
		return -1;

	for (i = 0; i < 16; ++i)
	{
		SANE_Int	j;
		SANE_Byte	write_data[32];
		SANE_Byte	read_data[32];
		SANE_Int	pagesetting;

		for (j = 0; j < 16; j++)
		{
			write_data[j * 2] = j * 2;
			write_data[j * 2 + 1] = i;
		}

		pagesetting = i * 4096;

		if (Hp35x0c_set_sram_page(iHandle,pagesetting) < 0||
		    Hp35x0c_write_sram(iHandle,32, write_data) < 0)
			return -1;
		if (i)
		{
			if (Hp35x0c_set_sram_page(iHandle,0) < 0 ||
			    Hp35x0c_read_sram(iHandle,32, read_data) < 0)
				return -1;
			if (!memcmp(read_data, write_data, 32))
			{
				sram_size = i * 0x20000;
				if (totalbytes)
					*totalbytes = sram_size;
				return 0;
			}
		}
		return -1;
	}
	return -1;
}

/****************************************************************************/
SANE_Int
Hp35x0c_is_rewound( SANE_Int iHandle )
/****************************************************************************/
{
	SANE_Byte	r;

	if (Hp35x0c_read_register_immediate( iHandle, 0x1d, 1, &r) < 0)
		return -1;
	if (r & 0x02)
		return 1;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_direction_forwards(SANE_Byte *regs)
/****************************************************************************/
{
	regs[0xc6] |= 0x08;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_direction_rewind(SANE_Byte *regs)
/****************************************************************************/
{
	regs[0xc6] &= 0xf7;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_stop_when_rewound( SANE_Byte *regs, SANE_Int	stop)
/****************************************************************************/
{
	if (stop)
		regs[0xb2] |= 0x10;
	else
		regs[0xb2] &= 0xef;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_distances( SANE_Int iHandle,SANE_Int location1,
		   SANE_Int location2)
/****************************************************************************/
{
	SANE_Byte	regbuffer[4];

	regbuffer[0] = location1;
	regbuffer[1] = location1 >> 8;
	regbuffer[2] = location2;
	regbuffer[3] = location2 >> 8;
	return Hp35x0c_set_register_immediate(iHandle,REG_DESTINATION_POSITION, 4, regbuffer);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_powersave_mode( SANE_Int iHandle,SANE_Int enable)
/****************************************************************************/
{
	SANE_Byte r;

	if (Hp35x0c_read_register_immediate(iHandle,REG_MOVE_CONTROL_TEST, 1, &r) < 0)
		return -1;
	if (r & 0x04)
	{
		if (enable == 1)
			return 0;
		r &= ~0x04;
	}
	else
	{
		if (enable == 0)
			return 0;
		r |= 0x04;
	}
	if (Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, r) < 0 ||
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, r) < 0)
		return -1;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_lamp_ready( SANE_Int iHandle )
/****************************************************************************/
{
	SANE_Byte r;

	if (Hp35x0c_read_register_immediate(iHandle,0xd1, 1, &r) < 0)
		return -1;
	if (r & 0x40)
		return 0;
	return 1;
}

/****************************************************************************/
SANE_Int
Hp35x0c_turn_off_lamp( SANE_Int iHandle )
/****************************************************************************/
{
	return Hp_rts_RegWrite(iHandle,0x3a, 0);
}

/****************************************************************************/
SANE_Int
Hp35x0c_turn_on_lamp( SANE_Int iHandle )
/****************************************************************************/
{
	SANE_Byte r3a;
	SANE_Byte r10;
	SANE_Byte r58;

	if (Hp35x0c_read_register_immediate(iHandle,0x3a, 1, &r3a) < 0 ||
		Hp35x0c_read_register_immediate(iHandle,0x10, 1, &r10) < 0 ||
		Hp35x0c_read_register_immediate(iHandle,0x58, 1, &r58) < 0)
		return -1;
	r3a |= 0x80;
	r10 |= 0x01;
	r58 &= 0x0f;
	if (Hp_rts_RegWrite(iHandle,0x3a, r3a) < 0 ||
		Hp_rts_RegWrite(iHandle,0x10, r10) < 0 ||
		Hp_rts_RegWrite(iHandle,0x58, r58) < 0)
		return -1;
	return 0;
}

/****************************************************************************/
/****************************************************************************/
#if 0
static SANE_Int
Hp35x0c_set_value_msbfirst(	SANE_Byte	*regs,
			SANE_Int		firstreg,
			SANE_Int		totalregs,
			SANE_Word	value)
/****************************************************************************/
{
	while (totalregs--)
	{
		regs[firstreg + totalregs] = value & 0xff;
		value >>= 8;
	}
	return 0;
}
#endif
/****************************************************************************/
SANE_Int
Hp35x0c_set_ccd_shift_clock_multiplier(	SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0xf0, 3, value);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_ccd_sample_clock_multiplier(	SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0xf6, 3, value);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_ccd_clock_reset_interval(	SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0xf9, 3, value);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_ccd_clamp_clock_multiplier(	SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0xfc, 3, value);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_movement_pattern(		SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0xc0, 3, value);
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_motor_movement_clock_multiplier(	SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	regs[0x40] = (regs[0x40] & ~0xc0) | (value << 6);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_motor_type(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	regs[0xc9] = (regs[0xc9] & 0xf8) | (value & 0x7);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_basic_calibration(		SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	redoffset1,
					SANE_Int	redoffset2,
					SANE_Int	redgain,
					SANE_Int	greenoffset1,
					SANE_Int	greenoffset2,
					SANE_Int	greengain,
					SANE_Int	blueoffset1,
					SANE_Int	blueoffset2,
					SANE_Int	bluegain)
{
	regs[0x05] = redoffset1;
	regs[0x02] = redoffset2;
	regs[0x08] = redgain;
	regs[0x06] = greenoffset1;
	regs[0x03] = greenoffset2;
	regs[0x09] = greengain;
	regs[0x07] = blueoffset1;
	regs[0x04] = blueoffset2;
	regs[0x0a] = bluegain;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_calibration_addresses(		SANE_Byte *regs,
/****************************************************************************/
					SANE_Word	redaddr,
					SANE_Word	blueaddr,
					SANE_Word	greenaddr)
{
	regs[0x84] = redaddr;
	regs[0x8e] = (regs[0x8e] * 0x0f) | ((redaddr >> 4) & 0xf0);
	Hp_rts_set_value_lsbfirst(regs, 0x85, 2, blueaddr);
	Hp_rts_set_value_lsbfirst(regs, 0x87, 2, greenaddr);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_lamp_duty_cycle(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	enable,
					SANE_Int	frequency,
					SANE_Int	offduty)
{
	if (enable)
		regs[0x3b] |= 0x80;
	else
		regs[0x3b] &= 0x7f;

	regs[0x3b] = (regs[0x3b] & 0x80) | ((frequency & 0x7) << 4) | (offduty & 0x0f);
	regs[0x3d] = (regs[0x3d] & 0x7f) | ((frequency & 0x8) << 4);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_data_feed_on(			SANE_Byte *regs)
/****************************************************************************/
{
	regs[0xb2] &= ~0x04;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_data_feed_off(			SANE_Byte *regs)
/****************************************************************************/
{
	regs[0xb2] |= 0x04;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_enable_ccd(				SANE_Byte *regs,
/****************************************************************************/
					SANE_Int enable)
{
	if (enable)
		regs[0x00] &= ~0x10;
	else
		regs[0x00] |= 0x10;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_cdss(				SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	val1,
					SANE_Int	val2)
{
	regs[0x28] = (regs[0x28] & 0xe0) | (val1 & 0x1f);
	regs[0x2a] = (regs[0x2a] & 0xe0) | (val2 & 0x1f);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_cdsc(				SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	val1,
					SANE_Int	val2)
{
	regs[0x29] = (regs[0x29] & 0xe0) | (val1 & 0x1f);
	regs[0x2b] = (regs[0x2b] & 0xe0) | (val2 & 0x1f);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_update_after_setting_cdss2(		SANE_Byte *regs)
/****************************************************************************/
{
	SANE_Int	fullcolour = (!(regs[0x2f] & 0xc0) && (regs[0x2f] & 0x04));
	SANE_Int value = regs[0x2a] & 0x1f;

	regs[0x2a] = (regs[0x2a] & 0xe0) | (value & 0x1f);

	if (fullcolour)
		value *= 3;
	if ((regs[0x40] & 0xc0) == 0x40)
		value += 17;
	else
		value += 16;

	regs[0x2c] = (regs[0x2c] & 0xe0) | (value  % 24);
	regs[0x2d] = (regs[0x2d] & 0xe0) | ((value + 2) % 24);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_cph0s(				SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	on)
{
	if (on)
		regs[0x2d] |= 0x20;	/* 1200dpi horizontal coordinate space */
	else
		regs[0x2d] &= ~0x20;	/* 600dpi horizontal coordinate space */
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_cvtr_lm(				SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	val1,
					SANE_Int	val2,
					SANE_Int	val3)
{
	regs[0x28] = (regs[0x28] & ~0xe0) | (val1 << 5);
	regs[0x29] = (regs[0x29] & ~0xe0) | (val2 << 5);
	regs[0x2a] = (regs[0x2a] & ~0xe0) | (val3 << 5);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_cvtr_mpt(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	val1,
					SANE_Int	val2,
					SANE_Int	val3)
{
	regs[0x3c] = (val1 & 0x0f) | (val2 << 4);
	regs[0x3d] = (regs[0x3d] & 0xf0) | (val3 & 0x0f);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_cvtr_wparams(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Word fpw,
					SANE_Word bpw,
					SANE_Word w)
{
	regs[0x31] = (w & 0x0f) | ((bpw << 4) & 0x30) | (fpw << 6);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_enable_movement(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	enable)
{
	if (enable)
		regs[0xc3] |= 0x80;
	else
		regs[0xc3] &= ~0x80;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_scan_frequency(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	frequency)
{
	regs[0x64] = (regs[0x64] & 0xf0) | (frequency & 0x0f);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_merge_channels(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	on)
{
	DBG( DBG_MSG , "start_scan: Hp35x0c_set_merge_channels\n");
	regs[0x2f] &= ~0x14;
	regs[0x2f] |= on ? 0x04 : 0x10;
#ifdef DEBUG
	Hp_rts_DumpBits(0x2f, regs[0x2f]);
#endif
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_channel(				SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	channel)
{
	DBG( DBG_MSG , "start_scan: Hp35x0c_set_channel %d\n",channel);
	regs[0x2f] = (regs[0x2f] & ~0xc0) | (channel << 6);
#ifdef DEBUG
	Hp_rts_DumpBits(0x2f, regs[0x2f]);
#endif
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_single_channel_scanning(		SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	on)
{
	if (on){
		regs[0x2f] |= 0x20;
		DBG( DBG_MSG , "start_scan: Hp35x0c_set_single_channel_scanning on\n");
	}else{
		regs[0x2f] &= ~0x20;
		DBG( DBG_MSG , "start_scan: Hp35x0c_set_single_channel_scanning off\n");
	}
#ifdef DEBUG
	Hp_rts_DumpBits(0x2f, regs[0x2f]);
#endif
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_colour_mode(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	on)
{
	if (on){
		DBG( DBG_MSG , "start_scan: Set register to color mode\n");
		regs[0x2f] |= 0x02;
	}else{
		DBG( DBG_MSG , "start_scan: Set register to gray mode\n");
		regs[0x2f] &= ~0x02;
	}
#ifdef DEBUG
	Hp_rts_DumpBits(0x2f, regs[0x2f]);
#endif
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_horizontal_resolution(		SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	resolution)
{
	if (regs[0x2d] & 0x20)
		regs[0x7a] = 1200 / resolution;
	else
		regs[0x7a] = 600 / resolution;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_last_sram_page(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	pagenum)
{
	Hp_rts_set_value_lsbfirst(regs, 0x8b, 2, pagenum);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_step_size(			SANE_Byte *regs,
/****************************************************************************/
					SANE_Int	stepsize)
{
	Hp_rts_set_value_lsbfirst(regs, 0xe2, 2, stepsize);
	Hp_rts_set_value_lsbfirst(regs, 0xe0, 2, 0);
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_set_all_registers( SANE_Int iHandle, void const *regs_)
/****************************************************************************/
{
	SANE_Byte	regs[255];

	memcpy(regs, regs_, 255);
	regs[32] &= ~0x40;

	if (Hp_rts_RegWrite(iHandle,0x32, regs[0x32]) < 0 ||
	    Hp35x0c_set_register_immediate(iHandle,0, 255, regs) < 0 ||
	    Hp_rts_RegWrite(iHandle,0x32, regs[0x32] | 0x40) < 0)
		return -1;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_adjust_misc_registers(SANE_Byte *regs)
/****************************************************************************/
{
	/* Mostly unknown purposes - probably no need to adjust */
	regs[0xc6] = (regs[0xc6] & 0x0f) | 0x20; /* Purpose unknown - appears to do nothing */
	regs[0x2e] = 0x86;			/* ???? - Always has this value */
	regs[0x30] = 2;				/* CCPL = 1 */
	regs[0xc9] |= 0x38;			/* Doesn't have any obvious effect, but the Windows driver does this */
	return 0;
}


#define NVR_MAX_ADDRESS_SIZE	11
#define NVR_MAX_OPCODE_SIZE		3
#define NVR_DATA_SIZE			8
#define	NVR_MAX_COMMAND_SIZE	((NVR_MAX_ADDRESS_SIZE + \
				  NVR_MAX_OPCODE_SIZE + \
				  NVR_DATA_SIZE) * 2 + 1)

/****************************************************************************/
static SANE_Int
Hp35x0c_nvram_enable_controller(SANE_Int iHandle,SANE_Int enable)
/****************************************************************************/
{
	SANE_Byte r;

	if (Hp35x0c_read_register_immediate(iHandle,0x1d, 1, &r) < 0)
		return -1;
	if (enable)
		r |= 1;
	else
		r &= ~1;
	return Hp_rts_RegWrite(iHandle,0x1d, r);

}

/****************************************************************************/
static SANE_Int
Hp35x0c_nvram_init_command(SANE_Int iHandle)
/****************************************************************************/
{
	SANE_Byte regs[13];

	if (Hp35x0c_read_register_immediate(iHandle,0x10, 13, regs) < 0)
		return -1;
	regs[2] |= 0xf0;
	regs[4] = regs[4] & 0x1f | 0x60;
	return Hp35x0c_set_register_immediate(iHandle,0x10, 13, regs);
}

/****************************************************************************/
static SANE_Int
Hp35x0c_nvram_init_stdvars(SANE_Int iHandle,SANE_Int block,
/****************************************************************************/
			SANE_Int *addrbits,
			SANE_Byte *basereg)
{
	SANE_Int	bitsneeded;
	SANE_Int	capacity;

	switch (block)
	{
	case 0:
		bitsneeded = 7;
		break;

	case 1:
		bitsneeded = 9;
		break;

	case 2:
		bitsneeded = 11;
		break;

	default:
		bitsneeded = 0;
		capacity = 1;
		while (capacity < block)
			capacity <<= 1, ++bitsneeded;
		break;
	}

	*addrbits = bitsneeded;

	if (Hp35x0c_read_register_immediate(iHandle,0x10, 1, basereg) < 0)
		return -1;

	*basereg &= ~0x60;
	return 0;
}

/****************************************************************************/
static void
Hp35x0c_nvram_set_half_bit(	SANE_Byte *buffer,
/****************************************************************************/
			SANE_Int	value,
			SANE_Byte stdbits,
			SANE_Int	whichhalf)
{
	*buffer = stdbits | (value ? 0x40 : 0) | (whichhalf ? 0x20 : 0);
}

/****************************************************************************/
static void
Hp35x0c_nvram_set_command_bit(SANE_Byte *buffer,
/****************************************************************************/
			 SANE_Int value,
			 SANE_Byte stdbits)
{
	Hp35x0c_nvram_set_half_bit(buffer, value, stdbits, 0);
	Hp35x0c_nvram_set_half_bit(buffer + 1, value, stdbits, 1);
}

/****************************************************************************/
void
Hp35x0c_nvram_set_addressing_bits(	SANE_Byte *buffer,
/****************************************************************************/
				SANE_Int	location,
				SANE_Int	addressingbits,
				SANE_Byte stdbits)
{
	SANE_Int	currentbit = 1 << (addressingbits - 1);

	while (addressingbits--)
	{
		Hp35x0c_nvram_set_command_bit(buffer,
					 (location & currentbit) ? 1 : 0,
					 stdbits);
		buffer += 2;
		currentbit >>= 1;
	}
}

/****************************************************************************/
SANE_Int
Hp35x0c_nvram_enable_write(SANE_Int iHandle,SANE_Int	addressingbits,
/****************************************************************************/
			SANE_Int	enable,
			SANE_Byte stdbits)
{
	SANE_Byte cmdbuffer[NVR_MAX_COMMAND_SIZE];
	SANE_Byte cmdsize = 6 + addressingbits * 2;

	Hp35x0c_nvram_set_command_bit(cmdbuffer, 1, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 2, 0, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 4, 0, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 6, enable, stdbits);
	if (addressingbits > 1)
		Hp35x0c_nvram_set_addressing_bits(cmdbuffer + 8, 0, addressingbits - 1, stdbits);

	if (Hp35x0c_nvram_enable_controller(iHandle,1) < 0 ||
	    Hp35x0c_send_command_immediate(iHandle,RTCMD_NVRAMCONTROL, 0, cmdsize, cmdsize, cmdbuffer, 0, 0) < 0 ||
	    Hp35x0c_nvram_enable_controller(iHandle,0) < 0)
	{
		return -1;
	}
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_nvram_write(SANE_Int iHandle, SANE_Int	block,
/****************************************************************************/
		SANE_Int	location,
		SANE_Char const *data,
		SANE_Int	bytes)
{
	SANE_Int addressingbits;
	SANE_Byte stdbits;
	SANE_Byte cmdbuffer[NVR_MAX_COMMAND_SIZE];
	SANE_Byte *address_bits;
	SANE_Byte *data_bits;
	SANE_Int	i;
	SANE_Int	cmdsize;
	SANE_Byte r;
	SANE_Byte cmd;

	/* This routine doesn't appear to work, but I can't see anything wrong with it */
	if (Hp35x0c_nvram_init_stdvars(iHandle,block, &addressingbits, &stdbits) < 0)
		return -1;

	cmdsize = (addressingbits + 8) * 2 + 6;
	address_bits = cmdbuffer + 6;
	data_bits = address_bits + (addressingbits * 2);

	Hp35x0c_nvram_set_command_bit(cmdbuffer, 1, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 2, 0, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 4, 1, stdbits);

	if (Hp35x0c_nvram_init_command(iHandle) < 0 ||
	    Hp35x0c_nvram_enable_write(iHandle,addressingbits, 1, stdbits) < 0)
		return -1;

	while (bytes--)
	{
		Hp35x0c_nvram_set_addressing_bits(address_bits, location, addressingbits, stdbits);
		Hp35x0c_nvram_set_addressing_bits(data_bits, *data++, 8, stdbits);

		if (Hp35x0c_nvram_enable_controller(iHandle,1) < 0 ||
		    Hp35x0c_send_command_immediate(iHandle,RTCMD_NVRAMCONTROL, 0, cmdsize, cmdsize, cmdbuffer, 0, 0) < 0 ||
		    Hp35x0c_nvram_enable_controller(iHandle,0) < 0)
			return -1;

		if (Hp35x0c_nvram_enable_controller(iHandle,1) < 0)
			return -1;
		for (i = 0; i < cmdsize; ++i)
		{
			Hp35x0c_nvram_set_half_bit(&cmd, 0, stdbits, i & 1);
			if (Hp35x0c_send_command_immediate(iHandle,RTCMD_NVRAMCONTROL, 0, 1, 1, &cmd, 0, 0) < 0 ||
			    Hp35x0c_read_register_immediate(iHandle,0x10, 1, &r) < 0)
			{
				return -1;
			}
			else if (r & 0x80)
			{
				break;
			}
		}
		if (Hp35x0c_nvram_enable_controller(iHandle,0) < 0)
			return -1;

		++location;
	}

	if (Hp35x0c_nvram_enable_write(iHandle,addressingbits, 0, stdbits) < 0)
		return -1;
	return 0;
}

/****************************************************************************/
SANE_Int
Hp35x0c_nvram_read(SANE_Int iHandle,SANE_Int	block,
/****************************************************************************/
		SANE_Int	location,
		SANE_Byte	*data,
		SANE_Int	bytes)
{
	SANE_Int addressingbits;
	SANE_Byte stdbits;
	SANE_Byte cmdbuffer[NVR_MAX_COMMAND_SIZE];
	SANE_Byte *address_bits;
	SANE_Byte readbit_command[2];
	SANE_Int	cmdsize;
	SANE_Byte r;
	SANE_Int	i;
	SANE_Char	c = 0;

	if (Hp35x0c_nvram_init_stdvars(iHandle,block, &addressingbits, &stdbits) < 0)
		return -1;

	cmdsize = addressingbits * 2 + 7;
	address_bits = cmdbuffer + 6;

	Hp35x0c_nvram_set_command_bit(cmdbuffer, 1, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 2, 1, stdbits);
	Hp35x0c_nvram_set_command_bit(cmdbuffer + 4, 0, stdbits);
	Hp35x0c_nvram_set_half_bit(cmdbuffer + cmdsize - 1, 0, stdbits, 0);

	Hp35x0c_nvram_set_half_bit(readbit_command, 0, stdbits, 1);
	Hp35x0c_nvram_set_half_bit(readbit_command + 1, 0, stdbits, 0);

	if (Hp35x0c_nvram_init_command(iHandle) < 0)
		return -1;

	while (bytes--)
	{
		Hp35x0c_nvram_set_addressing_bits(address_bits, location, addressingbits, stdbits);

		if (Hp35x0c_nvram_enable_controller(iHandle,1) < 0 ||
		    Hp35x0c_send_command_immediate(iHandle,RTCMD_NVRAMCONTROL, 0x1d, cmdsize, cmdsize, cmdbuffer, 0, 0) < 0)
			return -1;

		for (i = 0; i < 8; ++i)
		{
			c <<= 1;

			if (Hp35x0c_send_command_immediate(iHandle,RTCMD_NVRAMCONTROL, 0x1d, 2, 2, readbit_command, 0, 0) < 0 ||
			    Hp35x0c_read_register_immediate(iHandle,0x10, 1, &r) < 0)
				return -1;
			if (r & 0x80)
				c |= 1;
		}
		if (Hp35x0c_nvram_enable_controller(iHandle,0) < 0)
			return -1;

		*data++ = c;
		++location;
	}
	return 0;
}


/****************************************************************************/
SANE_Int
Hp35x0c_rewind(THWParams *pHWParams)
/* Moves scanner to home position */
/****************************************************************************/
{
	SANE_Byte regs[255];
	SANE_Int iHandle;

	regs[1] = 0; /* only for the compiler */
	iHandle = pHWParams->iXferHandle;
	DBG(DBG_MSG,"Hp35x0c_rewind : park to home...");
#ifdef DEBUG_HP3500
	return 0;
#endif
  if (Hp35x0c_is_rewound(iHandle )){
		DBG(DBG_MSG,"Hp35x0c_rewind : i'm home\n");
		return 0;
	}
	DBG(DBG_MSG,"Hp35x0c_rewind : i must moving\n");
#if 0
	/* read all registers */
	Hp35x0c_read_register_immediate(iHandle,0, 255, regs);

	/* set registes 60/1 and 62/3 */
	Hp_rts_set_noscan_distance(regs, 59998);
	Hp_rts_set_total_distance(regs, 59999);

	/* clear Bit 4 from register b2 */
	Hp35x0c_set_stop_when_rewound(regs, 0);

	DBG(DBG_MSG,"Hp35x0c_rewind : Hp_rts_RegWrite 0xc6\n");
	Hp_rts_RegWrite(iHandle,0xc6, 0);
	DBG(DBG_MSG,"Hp35x0c_rewind : Hp_rts_RegWrite 0xc6\n");
	Hp_rts_RegWrite(iHandle,0xc6, 0);

	/* set register e2 to stepsize, e0 to 0 */
	Hp35x0c_set_step_size(regs, 0x0abd);

	/* set Bit 3 from register c6 to 0 */
	Hp35x0c_set_direction_rewind(regs);

	/* fast move */
	regs[0x39] = 15;
	/* Motor enable: It set bit 7. clear bit 0-2 and set bit 0
		regs[0xc3] = (regs[0xc3] & 0xf8) | 1; */
	regs[0xc3] |= 0x80;

	/* setzt in Register c6 step size auf 3 */
	regs[0xc6] = (regs[0xc6] & 0xf8) | 3;

	DBG(DBG_MSG,"Hp35x0c_rewind : Hp35x0c_set_all_registers\n");
	Hp35x0c_set_all_registers(iHandle,regs);
#else
	memcpy(pHWParams->regs,Hp35x0c_rewind_regs,255);
	/*-- motor resolution divisor*/
	pHWParams->regs[0x39] =0x0f;

	/*-- motor movement clock multiplier*/
	pHWParams->regs[0x40] =0xa0;

	/*-- noscan distance*/
	pHWParams->regs[0x60] =0x70;
	pHWParams->regs[0x61] =0xe7;

	/*-- total distance =noscan distance+1*/
	pHWParams->regs[0x62] =0x71;
	pHWParams->regs[0x63] =0xe7;

	/*-- no image data, stop when home*/
	/*   (careful, value becomes 0x12 after write)*/
	pHWParams->regs[0xb2] =0x14;

	/*-- coordinate space denominator=3D1, motor enable*/
	pHWParams->regs[0xc3] =0x81;

	/*-- backward direction, step size 0.5 (0x01) or 1 (0x03)*/
	/*    pHWParams->regs[0xc6] =0x21;*/
	pHWParams->regs[0xc6] =0x23;

	/*-- bounds movement range 1*/
	pHWParams->regs[0xe0] =0xf4;
	pHWParams->regs[0xe1] =0x07;

	/*-- Step size movement range 1*/
	pHWParams->regs[0xe2] =0xd0;
	pHWParams->regs[0xe3] =0x00;

	Hp35x0c_set_all_registers(iHandle,pHWParams->regs);

#endif
	Hp_rts_start_moving(iHandle);
	DBG(DBG_MSG,"Hp35x0c_rewind : Test, if the scanner is moving\n");
	while (!Hp35x0c_is_rewound(iHandle) /*&&
		(Hp_rts_data_ready(iHandle,&n) ||
		 Hp_rts_is_moving(iHandle) > 0)*/)
	{
		/*if (n)
		{
			SANE_Byte	buffer[0xffc0];

			if (n > (SANE_Int)sizeof(buffer))
				n = (SANE_Int)sizeof(buffer);
			Hp_rts_read_data(iHandle,n, buffer);
			DBG(DBG_MSG,"====> has read data\n");
		}
		else
		{*/
			DBG(DBG_MSG,"w");
			usleep(10000);
#if 0 /* short fix, because the scanner doesn't move  */
			break;
#endif
		/*}*/
	}
	DBG(DBG_MSG," moving done\n");
	return(Hp_rts_stop_moving(iHandle));
}


/****************************************************************************/
SANE_Bool Hp35x0c_init_scan(THWParams *pHWParams, TScanParams *pParams,
                    TDataPipe *pDataPipe)
/* code comes from rts8801_scan */
/****************************************************************************/
{
	SANE_Int	iHandle;
	SANE_Byte	offdutytime;
	SANE_Int	ires;
	SANE_Byte r93setting;
	SANE_Int	tg_setting;
	SANE_Word x1,x2,y,h;

	iHandle = pHWParams->iXferHandle;
	r93setting = r93setting;

	for (ires = 0; Hp35x0c_resparms[ires].resolution &&
								 Hp35x0c_resparms[ires].resolution != pParams->iDpi; ++ires);
	if (Hp35x0c_resparms[ires].resolution == 0){
		DBG(DBG_MSG,"Hp35x0c_init_scan: did not found this resolution %d\n",pParams->iDpi);
		return SANE_FALSE;
	}

	DBG(DBG_MSG,"Hp35x0c_init_scan: Use ires entry %d (%ddpi) for the resolution %d dpi\n",ires,
			Hp35x0c_resparms[ires].resolution, pParams->iDpi);

#ifdef ENABLE_VI8920
	DBG(DBG_MSG,"Vi8920 is enabled and will be run us a HP3500c\n");
#endif

	/* Initialise and power up */
#ifdef DEBUG_HP3500
	Hp35x0c_set_all_registers(iHandle,Hp44x0_switch_on_regs);
#else
	Hp35x0c_set_all_registers(iHandle,Hp35x0c_initial_regs);
	Hp35x0c_set_powersave_mode(iHandle,0);

	Hp35x0c_rewind(pHWParams);
  /* Warm up the lamp */
	Hp35x0c_detect_sram(iHandle,&sram_size, &r93setting);
	Hp35x0c_turn_on_lamp(iHandle);
	sleep(20);
#endif

  /* Set scan parameters */
	Hp35x0c_read_register_immediate(iHandle,0, 255, pHWParams->regs);
	pHWParams->regs[255] = 0;

	Hp35x0c_enable_ccd(pHWParams->regs, 1);
	Hp35x0c_enable_movement(pHWParams->regs, 1);
	Hp35x0c_set_scan_frequency(pHWParams->regs, 1);

	Hp35x0c_adjust_misc_registers(pHWParams->regs);
	Hp35x0c_set_cvtr_wparams(pHWParams->regs, 3, 0, 6);
	Hp35x0c_set_cvtr_mpt(pHWParams->regs, 15, 15, 15);
	Hp35x0c_set_cvtr_lm(pHWParams->regs, 7, 7, 7);
	Hp35x0c_set_motor_type(pHWParams->regs, 2);

 	offdutytime = 0;
	if (Hp35x0c_nvram_read(iHandle,0, 0x7b, &offdutytime, 1) < 0 ||
      offdutytime >= 15)
	{
  	offdutytime = 6;
	}
	Hp35x0c_set_lamp_duty_cycle(pHWParams->regs,
			1,		/* On */
			10,		/* Frequency */
			offdutytime);	/* Off duty time */

	Hp35x0c_set_movement_pattern(pHWParams->regs, 0x800000);

	tg_setting = Hp35x0c_resparms[ires].tg;
	Hp35x0c_set_ccd_shift_clock_multiplier(pHWParams->regs,
				tg_info[tg_setting].tg_cph0p);
	Hp35x0c_set_ccd_clock_reset_interval(pHWParams->regs,
				tg_info[tg_setting].tg_crsp);
	Hp35x0c_set_ccd_clamp_clock_multiplier(pHWParams->regs,
				tg_info[tg_setting].tg_cclpp);

	Hp_rts_RegWrite(iHandle,0xc6, 0);
	Hp_rts_RegWrite(iHandle,0xc6, 0);

	Hp35x0c_set_step_size(pHWParams->regs, Hp35x0c_resparms[ires].step_size);
	Hp35x0c_set_direction_forwards(pHWParams->regs);

	Hp35x0c_set_stop_when_rewound(pHWParams->regs, 0);
	Hp35x0c_set_data_feed_on(pHWParams->regs);

	Hp35x0c_set_calibration_addresses(pHWParams->regs, 0, 0, 0);

	Hp35x0c_set_basic_calibration(pHWParams->regs,
				0xc8, 0xc8, 0x0a,
				0xc6, 0xc6, 0x0c,
				0xc6, 0xc6, 0x0b);
	pHWParams->regs[0x0b] = 0x70;/* If set to 0x71, the alternative, all values are low*/

	switch (pParams->mode) {
		case GRAY:
			DBG(DBG_ERR, "Hp35x0c_init_scan: Use gray parameter\n");
			Hp35x0c_set_channel(pHWParams->regs, RT_CHANNEL_RED);
			Hp35x0c_set_single_channel_scanning(pHWParams->regs, 1); /* 0x20 on / 0ff*/
			Hp35x0c_set_merge_channels(pHWParams->regs, 1); /* 0x14 on / off */
			Hp35x0c_set_colour_mode(pHWParams->regs, 0); /* 0x02 on / off */
			break;
		case COLOR:
			DBG(DBG_ERR, "Hp35x0c_init_scan: use color parameter\n");
			Hp35x0c_set_channel(pHWParams->regs, RT_CHANNEL_ALL);
			Hp35x0c_set_single_channel_scanning(pHWParams->regs, 0); /* 0x20 on / 0ff*/
			Hp35x0c_set_merge_channels(pHWParams->regs, 1); /* 0x14 on / off */
			/*	Hp35x0c_set_merge_channels(pHWParams->regs, 0); * 0x14 on / off */
			Hp35x0c_set_colour_mode(pHWParams->regs, 1); /* 0x02 on / off */
			break;

		default:
			DBG(DBG_ERR, "Hp35x0c_init_scan: Invalid parameter\n");
			return SANE_FALSE;
	}/* switch */

	Hp35x0c_set_motor_movement_clock_multiplier(pHWParams->regs,
		Hp35x0c_resparms[ires].motor_movement_clock_multiplier);

	Hp35x0c_set_cdss(pHWParams->regs, tg_info[tg_setting].tg_cdss1,
		tg_info[tg_setting].tg_cdss2);
	Hp35x0c_set_cdsc(pHWParams->regs, tg_info[tg_setting].tg_cdsc1,
		tg_info[tg_setting].tg_cdsc2);

	Hp35x0c_update_after_setting_cdss2(pHWParams->regs);

	Hp35x0c_set_last_sram_page(pHWParams->regs, (sram_size - 1) >> 5);

	pHWParams->regs[0x39] = Hp35x0c_resparms[ires].reg_39_value;
	pHWParams->regs[0xc3] = (pHWParams->regs[0xc3] & 0xf8) |
		 	Hp35x0c_resparms[ires].reg_c3_value;
	pHWParams->regs[0xc6] = (pHWParams->regs[0xc6] & 0xf8) |
			 Hp35x0c_resparms[ires].reg_c6_value;
	Hp35x0c_set_scan_frequency(pHWParams->regs, Hp35x0c_resparms[ires].scan_frequency);
	Hp35x0c_set_cph0s(pHWParams->regs, Hp35x0c_resparms[ires].cph0s);

	x1 = pParams->iX;
	x2 = pParams->iWidth + pParams->iX;
	y = pParams->iY;
	h = pParams->iLenght;/* * 2;*/
	DBG(DBG_MSG,"Hp35x0c_init_scan: calculate scandata x1 = %d; x2 = %d; lenght = %d\n",x1,x2,h);
	pDataPipe->iScanned = h;
	DBG(DBG_MSG,"Hp35x0c_init_scan: Hp35x0c_set_horizontal_resolution %d.\n",pParams->iDpi);
	Hp35x0c_set_horizontal_resolution(pHWParams->regs, pParams->iDpi);

	/* modify reg 60 + 61 */
/*	Hp_rts_set_noscan_distance(pHWParams->regs, y *
			 Hp35x0c_resparms[ires].scan_frequency -  1);*/
	Hp_rts_set_noscan_distance(pHWParams->regs, 0x01 );

	/* modify reg 62 + 63 */
/*	Hp_rts_set_total_distance(pHWParams->regs, Hp35x0c_resparms[ires].scan_frequency *
				(y + h + ((pParams->mode == COLOR) ? (Hp35x0c_resparms[ires].red_green_offset +
				Hp35x0c_resparms[ires].green_blue_offset) : 0) +
				Hp35x0c_resparms[ires].intra_channel_offset));*/
	Hp_rts_set_total_distance(pHWParams->regs, h );

	/* modify reg 66 + 67 */
/*	Hp_rts_set_scanline_start(pHWParams->regs, x1 * (1200 / pParams->iDpi) / (Hp35x0c_resparms[ires].cph0s ? 1 : 2));*/
	Hp_rts_set_scanline_start(pHWParams->regs, x1 );

	/* modify reg 6C + 6D */
/*	Hp_rts_set_scanline_end(pHWParams->regs, (x1 + x2) * (1200 / pParams->iDpi) / (Hp35x0c_resparms[ires].cph0s ? 1 : 2));*/
	Hp_rts_set_scanline_end(pHWParams->regs, x2 );

#if 1
	DBG(DBG_MSG,"Hp35x0c_init_scan: Hp35x0c_set_all_registers\n");
	Hp_rts_DumpHex(pHWParams->regs,255,16,SANE_TRUE);
	Hp_rts_DumpBits(0x2f, pHWParams->regs[0x2f]);
#endif
#ifdef DEBUG_HP3500
	return SANE_FALSE;
#endif
	Hp35x0c_set_all_registers(iHandle, pHWParams->regs);

	Hp_rts_RegWrite(iHandle, 0x2c, pHWParams->regs[0x2c]);

	Hp_rts_start_moving(iHandle);
	usleep(10000);
	/* wait for moving and check it */
	return(Hp_rts_Check_Moving (iHandle));
}

/****************************************************************************/
SANE_Bool Hp35x0c_init_power_on(THWParams *pHWParams)
/****************************************************************************/
{
	SANE_Int	iHandle;

	iHandle = pHWParams->iXferHandle;
	Hp35x0c_set_all_registers(iHandle,Hp35x0c_initial_regs);
	Hp35x0c_set_powersave_mode(iHandle,0);
#ifndef DEBUG_HP3500
	return(Hp35x0c_rewind(pHWParams));
#else
	return 0;
#endif
}
