# Samizdat message handling
#
#   Copyright (c) 2002-2006  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'fileutils'
require 'samizdat/engine'

class Message

  # relative path to multimedia message content
  #
  # if _format_ or _login_ are not specified, this Resource object should be
  # instantiated from a valid _id_ of an existing resource for this method to
  # work
  #
  def Message.content_path(id, format=nil, login=nil)
    if login.nil? or format.nil?
      if id.kind_of? Integer and id > 0
        message = Message.cached(id)
        format = message.format
        login = message.creator_login
      else
        raise RuntimeError,
          _("Can't derive message format and creator login name")
      end
    end
    ext = config['file_extension'][format]
    ext = format.sub(/\A.*\//, '') if ext.nil?
    '/' + login + '/' + id.to_s + '.' + ext
  end

  def Message.format_cacheable?(format)
    format.nil? or
      (config['format']['inline'].include?(format) and format =~ /^text\//)
  end

  # update html_full and html_short for a message
  # (running inside transaction is assumed)
  #
  def Message.update_html(id, content, format)
    db.do 'UPDATE Message SET html_full = ? WHERE id = ?',
      Template.format_inline_content(content, format), id

    full_size = content.size
    content = Template.limit_string(content, config['limit']['short'])
    if content.size < full_size
      db.do 'UPDATE Message SET html_short = ? WHERE id = ?',
        Template.format_inline_content(content, format), id
    else
      db.do 'UPDATE Message SET html_short = NULL WHERE id = ?', id
    end
  end

  # re-render html_full and html_short for a single messages
  #
  # command line:
  #
  # SAMIZDAT_SITE=samizdat SAMIZDAT_URI=/ ruby -r samizdat/engine -e 'Message.regenerate_html(1)'
  #
  def Message.regenerate_html(id)
    db.transaction do |db|
      content, format = db.select_one('SELECT content, format FROM Message WHERE id = ?', id)
      return unless content and Message.format_cacheable?(format)

      Message.update_html(id, content, format)
    end   # transaction
  end

  # re-render html_full and html_short for all messages
  #
  # command line:
  #
  # SAMIZDAT_SITE=samizdat SAMIZDAT_URI=/ ruby -r samizdat/engine -e 'Message.regenerate_all_html'
  #
  def Message.regenerate_all_html
    db.select_all('SELECT id, format FROM Message WHERE content IS NOT NULL') do |id, format|
      next unless Message.format_cacheable?(format)
      Message.regenerate_html(id)
    end
  end

  def Message.cached(id, showhidden=false)
    cache.fetch_or_add(%{message/#{id}/#{showhidden}}) do
      Message.new(id, showhidden)
    end
  end

  # load message by id
  #
  def initialize(id, showhidden=false)
    @id = id
    @title = rdf.get_property(@id, 'dc::title')
    raise ResourceNotFoundError, id.to_s if @title.nil?
    @content = rdf.get_property(@id, 's::content')
    @date = rdf.get_property(@id, 'dc::date')
    @lang = rdf.get_property(@id, 'dc::language')
    @format = rdf.get_property(@id, 'dc::format')
    @hidden = rdf.get_property(@id, 's::hidden')

    @creator = rdf.get_property(@id, 'dc::creator')
    if @creator
      @creator_name, @creator_login = rdf.select_one %{
SELECT ?name, ?login
WHERE (s::fullName #{@creator} ?name)
      (s::login #{@creator} ?login)}
    else
      @creator_login = 'guest'
    end

    @desc = rdf.get_property(@id, 'dc::description')
    @parent = rdf.get_property(@id, 's::inReplyTo')
    @current = rdf.get_property(@id, 'dct::isVersionOf')
    @open = rdf.get_property(@id, 's::openForAll')
    @nversions, = rdf.select_one %{
SELECT count(?version)
WHERE (dct::isVersionOf ?version #{@id})}

    hidden = " AND ?hidden = 'false'" unless showhidden

    @translations = rdf.select_all(%{
SELECT ?lang, ?msg, ?rating
WHERE (rdf::predicate ?stmt dc::relation)
      (rdf::subject ?stmt ?msg)
      (rdf::object ?stmt focus::Translation)
      (s::inReplyTo ?msg #{@id})
      (dc::language ?msg ?lang)
      (s::rating ?stmt ?rating)
      (s::hidden ?msg ?hidden)
LITERAL ?rating > 0 #{hidden}
ORDER BY ?rating DESC}, limit_page).collect {|m, l, r| [m, l, r] }   # DBI::Row
    @nreplies, = rdf.select_one %{
SELECT count(?msg)
WHERE (s::inReplyTo ?msg #{@id})
      (s::hidden ?msg ?hidden)
LITERAL ?msg IS NOT NULL #{hidden}}
    @nreplies -= @translations.size   # don't count translations
    @focuses = rdf.select_all(%{
SELECT ?focus
WHERE (rdf::subject ?stmt #{@id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt ?focus)
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0
ORDER BY ?rating DESC}, limit_page).collect {|f,| f }
    @nrelated, = rdf.select_one(%{
SELECT count(?related)
WHERE (rdf::subject ?stmt ?related)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id})
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0})

    @html_full = rdf.get_property(@id, 's::htmlFull')
    @html_short = rdf.get_property(@id, 's::htmlShort')
  end

  attr_reader :id, :title, :content, :date, :lang, :format, :hidden
  attr_reader :creator, :creator_name, :creator_login
  attr_reader :desc, :parent, :current, :open, :nversions
  attr_reader :translations, :nreplies, :focuses, :nrelated
  attr_reader :html_full, :html_short

  def inline?
    @format.nil? or config['format']['inline'].include? @format
  end

  # find available translation to the most preferred language
  #
  # returns Message object or self, when no translation is suitable
  #
  def select_translation(accept_language)
    return self unless @lang   # don't translate messages with no language
    t_id = nil   # translation id
    accept_language.each do |l|
      break if l == @lang   # message already in preferred language
      t_id = @translations.assoc(l)
      break unless t_id.nil?
    end
    t_id ? Message.cached(t_id[1]) : self
  end

  # add RSS item of the message to _maker_ feed
  #
  # _maker_ is assumed to provide RSS::Maker API
  #
  def rss(maker, session)
    item = maker.items.new_item
    item.link = session.base + @id.to_s
    item.date = @date.to_time
    t = select_translation(session.accept_language)
    item.title = t.title
    item.dc_language = t.lang
  end
end

class PublishMessage < Message

  # extract from session and check all parameters
  #
  def initialize(session)
    @session = session

    @id = 'upload'   # hack to deceive Template#message_content
    @date = Time.now
    @creator = session.id
    @creator_name = session.full_name
    @creator_login = session.login
    @nversions = 0
    @translations = []
    @nreplies = 0
    @focuses = []
    @nrelated = 0

    @title, @content, @focus, @rating, @lang, @format, @desc, @open, @parent, @new_parent, @action =
      session.params %w[title content focus rating lang format desc open parent new_parent action]
    @file = @session['file']   # just a file object, not its contents
    @file = nil if @file and 0 == @file.size

    check_new_message
  end

  # at least title and content (or file) are required
  #
  def enough_fields?
    case @action
    when :source
      false   # just viewing the source
    when :hide, :unhide
      @session.has_key?('confirm')
    when :reparent
      @parent and @session.has_key?('confirm')
    else
      @title and (@content or @file)
    end
  end

  # write message into the database, return its location
  #
  def publish
    id = @parent.id if @parent   # start with id of edited message
    db.transaction do |db|
      @content.gsub!(/\r\n/, "\n") if @content
      # todo: detect duplicates
      # todo: translate into RDF assert
      if [:hide, :unhide, :takeover, :displace, :reparent].include? @action
        log_moderation(@creator, @action.to_s, @parent.id)
      end

      if :edit == @action or :takeover == @action
        @parent or raise UserError, _('Reference to previous version lost')
        # save current version at new id
        db.do 'INSERT INTO Message (version_of, creator, title, language,
          format, description, content, open, html_full, html_short) SELECT id,
          creator, title, language, format, description, content, open,
          html_full, html_short FROM Message WHERE id = ?', @parent.id
        new_id, = db.select_one "SELECT MAX(id) FROM Resource"
        db.do "UPDATE Resource SET published_date =
          (SELECT published_date FROM Resource WHERE id = ?)
          WHERE id = ?", @parent.id, new_id
        # write new version at old id
        db.do 'UPDATE Message SET creator = ?, title = ?, language = ?,
          format = ?, description = ?, content = ? WHERE id = ?',
          @creator, @title, @lang, @format, @desc, @content, @parent.id
        db.do "UPDATE Resource SET published_date = current_timestamp WHERE id = ?",
          @parent.id
        # rename old file if old version had content in a file
        if @parent.format and not config['format']['inline'].include? @parent.format
          File.rename(content_filename(@parent.id).untaint,
            content_filename(new_id).untaint)
        end
        if :takeover == @action   # change s:openForAll
          db.do 'UPDATE Message SET open = ? WHERE id = ?',
            (@open or false), @parent.id
        end

      elsif :hide == @action or :unhide == @action
        db.do 'UPDATE Message SET hidden = ? WHERE id = ?',
          (:hide == @action), @parent.id

      elsif :displace == @action
        # displace content without saving previous version
        db.do "UPDATE Message SET creator = ?, title = ?, language = ?,
          format = ?, description = ?, content = ?, open = 'false' WHERE id = ?",
          @creator, @title, @lang, @format, @desc, @content, @parent.id

# todo: make Samizdat::RDF#assert grok this 
#
#        rdf.assert( %{
#ASSERT (dc::creator ?msg :creator)
#       (dc::title ?msg :title)
#       (dc::language ?msg :lang)
#       (dc::format ?msg :format)
#       (dc::description ?msg :desc)
#       (s::content ?msg :content)
#       (s::openForAll ?msg ?open)
#WHERE (s::id ?msg #{@parent.id})},
#          { :creator => @creator, :title => @title, :lang => @lang,
#            :format => @format, :desc => @desc, :content => @content,
#            :open => false } )

      elsif :reparent == @action
        rdf.assert( %{
UPDATE ?parent = :new_parent
WHERE (s::inReplyTo #{@parent.id} ?parent)},
          { :new_parent => @new_parent } )

      else   # new message
        id, = rdf.assert( %{
INSERT ?msg
WHERE (dc::creator ?msg :creator)
      (dc::title ?msg :title)
      (dc::language ?msg :language)
      (dc::format ?msg :format)
      (dc::description ?msg :desc)
      (s::content ?msg :content)
      (s::openForAll ?msg :open)
      (s::inReplyTo ?msg :parent)},
          { :creator => @creator, :title => @title, :language => @lang,
            :format => @format, :desc => @desc, :content => @content,
            :open => (@open or false), :parent => (@parent ? @parent.id : nil) } )
      end

      # generate html for new message
      if (not [:hide, :unhide, :reparent].include? @action) and Message.format_cacheable?(@format)
        Message.update_html(id, @content, @format)
      end

      # relate to focus if focus was selected
      if @focus
        resource = Resource.new(@session, id)
        focus = Focus.new(@session, @focus, resource)
        focus.rating = @rating
      end

      cache.flush   # some cached data is now obsolete
      if @file
        # todo: inject file into p2p net
        File.rename(@upload, content_filename(
          id, @format, @session.login).untaint)
      end
    end   # transaction

    # return location for redirect
    #
    if :reply == @action   # redirect to parent
      "#{@parent.id}#id#{id}"
    else
      id.to_s
    end
  end

  def preview_page
    t = @session.template
    warning = '<p>'+sprintf(_('Warning: content is longer than %s characters. In some situations, it will be truncated.'), config['limit']['short'])+'</p>' if @content and @content != Template.limit_string(@content, config['limit']['short'])
    t.page( _('Message Preview'),
      t.message(self, :preview) +
      warning.to_s +
      '<p>'+_("Press 'Back' button to change the message.")+'</p>' +
      t.form( 'message.rb',
        [:submit, 'confirm', _('Confirm')],
        [:hidden, 'title', @title],
        [:hidden, 'content', @file ? nil : @content],
        [:hidden, 'file', @file],
        [:hidden, 'focus', @focus],
        [:hidden, 'rating', @rating],
        [:hidden, 'lang', @lang],
        [:hidden, 'format', @format],
        [:hidden, 'desc', @desc],
        [:hidden, 'open', @open],
        [:hidden, 'parent', (@parent ? @parent.id : nil)],
        [:hidden, 'action', @action.to_s]
      )
    )
  end

  def edit_page
    t = @session.template
    if @edit   # show current content if inline
      @format = @parent.format
      @desc = @parent.desc
      @content = @parent.content if inline?
      # todo: message locking
    end
    head =
      case @action
      when :source
        @title
      when :reply
        _('Reply')
      when :edit
        _('Edit Message')
      when :hide
        _('Hide Message')
      when :unhide
        _('Unhide Message')
      when :takeover
        _('Take Over Message')
      when :displace
        _('Displace Message')
      when :reparent
        _('Reparent Message')
      else
        _('New Message')
      end
    pre_comment =
      case @action
      when :hide
        '<p class="moderation">' + _('The message will be hidden from public view.') + '</p>'
      when :unhide
        '<p class="moderation">' + _('The message will not be hidden from public view.') + '</p>'
      when :takeover
        '<p class="moderation">' + _('The message will be attributed to you after takeover.') + '</p>'
      when :displace
        '<p class="moderation">' + _('MESSAGE WILL BE COMPLETELY DISPLACED, NO RECOVERY WILL BE POSSIBLE. PLEASE PROVIDE DETAILED JUSTIFICATION FOR THIS ACTION.') + '</p>'
      when :reparent
        '<p class="moderation">' + _('This message will be moved to new parent') + '</p>' +
          Resource.new(@session, @parent.id).render(:short).to_s
      end
    post_comment =
      case @action
      when :reply
        t.box(_('Parent Message'),
          Resource.new(@session, @parent.id).render(:short))
      end

    # generate the form
    case @action
    when :source
      form = [
        [:label, 'content', _('Content')], [:textarea, 'content', @content]
      ]
    when :hide, :unhide
      form = [
        [:submit, 'confirm', _('Confirm')],
        [:hidden, 'parent', @parent.id],
        [:hidden, 'action', @action.to_s]
      ]
    when :reparent
      form = [
        [:label, 'new_parent', _('New Parent')],
          [:text, 'new_parent', @new_parent],
        [:submit, 'confirm', _('Confirm')],
        [:hidden, 'parent', @parent.id],
        [:hidden, 'action', @action.to_s]
      ]
    else
      form = [
        [:label, 'title', _('Title')], [:text, 'title', @title],
        [:label, 'content', _('Content')], [:textarea, 'content', @content],
        [:label, 'file', _('Upload content from file')],
          [:file, 'file', @file],
        [:hidden, 'parent', (@parent ? @parent.id : nil)],
        [:hidden, 'action', @action.to_s]
      ]

      # select focus for new message
      form.push(*t.focus_fields(nil, nil, false)) if
        @session.access('vote') and @parent.nil?

      # advanced parameters
      form.push(
        [:label, 'lang', _('Language of the message')],
          [:select, 'lang', ([@session.language] +
            config['locale']['languages'].to_a).uniq!, @lang],
        [:label, 'format', _('Format')],
          [:select, 'format', [[nil, _('Default')]] +
            config['format']['inline'].to_a, @format ],
        [:label, 'desc',
          _('Reference to description (ID or URL of another message on this site)')],
          [:text, 'desc', @desc],
        [:label, 'open', _('Editing is open for all members')],
          [:checkbox, 'open', @open,
            ((:edit == @action or :displace == @action) ? :disabled : nil)]
      ) if 'advanced' == @session.cookie('ui')

      form.push([:br], [:submit, 'preview', _('Preview')])
    end

    t.page(head, pre_comment.to_s + t.form('message.rb', *form) + post_comment.to_s)
  end

  # check if multimedia content upload is possible
  #
  def upload_enabled?
    if config.content_location then
      dir = @session.filename(config.content_location).untaint
      File.directory?(dir) and File.writable?(dir)
    end
  end

private

  # multimedia message content filename (wrapper around Message.content_path)
  #
  def content_filename(id, format=nil, login=nil)
    @session.filename(config.content_location) +
      Message.content_path(id, format, login)
  end

  # valid message publishing actions
  #
  ACTION = %w[source reply edit hide unhide takeover displace reparent]

  # actions that pre-load or modify parent message
  #
  EDIT = %w[source edit hide unhide takeover displace]

  # actions that require moderatorial priviledges
  #
  MODERATION = %w[hide unhide takeover displace reparent]

  # input validation
  #
  def check_new_message
      check_parent
      check_action if @parent

      if :reparent == @action
        @new_parent &&= check_reference @new_parent
      else
        check_focus if @focus
        @lang = check_lang
        @desc &&= check_reference @desc
        check_format
        check_file
      end
  end

  def check_parent
    if @parent
      raise UserError, _('Bad input') unless
        @parent = Resource.validate_id(@parent)
      @parent = Message.cached(@parent)
      @title ||= @parent.title
    end
  end

  def check_action
    ACTION.each do |action|
      if action == @action or @session.has_key?(action)
        @action = action.to_sym 
        break
      end
    end

    # default to reply when not specified or not authorized
    @action = :reply if
      not @action.kind_of?(Symbol) or
      (:edit == @action and not @session.id) or
      (MODERATION.include?(@action) and not @session.moderator)

    @edit = EDIT.include?(@action.to_s)

    # don't allow to edit or reply to old versions
    current = rdf.get_property(@parent.id, 'dct::isVersionOf')
    raise UserError, sprintf(_('Use only <a href="%s">current version</a> for replies and edits'), current) if
      current and not [:source, :displace].include? @action
    if :edit == @action   # check if message is open for editing
      owner = rdf.get_property(@parent.id, 'dc::creator')
      @open = rdf.get_property(@parent.id, 's::openForAll')
      raise UserError, _('You are not allowed to edit this message') if
        owner != @session.id and not @open
    end
  end

  def check_focus
    @focus = Resource.validate_id(@focus)
    @rating = Focus.validate_rating(@rating)
    if @focus and @rating and @session.access('vote')
      @focuses = [@focus]   # get it displayed in preview page
    else
      @focus = nil
      @rating = nil
    end
  end

  def check_lang
    raise UserError, _('Specified language is not supported on this site') if
      @lang and not config['locale']['languages'].include?(@lang)
    @lang = @parent.lang if @lang.nil? and @parent
    @lang or @session.language
  end

  # desc or new_parent should refer to an existing message and shouldn't recurse
  #
  def check_reference(link)
    link.gsub!(Regexp.new('\A' + Regexp.escape(@session.base) + '(.+)\z'), '\1')
    link = Resource.validate_id(link)
    raise UserError, _('Invalid message reference') if
      link.nil? or @parent.id == link
    label, = db.select_one("SELECT label FROM Resource WHERE literal = 'false' AND uriref = 'false' AND id = ?", link)
    raise UserError, _('Message reference should refer to an existing message') unless 'Message' == label
    link
  end

  # detect and check format
  #
  def check_format
    if @file.methods.include? 'content_type' and @file.size > 0
      @format = @file.content_type.strip   # won't work with FastCGI
      @format = 'image/jpeg' if @format == 'image/pjpeg'   # MSIE...
      @format = 'image/png' if @format == 'image/x-png'
    end
    raise UserError, sprintf(_("Format '%s' is not supported"),
      CGI.escapeHTML(@format)) if
        @format and
        (not config['format'].values.flatten.include?(@format) or
         not (upload_enabled? or config['format']['inline'].include?(@format)))
  end

  # if content is uploaded from file, perform all necessary file operations,
  # set @content to nil
  #
  def check_file
    if @file and @file.size > 0   # content uploaded from file
      @content = nil
      @format.nil? and raise UserError,
        _('It is not possible to upload a file without specifying format')
      # todo: fine-grained size limits
      @file.size > config['limit']['content'] and raise UserError,
        sprintf(_('Uploaded file is larger than %s bytes limit'),
          config['limit']['content'])

      @upload = content_filename('upload', @format, @session.login)
      @upload.untaint   # security: keep format and login controlled

      if (@file.kind_of? StringIO or @file.kind_of? Tempfile) and not @session.has_key? 'confirm'   # new upload
        if inline?
          @content = @file.read   # transform to inline message
          @file = nil
        else
          upload_enabled? or raise UserError,
            _('Multimedia upload is disabled on this site')
          FileUtils.mkdir_p(File.dirname(@upload)) unless
            File.exists?(File.dirname(@upload))
          if @file.kind_of? Tempfile   # copy large files directly
            FileUtils.cp(@file.path, @upload)
          else   # StringIO
            File.open(@upload, 'w') {|f| f.write(@file.read) }
          end
          @file = true
        end

      elsif @file.kind_of? String and @session.has_key? 'confirm'
        inline? and raise UserError, 'Unexpected inline upload confirm'
        @file = nil unless File.exists?(@upload)
      else
        raise UserError, 'Unexpected upload state'
      end
    end   # at this point, file is true and content is nil if upload is ready

    if @content   # inline message
      @file = nil
      inline? or raise UserError,
        sprintf(_('You should upload %s content from file'), @format)
    end
  end
end
