#!/usr/bin/env ruby
#
# Samizdat search query construction
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

# validate query (syntax, must-bind list size, number of clauses)
#
def validate_query(query)
  begin
    q = Samizdat::SquishQuery.new(query)
    sql, = rdf.select(q)
  rescue ProgrammingError
    raise UserError, _('Error in your query: ') + CGI.escapeHTML($!.message)
  end

  (q.nodes.size != 1) and raise UserError,
_('Must-bind list should contain only one blank node, filters based on queries with a complex answer pattern are not implemented')

  (q.pattern.size > config['limit']['pattern']) and raise UserError,
    sprintf(_('User-defined query pattern should not be longer than %s clauses'), config['limit']['pattern'])

  q
end

# results of a query run
#
def collect_results(query, skip)
  list = cache.fetch_or_add(%{#{digest(query)}/run/#{skip}}) do
    q = validate_query(query)
    rdf.select_all(q, limit_page, limit_page * skip).collect {|m,| m }
  end
  list.collect {|msg| yield msg }
end

# RSS feed of a query run
#
def feed_rss(session, query)
  session.options.update('type' => 'application/xml')

  require 'rss/maker'
  RSS::Maker.make("1.0") do |maker|
    maker.channel.about = session.base +
      %{query.rb?query=#{CGI.escape(query)}&amp;format=rss}
    title = config['site']['name'] + ' / ' + _('Search Result')
    maker.channel.title = title
    maker.channel.description = %{<pre>#{CGI.escapeHTML(query)}</pre>}
    maker.channel.link = session.base +
      %{query.rb?query=#{CGI.escape(query)}&amp;run}

    if config['site']['icon']
      maker.image.title = title
      maker.image.url = config['site']['icon']
    end

    collect_results(query, 0) do |id|
      r = Resource.new(session, id).render(:short)
      if 'Message' == r[:type]
        Message.cached(id).rss(maker, session)
      else
        item = maker.items.new_item
        item.link = session.base + id.to_s
        item.date = rdf.get_property(id, 'dc::date').to_time
        item.title = r[:head]
      end
    end
  end
end

request do |session|
  namespaces = config['ns']   # standard namespaces

  t = session.template
  query, substring, skip = session.params %w[query substring skip]
  skip = skip.to_i

  next feed_rss(session, query).to_s if 'rss' == session['format']

  # define query
  #

  if substring and session.has_key? 'search'
    # search messages by a substring
    query = %{SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::title ?msg ?title)
LITERAL ?title ILIKE '%#{substring.gsub(/%/, '\\\\\\%')}%'
ORDER BY ?date DESC}
    session['run'] = 'Run'
  end

  if session.has_key? 'update'   # regenerate query from form data
    nodes, literal, order, order_dir, using =
      session.params %w[nodes literal order order_dir using]
    # generate namespaces
    namespaces.update(Hash[*using.split])
    using = {}
    # generate query pattern
    pattern = []
    session.keys.grep(/\A(predicate|subject|object)_(\d{1,2})\z/) do |key|
      value, = session.params([key])
      next unless value
      i = $2.to_i - 1
      pattern[i] = [] unless pattern[i]
      pattern[i][ %w[predicate subject object].index($1) ] = value
      namespaces.each do |p, uri|
        if /\A#{p}::/ =~ value
          using[p] = uri   # only leave used namespace prefixes
          break
        end
      end
    end

    query = "SELECT #{nodes}\nWHERE " +
      pattern.collect {|predicate, subject, object|
        "(#{predicate} #{subject} #{object})" if
          predicate and subject and object and predicate !~ /\s/
          # whitespace is only present in 'BLANK CLAUSE'
        }.compact.join("\n      ")
      query += "\nLITERAL #{literal}" if literal
      query += "\nORDER BY #{order} #{order_dir}" if order
      query += "\nUSING " + using.to_a.collect {|n|
        n.join(' FOR ')
      }.join("\n      ")
      session['run'] = 'Run'
  end

  if query.nil? or session.keys.size == 0   # default query
    query = %{SELECT ?resource
WHERE (dc::date ?resource ?date)
ORDER BY ?date DESC}
  end

  q = validate_query(query)

  # act on query
  #

  if session.has_key? 'run'
    title = _('Search Result') +
      (skip > 0? sprintf(_(', page %s'), skip + 1) : '')
    page = collect_results(query, skip) {|id| Resource.new(session, id).render }
    page =
      if page.size > 0
        rss_link = %{query.rb?query=#{CGI.escape(query)}&amp;format=rss}
        rss = { title => rss_link } unless skip > 0

        foot = ('advanced' == session.cookie('ui') and not skip > 0)?
          t.form('message.rb',
            [:hidden, 'format', 'application/x-squish'],
            [:hidden, 'content', query],
            [:submit, 'preview', _('Publish This Query')]) :
          ''
        t.list(page, t.nav_rss(rss_link) + t.nav(page.size, skip + 1,
          %{query.rb?run&amp;query=#{CGI.escape(query)}&amp;}), foot)
      else
        '<p>'+_('No matching resources found.')+'</p>'
      end
  end

  page = page ? [[title, page]] : []   # single- or multi-section page

  # query construction interface
  #

  edit = t.box(_('Select Target'),
    t.form_field(:select, 'nodes', q.pattern.collect {|p, s, o| s }.uniq))

  i = 0
  edit += t.box( _('Query Pattern (predicate subject object)'),
    ((q.pattern.size >= config['limit']['pattern'])?
      q.pattern :
      q.pattern + ['']
    ).collect {|clause|
      predicate, subject, object = clause.collect {|uri| q.ns_shrink(uri) }
      i += 1

      %{#{i}. } +
      # todo: add external properties to the options list
      t.form_field(:select, "predicate_#{i}",
        [_('BLANK CLAUSE ')] + config['map'].keys.sort, predicate) +
        # make sure 'BLANK CLAUSE' includes whitespace in all translations
      t.form_field(:text, "subject_#{i}", subject) +
      t.form_field(:text, "object_#{i}", q.substitute_literals(object)) +
      t.form_field(:br)
      # todo: select subject and object from variables and known urirefs
    }.join
  )

  edit += t.box(_('Literal Condition'),
    t.form_field(:text, 'literal', q.substitute_literals(q.literal)))

  edit += t.box(_('Order By'),
    t.form_field(:select, 'order',
      q.pm.keys.grep(Samizdat::SquishQuery::BN), q.order) +
    t.form_field(:select, 'order_dir',
      [['ASC', _('Ascending')], ['DESC', _('Descending')]], q.order_dir))

  edit += t.box(nil, '<pre>USING ' +
    q.ns.to_a.collect {|n| n.join(' FOR ') }.join("\n      ") + '</pre>' +
    t.form_field(:hidden, 'using', q.ns.to_a.flatten.join(' ')))

  # wrap edit in other query variants
  editbox = t.box(_('Search By Substring'),
    t.form_field(:text, 'substring', substring) +
    t.form_field(:submit, 'search', _('Search')))
  if 'advanced' == session.cookie('ui')
    editbox +=
      t.box(_('Construct Query'), edit +   # here
        t.form_field(:submit, 'update', _('Update'))) +
      t.box(_('Edit Raw Query'),
        t.form_field(:textarea, 'query', query) +
        t.form_field(:label) +
        t.form_field(:submit, 'run', _('Run')))
  end

  # wrap editbox in form tag
  editbox = %{<form action="query.rb" method="post">#{editbox}</form>}

  page.push [_('Edit Query'), editbox] unless skip > 0
  page.collect! {|h, b| [h, q.substitute_literals(b)] }

  t.page(nil, page, {:rss => rss})
end
