/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2000,2004 Rainer Wichmann                                 */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */


#include "config_xor.h"


#include <stdio.h>
#include <string.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

/* replace #if 0 by #if 1 and set an appropriate path in front of '/pdbg.'
 * for debugging
 */
#if 0
#define PDGBFILE "/pdbg."
#endif


#if defined(PDGBFILE)
static FILE * pdbg = NULL;
static FILE * pdbgc = NULL;
#define PDBG_OPEN    if (pdbg == NULL) pdbg = fopen(PDGBFILE"main",  "a")  
#define PDBG_CLOSE   fclose (pdbg); pdbg = NULL
#define PDBG(arg)    fprintf(pdbg,  "PDBG: step %d\n", arg); fflush(pdbg)
#define PDBG_D(arg)  fprintf(pdbg,  "PDBG: %d\n", arg); fflush(pdbg)
#define PDBG_S(arg)  fprintf(pdbg,  "PDBG: %s\n", arg); fflush(pdbg)

#define PDBGC_OPEN   if (pdbgc == NULL) pdbgc = fopen(PDGBFILE"child", "a")  
#define PDBGC_CLOSE  fclose (pdbgc); pdbgc = NULL
#define PDBGC(arg)   fprintf(pdbgc, "PDBG: step %d\n", arg); fflush(pdbgc)
#define PDBGC_D(arg) fprintf(pdbgc, "PDBG: %d\n", arg); fflush(pdbgc)
#define PDBGC_S(arg) fprintf(pdbgc, "PDBG: %s\n", arg); fflush(pdbgc)
#else
#define PDBG_OPEN    
#define PDBG_CLOSE   
#define PDBG(arg)    
#define PDBG_D(arg)  
#define PDBG_S(arg)  
#define PDBGC_OPEN    
#define PDBGC_CLOSE   
#define PDBGC(arg)    
#define PDBGC_D(arg)  
#define PDBGC_S(arg)  
#endif


#include <stdlib.h>
#include <pwd.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/wait.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif


#include "samhain.h"
#include "sh_utils.h"
#include "sh_unix.h"
#include "sh_tiger.h"
#include "sh_extern.h"
#include "sh_calls.h"
#define SH_NEED_PWD_GRP 1
#include "sh_static.h"


#undef  FIL__
#define FIL__  _("sh_extern.c")

extern int get_the_fd (SL_TICKET ticket);

/*
 * -- generic safe popen
 */

int sh_ext_popen (sh_tas_t * task)
{
  long status = 0;
  int    flags;
  char * tmp;
  char * tmp2;
  int    errnum;
  int    pipedes[2];
  FILE * outf = NULL;
  char * envp[1];
  char * argp[1];

  char * errfile;

  static int some_error = 0;

#if defined (__linux__)
  SL_TICKET   fd  = -1;
  char        pname[128];
  int         pfd = -1;
#endif
  
  SL_ENTER(_("sh_ext_popen"));

  /* Linux, HP-UX and FreeBSD will happily accept envp = argp = NULL
   * Solaris (and probably some other Unices) 
   *         needs a valid *envp[] with envp[0] = NULL;
   *         and similarly for argp
   */
  envp[0] = NULL;
  argp[0] = NULL;

  /* 
   * --  check whether path is trustworthy
   */
  if ((uid_t) -1 != task->trusted_users[0])
    {
      status = sl_trustfile(task->command, task->trusted_users, NULL);
    }

  PDBG_OPEN;
  PDBG_D( (int) status);

  if ( SL_ENONE != status)
    { 
      PDBG_S("SL_ENONE != status");
      if (some_error == 0)
	{
	  tmp  = sh_util_safe_name (task->command);
	  errfile = sl_trust_errfile();
	  if (errfile[0] != '\0')
	    {
	      tmp2  = sh_util_safe_name (sl_trust_errfile());
	      sh_error_handle((-1), FIL__, __LINE__, status, MSG_E_TRUST2,
			      sl_error_string((int)status), tmp, tmp2);
	      SH_FREE(tmp2);  
	    }
	  else
	    {
	      sh_error_handle((-1), FIL__, __LINE__, status, MSG_E_TRUST1,
			      sl_error_string((int)status), tmp);
	    }
	  SH_FREE(tmp);
	}
      some_error = 1;
      SL_RETURN ((-1), _("sh_ext_popen"));
    }

  PDBG(1);

  /* 
   * --  check whether the checksum is correct; with linux emulate fdexec
   */
#if !defined(__linux__) && !defined(SL_DEBUG)
  if (task->checksum[0]  != '\0')
    {
      PDBG_S("checksum test");
      if (0 != sl_strcmp(task->checksum, 
			 sh_tiger_hash (task->command, TIGER_FILE, 0))
	  )
	{
	  PDBG_S("checksum mismatch");
	  if (some_error == 0)
	    {
	      tmp  = sh_util_safe_name (task->command);
	      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_HASH, tmp);
	      SH_FREE(tmp);
	    }
	  some_error = 1;
	  SL_RETURN ((-1), _("sh_ext_popen"));
 	}
    }
#endif

  some_error = 0;

  PDBG(2);

  /* 
   * -- Create the pipe 
   */
  if (aud_pipe(FIL__, __LINE__, pipedes) < 0) 
    {
      PDBG_S("pipe() failure");
      errnum = errno;
      sh_error_handle((-1), FIL__, __LINE__, errnum, MSG_E_SUBGEN, 
		      sh_error_message(errnum), _("pipe"));
      SL_RETURN ((-1), _("sh_ext_popen"));
    }

  PDBG(3);
  
  /* 
   * -- Fork 
   */
  task->pid = aud_fork(FIL__, __LINE__);

  if (task->pid == (pid_t) - 1) 
    {
      PDBG_S("fork() failure");
      /*@-usedef@*/
      (void) close(pipedes[0]);
      (void) close(pipedes[1]);
      /*@+usedef@*/
      errnum = errno;
      sh_error_handle((-1), FIL__, __LINE__, errnum, MSG_E_SUBGEN, 
		      sh_error_message(errnum), _("fork"));
      SL_RETURN ((-1), _("sh_ext_popen"));
    }
  
  PDBG(4);

  if (task->pid == (pid_t) 0) 
    {
      /* 
       * -- fork again, if requested
       */
      if (S_TRUE == task->fork_twice)
	{
	  task->pid = aud_fork(FIL__, __LINE__);

	  if (task->pid == (pid_t) - 1) 
	    {
	      aud__exit (FIL__, __LINE__, EXIT_FAILURE);
	    }
	}

      if (task->pid == (pid_t) 0)
	{
	  PDBGC_OPEN;
	  PDBGC(1);

	  /*
	   * -- grandchild - make write side of the pipe stdin 
	   */
	  if (task->rw == 'w')
	    {
	      if (retry_aud_dup2(FIL__, __LINE__, 
				 pipedes[STDIN_FILENO], STDIN_FILENO) < 0)
		aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	    }
	  else
	    {
	      if (retry_aud_dup2(FIL__, __LINE__,
				 pipedes[STDOUT_FILENO], STDOUT_FILENO) < 0)
		aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	    }
	  PDBGC(2);
	    
	  
	  /* close the pipe descriptors 
	   */
	  (void) close   (pipedes[STDIN_FILENO]);
	  (void) close   (pipedes[STDOUT_FILENO]);
	  
	  /* don't leak file descriptors
	   */
#if !defined(PDGBFILE)
	  sh_unix_closeall (3, task->com_fd); /* in child process */
#endif

	  /* drop root privileges, if possible && requested
	   */
	  if (task->privileged == 0 && 0 == getuid())
	    {
	      PDBGC_S("privileged");

	      /* zero priv info
	       */
	      memset(skey, 0, sizeof(sh_key_t));

	      (void) aud_setgid(FIL__, __LINE__,(gid_t) task->run_user_gid);
	      (void) aud_setuid(FIL__, __LINE__,(uid_t) task->run_user_uid);
	      /* make sure we cannot get root again
	       */
	      if (aud_setuid(FIL__, __LINE__,0) >= 0)
		aud__exit(FIL__, __LINE__,EXIT_FAILURE);
	    }
	  
	  PDBGC(3);
	  (void) fflush(NULL);
	  
	  if (task->rw == 'w')
	    {
	      PDBGC_S("w");
	      (void) fcntl  (STDOUT_FILENO, F_SETFD, FD_CLOEXEC);
	      (void) fcntl  (STDERR_FILENO, F_SETFD, FD_CLOEXEC);
	      /*
	      freopen(_("/dev/null"), "r+", stderr);
	      freopen(_("/dev/null"), "r+", stdout);
	      */
	    }
	  else
	    {
	      PDBGC_S("r");
	      (void) retry_aud_dup2 (FIL__, __LINE__, 
				     STDOUT_FILENO, STDERR_FILENO);
	      (void) fcntl  (STDIN_FILENO, F_SETFD, FD_CLOEXEC);
	      /*
	      freopen(_("/dev/null"), "r+", stdin);
	      */
	    }
	  
	  PDBGC(4);
	  
	  
#if defined(__linux__)
	  /* 
	   * --  emulate an fdexec with checksum testing
	   */
	  if (task->checksum[0]  != '\0')
	    {
	      PDBGC_S("fexecve");
	      if (task->com_fd != (-1))
		{
		  pfd = retry_aud_dup(FIL__, __LINE__, task->com_fd);
		  if (pfd < 0)
		    {
		      PDBGC_S("fexecve: dup2 failed");
		      aud__exit(FIL__, __LINE__, EXIT_FAILURE);
		    }
		}
	      else
		{
		  fd = 
		    sl_open_read(task->command, 
				 task->privileged==0 ? SL_NOPRIV : SL_YESPRIV);
		  tiger_fd = fd;
		  if (0 != sl_strcmp(task->checksum, 
				     sh_tiger_hash (task->command, 
						    TIGER_FD, 0)))
		    {
		      PDBGC_S("fexecve: checksum mismatch");
		      aud__exit(FIL__, __LINE__, EXIT_FAILURE);
		    }
		  pfd = get_the_fd(fd);
		}
              
	      PDBGC(5);
	      sprintf(pname, _("/proc/self/fd/%d"),      /* known to fit  */
			   pfd);
              if (access(pname, R_OK|X_OK) == 0) 
		{
		  PDBGC(6);
		  PDBGC_CLOSE;
		  fcntl  (pfd, F_SETFD, FD_CLOEXEC);
		  retry_aud_execve (FIL__, __LINE__, 
				    pname, 
				    (task->argc == 0) ? NULL : task->argv, 
				    (task->envc == 0) ? NULL : task->envv
				    );
		  
		  errnum = errno;
		  PDBGC_OPEN;
		  PDBGC_S(strerror(errnum));
		  PDBGC_S(task->command);
		  PDBGC_S("fexecve: failed");
		  PDBGC_CLOSE;
		  /* failed 
		   */
		  aud__exit(FIL__, __LINE__, EXIT_FAILURE);
              }
	      PDBGC_S("fexecve: not working");
	      /* 
	       * procfs not working, go ahead; checksum is tested already
	       */
	      if (fd != -1)
		sl_close(fd);
	      else if (pfd != -1)
		close(fd);
	    }
#endif

	  PDBGC_S(" -- non fexecve --");
	  /* 
	   * --  execute path if executable
	   */
	  if (0 == access(task->command, R_OK|X_OK))
	    {
	      PDBGC(5);
	      PDBGC_CLOSE;
	      (void) retry_aud_execve (FIL__, __LINE__, 
				       task->command, 
				       (task->argc == 0) ? argp : task->argv, 
				       (task->envc == 0) ? envp : task->envv
				       );
	    }
	  errnum = errno;
	  PDBGC_OPEN;
	  PDBGC_S(strerror(errnum));
	  PDBGC_S(task->command);
	  PDBGC_S("execve: failed");
	  PDBGC_CLOSE;
	  /* failed 
	   */
	  aud__exit(FIL__, __LINE__, EXIT_FAILURE);
	}
      /* 
       * if we have forked twice, this is parent::detached_subprocess
       */
      if (S_TRUE == task->fork_twice)
	{
	  aud__exit (FIL__, __LINE__, 0);
	}
    }

  
  /*
   * -- parent; task->pid is child pid; exit status is status of
   *    grandchild if exited
   */
  if (S_TRUE == task->fork_twice)
    {
      (void) waitpid (task->pid, NULL, 0);
    }

  PDBG(5);
  /* open an output stream on top of the write side of the pipe
   */
  if (task->rw == 'w')
    {
      PDBG_S("is w");
      (void) close (pipedes[STDIN_FILENO]);
      (void) retry_fcntl (FIL__, __LINE__, pipedes[STDOUT_FILENO], 
			  F_SETFD, FD_CLOEXEC);
      outf = fdopen (pipedes[STDOUT_FILENO], "w");
    }
  else
    {
      PDBG_S("is r");
      (void) close (pipedes[STDOUT_FILENO]);
      (void) retry_fcntl (FIL__, __LINE__, pipedes[STDIN_FILENO], 
			  F_SETFD, FD_CLOEXEC);
      outf = fdopen (pipedes[STDIN_FILENO], "r");
    }

  if (outf == NULL) 
    {
      errnum = errno;
      PDBG_S("outf == NULL");
      tmp  = sh_util_safe_name (task->command);
      
      if (task->privileged == 0 && 0 == getuid())
	sh_error_handle((-1), FIL__, __LINE__, errnum, MSG_NOEXEC,
			(UID_CAST) task->run_user_uid, tmp);
      else
	sh_error_handle((-1), FIL__, __LINE__, errnum, MSG_NOEXEC,
			(UID_CAST) getuid(), tmp);

      SH_FREE(tmp);

      (void) aud_kill (FIL__, __LINE__, task->pid, SIGKILL);
      (void) close (pipedes[STDOUT_FILENO]);
      (void) close (pipedes[STDIN_FILENO]);
      (void) waitpid (task->pid, NULL, 0);
      task->pid = 0;

      SL_RETURN ((-1), _("sh_ext_popen"));
    }
  
  if (task->rw == 'w')
    task->pipeFD   = pipedes[STDOUT_FILENO];
  else
    task->pipeFD   = pipedes[STDIN_FILENO];

  PDBG_D(task->pipeFD);

  task->pipeTI = sl_make_ticket(task->pipeFD, _("pipe"));

  flags = (int) retry_fcntl (FIL__, __LINE__, task->pipeFD, F_GETFL, 0);
  if (flags != (-1))
    (void) retry_fcntl (FIL__, __LINE__, task->pipeFD, 
			F_SETFL, flags|O_NONBLOCK);
  task->pipe     = outf;

  PDBG_S("return from popen");
  PDBG_CLOSE;
  
  SL_RETURN (0, _("sh_ext_popen"));
}

/*
 * -- close the pipe
 */
extern int flag_err_debug;

int sh_ext_pclose (sh_tas_t * task)
{
  int   status = 0;
  int   retry  = 0;
  pid_t retval;
  char  infomsg[256];

  SL_ENTER(_("sh_ext_pclose"));

  PDBGC_OPEN;
  PDBG_S(" -> pclose");
  (void) fflush(task->pipe);
  (void) fclose(task->pipe);
  if (!SL_ISERROR(task->pipeTI))
    (void) sl_close(task->pipeTI);

  task->pipe     = NULL;
  task->pipeFD   = (-1);
  task->pipeTI   = SL_ETICKET;

  if (S_FALSE == task->fork_twice)
    {
      infomsg[0] = '\0';

    nochmal:
      retval = waitpid(task->pid, &(task->exit_status), WNOHANG|WUNTRACED);
      /*@-bufferoverflowhigh@*/
      if (task->pid == retval)
	{
	  if (WIFEXITED(task->exit_status) != 0)
	    {
	      task->exit_status = WEXITSTATUS(task->exit_status);
	      if ((flag_err_debug == SL_TRUE) || (task->exit_status != 0))
		sprintf(infomsg,                         /* known to fit  */
			_("Subprocess exited normally with status %d"),
			task->exit_status);
	    }
	  else if (WIFSIGNALED(task->exit_status) != 0)
	    {
	      sprintf(infomsg,                           /* known to fit  */
		      _("Subprocess terminated by signal %d"),
		      WTERMSIG(task->exit_status));
	      task->exit_status = EXIT_FAILURE;
	    }
	  else if (WIFSTOPPED(task->exit_status) != 0)
	    {
	      sprintf(infomsg,                           /* known to fit  */
		      _("Subprocess stopped by signal %d, killing"),
		      WSTOPSIG(task->exit_status));
	      task->exit_status = EXIT_FAILURE;
	      (void) aud_kill (FIL__, __LINE__, task->pid, 9);
	      (void) retry_msleep (0, 30);
	      (void) waitpid (task->pid, NULL, WNOHANG|WUNTRACED);
	    }
	  else
	    {
	      sprintf(infomsg,                           /* known to fit  */
		      _("Subprocess exit status unknown"));
	      task->exit_status = EXIT_FAILURE;
	    }
	}
      else if (0 == retval)
	{
	  if (retry < 3)
	    {
	      ++retry;
	      (void) retry_msleep(0, (retry * 30));
	      goto nochmal;
	    }
	  (void) aud_kill (FIL__, __LINE__, task->pid, 9);
	  sprintf(infomsg,                               /* known to fit  */
		  _("Subprocess not yet exited, killing"));
	  task->exit_status = EXIT_FAILURE;
	  (void) waitpid (task->pid, NULL, 0);
	}
      else
	{
	  sprintf(infomsg,                               /* known to fit  */
		  _("Waitpid returned error %d\n"), errno);
	  task->exit_status = EXIT_FAILURE;
	}
      /*@+bufferoverflowhigh@*/
      status = task->exit_status;
      if (flag_err_debug == SL_TRUE)
	{
	  sh_error_handle(SH_ERR_ALL, FIL__, __LINE__, task->exit_status, 
			  MSG_E_SUBGEN, infomsg, _("sh_ext_pclose"));
	}
      else if (status != 0)
	{
	  sh_error_handle(SH_ERR_INFO, FIL__, __LINE__, task->exit_status, 
			  MSG_E_SUBGEN, infomsg, _("sh_ext_pclose"));
	}
    }

  task->pid = 0;
  task->exit_status = 0;
  PDBG_S(" <--");
  PDBG_CLOSE;
  SL_RETURN (status, _("sh_ext_pclose"));
}

void sh_ext_tas_init (sh_tas_t * tas)
{
  int i;

  tas->command       = NULL;
  tas->argc          = 0;
  tas->envc          = 0;
  tas->checksum[0]   = '\0';
  tas->pipeFD        = (-1);
  tas->pipeTI        = SL_ETICKET;
  tas->pid           = (pid_t) -1;
  tas->privileged    = 1;
  tas->pipe          = NULL;
  tas->rw            = 'w';
  tas->exit_status   = 0;
  tas->fork_twice    = S_TRUE;

  for (i = 0; i < 32; ++i)
    {
      tas->argv[i]          = NULL;
      tas->envv[i]          = NULL;
      tas->trusted_users[i] = (uid_t) -1;
    }

  tas->run_user_uid     = (uid_t) getuid();
  tas->run_user_gid     = (gid_t) getgid();

  tas->com_fd = -1;
  tas->com_ti = -1;
  return;
}


int sh_ext_tas_add_envv(sh_tas_t * tas, char * key, char * val)
{
  size_t sk = 0, sv = 0;
  int    si;

  SL_ENTER(_("sh_ext_tas_add_envv"));

  if (tas == NULL ||  (key == NULL      && val == NULL)      || 
      tas->envc >= 30)
    {
      SL_RETURN (-1, _("sh_ext_tas_add_envv"));
    }
  if (key != NULL)
    sk = strlen(key) + 1;
  if (val != NULL)
    sv = strlen(val) + 1;

  si = tas->envc;
  tas->envv[si] = SH_ALLOC(sk + sv);

  if (key != NULL)
    {
      (void) sl_strlcpy(tas->envv[si], key, sk+sv);
      (void) sl_strlcat(tas->envv[si], "=", sk+sv);
      if (val != NULL)
	(void) sl_strlcat(tas->envv[si], val, sk+sv);
    }
  else
    (void) sl_strlcpy(tas->envv[si], val, sv);

  ++(tas->envc);
  SL_RETURN ((tas->envc), _("sh_ext_tas_add_envv"));
}

int sh_ext_tas_rm_argv(sh_tas_t * tas)
{
  int last;

  SL_ENTER(_("sh_ext_tas_rm_argv"));
  if (tas == NULL || tas->argc == 0)
    {
      SL_RETURN (-1, _("sh_ext_tas_rm_argv"));
    }

  last = (tas->argc - 1);
  --(tas->argc);
  SH_FREE(tas->argv[last]);
  tas->argv[last] = NULL;
  SL_RETURN ((tas->argc), _("sh_ext_tas_rm_argv"));
}

int sh_ext_tas_add_argv(sh_tas_t * tas, char * val)
{
  size_t sv = 0;
  int    si;

  SL_ENTER(_("sh_ext_tas_add_argv"));

  if (tas == NULL ||  val == NULL  || 
      tas->argc >= 30)
    {
      SL_RETURN (-1, _("sh_ext_tas_add_argv"));
    }

  if (val != NULL)
    sv = strlen(val) + 1;

  si = tas->argc;
  tas->argv[si] = SH_ALLOC(sv);

  (void) sl_strlcpy(tas->argv[si], val, sv);

  ++(tas->argc);
  SL_RETURN ((tas->argc), _("sh_ext_tas_add_argv"));
}

void sh_ext_tas_command(sh_tas_t * tas, char * command)
{
  size_t len = sl_strlen(command);
  tas->command = SH_ALLOC(len+1);
  (void) sl_strlcpy(tas->command, command, len+1);
  return;
}

void sh_ext_tas_free(sh_tas_t * tas)
{
  int i;
  if (NULL != tas->command)    SH_FREE(tas->command);
  
  for (i = 0; i < 32; ++i)
    {
      if (NULL != tas->argv[i])   SH_FREE(tas->argv[i]);
      if (NULL != tas->envv[i])   SH_FREE(tas->envv[i]);
    }

  if (tas->com_ti != (-1))
    {
      (void) sl_close(tas->com_ti);
      tas->com_ti = -1;
      tas->com_fd = -1;
    }

  return;
}

/* ---------------  EXTERN STUFF ------------------- */

#if defined(WITH_EXTERNAL)

typedef struct _sh_com_t
{
  char     type[4];

  int      for_c;
  char   * for_v[32];
  int      fand_c;
  char   * fand_v[32];
  int      fnot_c;
  char   * fnot_v[32];
  time_t   deadtime;
  time_t   last_run;

  sh_tas_t tas;

  struct _sh_com_t * next;

} sh_com_t;

static
void set3 (char * pos, char c1, char c2, char c3)
{
  pos[0] = c1;
  pos[1] = c2;
  pos[2] = c3;
  pos[3] = '\0';
  return;
}



/* initialize the external command structure
 */
static
sh_com_t * command_init(void)
{
  int         i;
  uid_t       ff_euid;
  sh_com_t  * ext_com = NULL;

  SL_ENTER(_("command_init"));

  ext_com = (sh_com_t *) SH_ALLOC(sizeof(sh_com_t));

  if (!ext_com)
    {
      SL_RETURN( NULL, ("command_init"));
    }

  sh_ext_tas_init (&(ext_com->tas));

  (void) sl_get_euid(&ff_euid);
  ext_com->tas.trusted_users[0] = (uid_t) 0;
  ext_com->tas.trusted_users[1] = (uid_t) (ff_euid);

  /* ------------------------------------------------- */

  set3(ext_com->type, 'l', 'o', 'g');
  ext_com->for_c        = 0;
  ext_com->fand_c       = 0;
  ext_com->fnot_c       = 0;
  ext_com->deadtime     = 0;
  ext_com->last_run     = 0;

  for (i = 0; i < 32; ++i)
    {
      ext_com->for_v[i]         = NULL;
      ext_com->fand_v[i]        = NULL;
      ext_com->fnot_v[i]        = NULL;
    }
  ext_com->next             = NULL;

  SL_RETURN( ext_com, ("command_init"));
}

/* the list of external commands
 */
static sh_com_t * ext_coms   = NULL;

/* if -1, allocation of last command has failed,
 * thus don't fill in options
 */
static int ext_failed = -1;

static
int sh_ext_add_envv(char * key, char * val)
{
  SL_ENTER(_("sh_ext_add_envv"));

  if (ext_coms == NULL || ext_failed == (-1) || 
      (key == NULL      && val == NULL)      || 
      ext_coms->tas.envc >= 30)
    {
      SL_RETURN (-1, _("sh_ext_add_envv"));
    }

  (void) sh_ext_tas_add_envv(&(ext_coms->tas), key, val);

  SL_RETURN (0, _("sh_ext_add_envv"));
}



static 
int sh_ext_init(char * command)
{
  sh_com_t * retval;
  size_t     size;

  SL_ENTER(_("sh_ext_init"));

  if (command == NULL)
    {
      SL_RETURN (-1, _("sh_ext_init"));
    }
  size = strlen(command);
  if (command[0] != '/' || size < 2)
    {
      SL_RETURN (-1, _("sh_ext_init"));
    }

  if (NULL == (retval = command_init()))
    {
      SL_RETURN (-1, _("sh_ext_init"));
    }

  sh_ext_tas_command(&(retval->tas), command);

  if (sh.timezone != NULL)
    {
      (void) sh_ext_add_envv( "TZ", sh.timezone);
    }

  retval->next = ext_coms;
  ext_coms     = retval;
  SL_RETURN (0, _("sh_ext_init"));
}

static
int sh_ext_uid (char * user, /*@out@*/uid_t * uid, /*@out@*/gid_t * gid)
{
  struct passwd * tempres;

  SL_ENTER(_("sh_ext_uid"));

  *uid = (uid_t)-1; *gid = (gid_t)-1;

  if (user == NULL)
    {
      SL_RETURN (-1, _("sh_ext_uid"));
    }
  tempres = sh_getpwnam(user);

  if (NULL != tempres) 
    {
      *uid = tempres->pw_uid;  
      *gid = tempres->pw_gid;
      SL_RETURN (0, _("sh_ext_uid"));
    } 

  SL_RETURN (-1, _("sh_ext_uid"));
}


static
int sh_ext_add (char * argstring, int * ntok, char * stok[])
{
  int    i = 0;
  size_t s;
  char * p;

  SL_ENTER(_("sh_ext_add"));

  if (NULL == argstring)
    {
      SL_RETURN((-1), _("sh_ext_add")); 
    }

  do
    {
      if (i == 0)
	p = strtok (argstring, ", \t");
      else
	p = strtok (NULL, ", \t");
      if (p == NULL)
	break;

      s = strlen(p) + 1;
      if (stok[i] != NULL)
	SH_FREE(stok[i]);
      stok[i] = SH_ALLOC(s);
      (void) sl_strlcpy(stok[i], p, s);

      ++i;
      if (i == 30)
	break;
    }
  while (p != NULL);

  *ntok = i;

  SL_RETURN (0, _("sh_ext_add"));
}

/*********************************************************
 *
 * Public functions
 *
 *
 *********************************************************/
 
/* 
 * -- start a new external command, and add it to the list
 */ 
int sh_ext_setcommand(char * cmd)
{
  int i;

  SL_ENTER(_("sh_ext_setcommand"));
  if ( (i = sh_ext_init(cmd)) < 0)
    ext_failed = -1;
  else
    ext_failed = 0;
  SL_RETURN( i, _("sh_ext_setcommand"));
}


/* 
 * -- clean up the command list
 */
int sh_ext_cleanup(void)
{
  int i;
  sh_com_t * retval;

  SL_ENTER(_("sh_ext_cleanup"));

  while (ext_coms != NULL)
    {
      retval   = ext_coms;
      ext_coms = retval->next;

      sh_ext_tas_free (&(retval->tas));

      for (i = 0; i < 32; ++i)
	{
	  if (NULL != retval->for_v[i])  SH_FREE(retval->for_v[i]);
	  if (NULL != retval->fand_v[i]) SH_FREE(retval->fand_v[i]);
	  if (NULL != retval->fnot_v[i]) SH_FREE(retval->fnot_v[i]);
	}

      SH_FREE(retval);

    }

  SL_RETURN (0, _("sh_ext_cleanup"));
}

/*
 * -- add keywords to the OR filter
 */
int sh_ext_add_or (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->for_c), ext_coms->for_v));
}

/*
 * -- add keywords to the AND filter
 */
int sh_ext_add_and (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->fand_c), ext_coms->fand_v));
}

/*
 * -- add keywords to the NOT filter
 */
int sh_ext_add_not (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->fnot_c), ext_coms->fnot_v));
}

/*
 * -- add keywords to the CL argument list
 */
int sh_ext_add_argv (char * str)
{
  if (ext_coms == NULL || ext_failed == (-1))
    return (-1);
  return (sh_ext_add (str, &(ext_coms->tas.argc), ext_coms->tas.argv));
}

/*
 * -- add a path to the environment
 */
int sh_ext_add_default (char * dummy)
{
  /* while this assignment looks ridiculous, it is here to avoid
   * an 'unused parameter' warning
   */
  char * p = (dummy == NULL ? dummy : NULL);
  int    i;

  SL_ENTER(_("sh_ext_add_default"));
  if (dummy[0] == 'n' ||  dummy[0] == 'N' ||
      dummy[0] == 'f' ||  dummy[0] == 'F' || dummy[0] == '0')
    {
      SL_RETURN(0, _("sh_ext_add_default"));
    }
  p = sh_unix_getUIDdir (SH_ERR_ERR, (uid_t) ext_coms->tas.run_user_uid);
  if (p)
    (void) sh_ext_add_envv (_("HOME"), p);
  (void) sh_ext_add_envv (_("SHELL"), _("/bin/sh")); 
  (void) sh_ext_add_envv (_("PATH"),  _("/sbin:/usr/sbin:/bin:/usr/bin")); 
  i = (p == NULL ? (-1) :  0);
  SL_RETURN(i, _("sh_ext_add_default"));
}

/*
 * -- add an environment variable
 */
int sh_ext_add_environ (char * str)
{
  int i;
  SL_ENTER(_("sh_ext_add_environ"));
  i = sh_ext_add_envv (NULL, str);
  SL_RETURN(i, _("sh_ext_add_environ"));
}

/*
 * -- set deadtime
 */
int sh_ext_deadtime (char * str)
{
  long    deadtime = 0;
  char  * tail     = NULL;

  SL_ENTER(_("sh_ext_deadtime"));

  if (ext_coms == NULL || ext_failed == (-1) || str == NULL)
    {
      SL_RETURN (-1, ("sh_ext_deadtime"));
    }
  deadtime = strtol(str, &tail, 10);
  if (tail == str || deadtime < 0 || deadtime == LONG_MAX)
    {
      SL_RETURN (-1, ("sh_ext_deadtime"));
    }
  
  ext_coms->deadtime = (time_t) deadtime;  
  SL_RETURN (0, ("sh_ext_deadtime"));  
}

/*
 * -- define type
 */
int sh_ext_type (char * str)
{
  SL_ENTER(_("sh_ext_type"));

  if (ext_coms == NULL || ext_failed == (-1) || str == NULL)
    {
      SL_RETURN((-1), _("sh_ext_type"));
    }

  if (strlen(str) != 3)
    {
      SL_RETURN((-1), _("sh_ext_type"));
    }

  set3(ext_coms->type, str[0], str[1], str[2]);

  if      (str[0] == 'l' && str[1] == 'o' && str[2] == 'g')
    ext_coms->tas.rw = 'w';
  else if (str[0] == 's' && str[1] == 'r' && str[2] == 'v')
    ext_coms->tas.rw = 'w';
  else if (str[0] == 'm' && str[1] == 'o' && str[2] == 'n')
    ext_coms->tas.rw = 'r';
  else
    {
      SL_RETURN((-1), _("sh_ext_type"));
    }

  SL_RETURN(0, _("sh_ext_type"));
} 
  


/*
 * -- define checksum
 */
int sh_ext_checksum (char * str)
{
  SL_ENTER(_("sh_ext_checksum"));
  if (ext_coms == NULL || ext_failed == (-1) || str == NULL)
    {
      SL_RETURN((-1), _("sh_ext_checksum"));
    }

  if (sl_strlen(str) != KEY_LEN)
    {
      SL_RETURN((-1), _("sh_ext_checksum"));
    }

  (void) sl_strlcpy (ext_coms->tas.checksum, str, KEY_LEN+1);

  SL_RETURN((0), _("sh_ext_checksum"));
}

/*
 * -- choose privileges
 */
int sh_ext_priv (char * c)
{

  uid_t me_uid;
  gid_t me_gid;

  SL_ENTER(_("sh_ext_priv"));
  if (0 == sh_ext_uid (c, &me_uid, &me_gid))
    {
      ext_coms->tas.run_user_uid = me_uid;
      ext_coms->tas.run_user_gid = me_gid;
      if (me_uid != (uid_t) 0)
	ext_coms->tas.privileged   = 0;
      SL_RETURN((0), _("sh_ext_priv"));
    }

  SL_RETURN (-1, _("sh_ext_priv"));
}




/*
 * -- check filters
 */
static int sh_ext_filter (char * message, sh_com_t * task)
{
  int i;
  int j = 0;
  time_t now_time;

  SL_ENTER(_("sh_ext_filter"));

  /* Presence of any of these keywords prevents execution.
   */
  if (task->fnot_c > 0)
    {
      for (i = 0; i < task->fnot_c; ++i)
	{
	  if (NULL != sl_strstr(message, task->fnot_v[i]))
	    {
	      SL_RETURN ((-1), _("sh_ext_filter"));
	    }
	}
    }

  /* Presence of all of these keywords is required for execution.
   */
  if (task->fand_c > 0)
    {
      j = 0;

      for (i = 0; i < task->fand_c; ++i)
	if (NULL != sl_strstr(message, task->fand_v[i]))
	  ++j;

      if (j != task->fand_c)
	{
	  SL_RETURN ((-1), _("sh_ext_filter"));
	}

    }

  /* Presence of at least one of these keywords is required for execution.
   */
  if (task->for_c > 0)
    {
      for (i = 0; i < task->for_c; ++i)
	{
	  if (NULL != sl_strstr(message, task->for_v[i]))
	    {
	      if (task->deadtime != (time_t) 0)   /* deadtime */
		{
		  now_time = time (NULL);
		  
		  if (task->last_run == (time_t) 0)
		    {
		      task->last_run = now_time;
		    }
		  else if ((time_t)(now_time-task->last_run) < task->deadtime)
		    {
		      SL_RETURN ((-1), _("sh_ext_filter"));
		    }
		  else
		    {
		      task->last_run = now_time;
		    }
		}
	      SL_RETURN ((0), _("sh_ext_filter"));
	    }
	}
      SL_RETURN ((-1), _("sh_ext_filter"));
    }

  SL_RETURN ((0), _("sh_ext_filter"));
}



/*
 * -- execute external script/program
 */
int sh_ext_execute (char t1, char t2, char t3, /*@null@*/char * message, 
		    size_t msg_siz)
{
  int        caperr;
  sh_com_t * listval = ext_coms;
  int        status = 0;
  char     * tmp;

  static  int some_error = 0;

  struct  sigaction  new_act;
  struct  sigaction  old_act;

  SL_ENTER(_("sh_ext_execute"));

  PDBG_OPEN;

  if (listval == NULL || message == NULL)
    {
      SL_RETURN ((-1), _("sh_ext_execute"));
    }

  PDBG(-1);

  if (msg_siz == 0)
    msg_siz = sl_strlen(message);


  /* ignore SIGPIPE (instead get EPIPE if connection is closed)
   */
  new_act.sa_handler = SIG_IGN;
  (void) retry_sigaction (FIL__, __LINE__, SIGPIPE, &new_act, &old_act);

  while (listval != NULL)
    {
      PDBG(-2);
      if (t1 == listval->type[0] &&
	  t2 == listval->type[1] &&
	  t3 == listval->type[2] &&
	  0 == sh_ext_filter (message, listval))
	{
	  PDBG(-3);

	  if (0 != (caperr = sl_get_cap_sub()))
	    {
	      sh_error_handle((-1), FIL__, __LINE__, caperr, MSG_E_SUBGEN,
			      sh_error_message (caperr), 
			      _("sl_get_cap_sub"));
	    }
	  if (0 == sh_ext_popen (&(listval->tas)))
	    {
	      PDBG(-4);
	      if (NULL != listval->tas.pipe && listval->tas.rw == 'w')
		{
		  PDBG(-5);
		  if (message != NULL)
		    {
		      PDBG(-6);
		      status = (int) write (listval->tas.pipeFD, 
					    message, msg_siz);
		      if (status >= 0)
			status = (int) write (listval->tas.pipeFD, "\n", 1);
		    }
		  PDBG_D(status);
		  if (status >= 0)
		    status = (int) write (listval->tas.pipeFD, "[", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = (int) write (listval->tas.pipeFD, "E", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = (int) write (listval->tas.pipeFD, "O", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = (int) write (listval->tas.pipeFD, "F", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = (int) write (listval->tas.pipeFD, "]", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    status = (int) write (listval->tas.pipeFD, "\n", 1);
		  PDBG_D(status);
		  if (status >= 0)
		    {
		      some_error = 0;
		    }
		  if ((status < 0) && (some_error == 0))
		    {
		      some_error = 1;
		      PDBG_S("some error");
		      PDBG_D(status);
		      tmp  = sh_util_safe_name (listval->tas.command);

		      if (tmp)
			{
			  if (listval->tas.privileged == 0 && 
			      (0 == getuid() || 0 != sl_is_suid()) )
			    sh_error_handle((-1), FIL__, __LINE__, 0, 
					    MSG_NOEXEC,
					    (UID_CAST) listval->tas.run_user_uid, 
					    tmp);
			  else
			    sh_error_handle((-1), FIL__, __LINE__, 0, 
					    MSG_NOEXEC,
					    (UID_CAST) getuid(), tmp);
			  
			  SH_FREE(tmp);
			}

		    } 
		  PDBG(-7);
		  (void) fflush(listval->tas.pipe);
		}
	      PDBG(-8);
	      (void) sh_ext_pclose(&(listval->tas));
	    }
	  else
	    {
	      PDBG_S("0 != sh_ext_popen()");
	    }
	  if (0 != (caperr = sl_drop_cap_sub()))
	    {
	      sh_error_handle((-1), FIL__, __LINE__, caperr, MSG_E_SUBGEN,
			      sh_error_message (caperr), 
			      _("sl_drop_cap_sub"));
	    }

	}
      listval = listval->next;
    }
  PDBG_S("no more commands");

  /* restore old signal handler
   */
  (void) retry_sigaction (FIL__, __LINE__, SIGPIPE, &old_act, NULL);
  PDBG_S("return");
  PDBG_CLOSE;

  SL_RETURN ((0), _("sh_ext_execute"));
}
  
  
/* #if defined(WITH_EXTERNAL) */
#endif
