#!/usr/bin/env python3


# jp: a SAT command line tool
# Copyright (C) 2009-2020 Jérôme Poisson (goffi@goffi.org)

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import os
import os.path
import asyncio
from . import base
from sat.core.i18n import _
from sat_frontends.jp.constants import Const as C
from sat.tools import config


__commands__ = ["Avatar"]
DISPLAY_CMD = ["xv", "display", "gwenview", "showtell"]


class Get(base.CommandBase):
    def __init__(self, host):
        super(Get, self).__init__(
            host, "get", use_verbose=True, help=_("retrieve avatar of an entity")
        )

    def add_parser_options(self):
        self.parser.add_argument("jid", help=_("entity"))
        self.parser.add_argument(
            "-s", "--show", action="store_true", help=_("show avatar")
        )

    async def showImage(self, path):
        sat_conf = config.parseMainConf()
        cmd = config.getConfig(sat_conf, "jp", "image_cmd")
        cmds = [cmd] + DISPLAY_CMD if cmd else DISPLAY_CMD
        for cmd in cmds:
            try:
                process = await asyncio.create_subprocess_exec(cmd, path)
                ret = await process.wait()
            except OSError:
                continue

            if ret in (0, 2):
                # we can get exit code 2 with display when stopping it with C-c
                break
        else:
            # didn't worked with commands, we try our luck with webbrowser
            # in some cases, webbrowser can actually open the associated display program.
            # Note that this may be possibly blocking, depending on the platform and
            # available browser
            import webbrowser

            webbrowser.open(path)

    async def start(self):
        try:
            avatar_path = await self.host.bridge.avatarGet(
                self.args.jid,
                False,
                False,
                self.profile,
            )
        except Exception as e:
            self.disp(f"can't retrieve avatar: {e}", error=True)
            self.host.quit(C.EXIT_BRIDGE_ERRBACK)

        if not avatar_path:
            self.disp(_("No avatar found."), 1)
            self.host.quit(C.EXIT_NOT_FOUND)

        self.disp(avatar_path)
        if self.args.show:
            await self.showImage(avatar_path)

        self.host.quit()


class Set(base.CommandBase):
    def __init__(self, host):
        super(Set, self).__init__(
            host, "set", use_verbose=True, help=_("set avatar of the profile")
        )

    def add_parser_options(self):
        self.parser.add_argument(
            "image_path", type=str, help=_("path to the image to upload")
        )

    async def start(self):
        path = self.args.image_path
        if not os.path.exists(path):
            self.disp(_(f"file {path!r} doesn't exist!"), error=True)
            self.host.quit(C.EXIT_BAD_ARG)
        path = os.path.abspath(path)
        try:
            await self.host.bridge.avatarSet(path, self.profile)
        except Exception as e:
            self.disp(f"can't set avatar: {e}", error=True)
            self.host.quit(C.EXIT_BRIDGE_ERRBACK)
        else:
            self.disp(_("avatar has been set"), 1)
            self.host.quit()


class Avatar(base.CommandBase):
    subcommands = (Get, Set)

    def __init__(self, host):
        super(Avatar, self).__init__(
            host, "avatar", use_profile=False, help=_("avatar uploading/retrieving")
        )
